#include <math.h>
#include <QtGui>
#include <QColor>
#include <QDebug>
#include <QPainter>
#include <QStringBuilder>
#include "colors.h"
#include "fractalPixmap.h"

int CFractalPixmap::fractalCounter=0;

CFractalPixmap::CFractalPixmap(const FractalSettings fractalSettings)
	: QObject(), QGraphicsPixmapItem()
{
    mutex.lock();
    settings = fractalSettings;
    fractalIndex = fractalCounter++;
    mutex.unlock();

    setCacheMode(DeviceCoordinateCache);
    setAcceptHoverEvents(true);

    initialize();
    progressRefreshIsNeeded = true;

    hasVerticalSymmetry = (settings.center.y()==0);

    if (settings.stepBetweenPoints==0)
    {
        switch (settings.fractalType) {
	case MANDELBROT:
	case BUDDHABROT:
            settings.stepBetweenPoints = 2.0/settings.width;
	    break;
	default:
             settings.stepBetweenPoints = 4.0/settings.width;
	}
    }

    dataSize	= settings.width * settings.height;

    int subCount = settings.subStepX*settings.subStepY*settings.repeatAll;
    int subIndex=0;
    subFractals.resize(subCount);

    for (int r=0; r<settings.repeatAll; r++)
	for (int i=0; i<settings.subStepX; i++)
	{
	    for (int j=0; j<settings.subStepY; j++)
	    {
		subFractals[subIndex].globale      = settings;
		subFractals[subIndex].status       = NOTSTARTED;
		subFractals[subIndex].stepX        = settings.subStepX * settings.stepBetweenPoints;
		subFractals[subIndex].stepY        = settings.subStepY * settings.stepBetweenPoints;

		subFractals[subIndex].startX = settings.range.left() + i * settings.stepBetweenPoints;
		subFractals[subIndex].offsetX = i;

		subFractals[subIndex].startY = settings.range.bottom() + j * settings.stepBetweenPoints;
		subFractals[subIndex].offsetY = j;
		subFractals[subIndex].width  = settings.width/settings.subStepX;
		subFractals[subIndex].height = settings.height/settings.subStepY;
		subFractals[subIndex].pDensityMap = &densityMap;
		subIndex++;
	    }
	}
    updateProgressRange(0,subFractals.count());
    updateProgressValue(0);

    connect(&watcher,	SIGNAL(progressValueChanged(int)),
	    this,	SLOT(updateProgressValue(int)));
    connect(&watcher,	SIGNAL(progressRangeChanged(int,int)),
            this,	SLOT(updateProgressRange(int,int)));
    connect(&watcher,	SIGNAL(finished()),
	    this,	SLOT(renderFull()));
    connect(&watcher,	SIGNAL(finished()),
	    this,	SLOT(updateAllFinished()));
}

CFractalPixmap::~CFractalPixmap()
{
    watcher.cancel();
    watcher.waitForFinished();
}

void CFractalPixmap::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    emit clicked(fractalIndex);
}

void CFractalPixmap::generateColorMap()
{
    for (int i = 0; i < COLORMAP_SIZE; ++i)
	colorMap[i] = rainbow(i,COLORMAP_SIZE);
}

void CFractalPixmap::start()
{
    connect(&timer, SIGNAL(timeout()),
	    this, SLOT(renderProgress()));
    timer.start(200);

    timeStart = QTime::currentTime();
    future = QtConcurrent::mappedReduced(subFractals, doCalculate, doPostProcess);

    watcher.setFuture(future);

}

void CFractalPixmap::initialize()
{
    QString numStr = QString().sprintf("%03d", fractalIndex);
    QString typeStr = tr(CSettings::typeStrings[settings.fractalType]);
    fractalName = numStr % QLatin1String("_") % typeStr;

    // Loading the icon as an initial preview
    pixmap.load(QLatin1String(CSettings::iconFileNames[settings.fractalType]));
    setPixmap(pixmap);

    // Preparing the full image
    fullImage = new QImage(settings.width, settings.height, QImage::Format_RGB32);
    fullImage->fill(Qt::black);

    if (settings.fractalType == BUDDHABROT)
    {
        densityMap.resize(settings.width*settings.height);
        densityMax=0;
    }
    else
    {
        densityMap.resize(0);
    }

}

void CFractalPixmap::renderProgress()
{
    if (progressRefreshIsNeeded)
    {
        uint color;
	int length = static_cast<int>((fullImage->width()-4) * progressValue / (progressMax-progressMin));
        for (int i=0; i<length; i++)
        {
	    color = rainbow(i,fullImage->width());
            for (int j=0; j<5; j++)
		fullImage->setPixel(i, j+PREVIEW_SIZE/2, color);
        }
    }
    progressRefreshIsNeeded = false;
}

void CFractalPixmap::renderFull()
{
    timeStop = QTime::currentTime();
    int timeInterval = timeStart.msecsTo(timeStop);
    qDebug() << "Calculation time:" << QString::number(timeInterval);
    uint color;
    int index=0;
    double q;

    future.pause();
    //densityMap = future.result();
    densityMap = future.resultAt(0);
    if (settings.fractalType==BUDDHABROT)
    {
	densityMax=0;
	for (int i=0; i<densityMap.count(); i++)
	    densityMax = qMax (densityMap.at(i),densityMax);

	for (int j=0; j<settings.height; j++)
	{
	    for (int i=0; i<settings.width; i++)
	    {
		q=static_cast<double>(densityMap.at(index)*255)/densityMax;
		color = qRgb(q,q,q);
		fullImage->setPixel(i,j,color);
		index ++;
	    }
	}
    }
    else
    {
	densityMax=0;
	for (int i=0; i<densityMap.count(); i++)
	    densityMax = qMax (densityMap.at(i),densityMax);

	for (int j=0; j<settings.height; j++)
	{
	    for (int i=0; i<settings.width; i++)
	    {
		color = rainbow(densityMap.at(index),settings.maxIterations);
		fullImage->setPixel(i,j,color);
		index ++;
	    }
	}
    }
    future.resume();
    progressRefreshIsNeeded=false;
}

void CFractalPixmap::showPreview(int width, int height)
{
    if (width<=0 || height<=0)
	setPixmap(QPixmap::fromImage(fullImage->scaled(QSize(PREVIEW_SIZE, PREVIEW_SIZE), Qt::KeepAspectRatio, Qt::SmoothTransformation)));
    else
	setPixmap(QPixmap::fromImage(fullImage->scaled(QSize(width, height), Qt::KeepAspectRatio, Qt::SmoothTransformation)));
}

void CFractalPixmap::showFull(int width, int height)
{
    if (width<=0 || height<=0)
	setPixmap(QPixmap::fromImage(*fullImage));
    else
    {
	QImage scaledImage=fullImage->scaled(QSize(width, height), Qt::KeepAspectRatio, Qt::SmoothTransformation);
	imageSize = scaledImage.size();
	setPixmap(QPixmap::fromImage(scaledImage));
    }
}


SubResult CFractalPixmap::doCalculate(const SubDefinition &sub)
{
//    sub.status = RUNNING;
//    QVector<int> result;
    SubResult result;

    switch (sub.globale.fractalType) {
	case MANDELBROT:
		result = calculateMandelbrot(sub);
		break;
	case BUDDHABROT:
		result = calculateBuddhabrot(sub);
		break;
	case JULIA:
		result = calculateJulia(sub);
		break;
	default:
		break;
		result = calculateMandelbrot(sub);
    }
//    sub.status = CALCULATED;
    return result;
}

SubResult CFractalPixmap::calculateMandelbrot(const SubDefinition &sub)
{
    qDebug() << "Calculate";
    SubResult result;
    result.subDefinition=sub;
    int index=0;
    double X, Y;
    double Cr, Ci, Zr, Zi, NewZr;
    bool doesEscape;
    int iterations;

    result.iVector.resize(sub.width*sub.height);
    Y = sub.startY;
    for (int j=0; j<sub.height; j++)
    {
	X = sub.startX;
	for (int i=0; i<sub.width; i++)
	{
	    Zr=Zi=0.0;
	    Cr=X;
	    Ci=Y;
	    doesEscape= false;
	    iterations=0;
	    while ((!doesEscape) && (iterations<sub.globale.maxIterations))
	    {
		NewZr = Zr*Zr-Zi*Zi + Cr;
		Zi = 2.0*Zr*Zi	 + Ci;
		Zr = NewZr;
		doesEscape = (Zr*Zr + Zi*Zi > 4.0);
		if (!doesEscape) iterations++;
	    }

	    if (doesEscape)
		result.iVector[index] = iterations;
	    else
		result.iVector[index] = -1;

	    index++;
	    X += sub.stepX;
	}
	Y += sub.stepY;
    }
    return result;
}

SubResult CFractalPixmap::calculateBuddhabrot(const SubDefinition &sub)
{
    SubResult result;
    result.subDefinition=sub;
    QVector<double> zBuffer;
    zBuffer.resize(2*sub.globale.maxIterations);
    double X, Y, Cr, Ci, Zr, Zi, NewZr;
    bool doesEscape;
    int iterations, iterIndex, ii, jj, buffIndex=0, randIndex1=0, randIndex2=0;

    double Z0r=sub.globale.range.left();
    double Z0i=sub.globale.range.bottom();
    double K=1.0/sub.globale.stepBetweenPoints;

    QVector<double> randomVector1, randomVector2;
    double range = sub.globale.stepBetweenPoints;
    int size = static_cast<int>(sqrt(sub.globale.width * sub.globale.height));
    randomVector1.resize(size);
    randomVector2.resize(size+1);
    for (int i=0; i<size; i++)
	randomVector1[i]=static_cast<double> (range*qrand()/RAND_MAX);
    for (int i=0; i<size+1; i++)
	randomVector2[i]=static_cast<double> (range*qrand()/RAND_MAX);

    Y = sub.startY;
    for (int j=0; j<sub.height; j++)
    {
	X = sub.startX;
	for (int i=0; i<sub.width; i++)
	{
	    Zr=Zi=0.0;
	    Cr=X + randomVector1[randIndex1++];
	    Ci=Y + randomVector2[randIndex2++];
	    if (randIndex1>=randomVector1.count()) randIndex1=0;
	    if (randIndex2>=randomVector2.count()) randIndex2=0;

	    doesEscape= false;
	    iterations=iterIndex=buffIndex=0;
	    while ((!doesEscape) && (iterations<sub.globale.maxIterations))
	    {
		NewZr = Zr*Zr-Zi*Zi + Cr;
		Zi = 2.0*Zr*Zi	 + Ci;
		Zr = NewZr;
		doesEscape = (Zr*Zr + Zi*Zi > 4.0);
		if (!doesEscape)
		{
		    zBuffer[buffIndex++]=Zr;
		    zBuffer[buffIndex++]=Zi;
		    iterations++;
		}
	    }

	    if (doesEscape)
		for (int m=0; m<iterations; m++)
		{
		    // Complex coordinates to map coordinates
		    ii = static_cast<int>((zBuffer[2*m]  -Z0r)*K);
		    jj = static_cast<int>((zBuffer[2*m+1]-Z0i)*K);
		    if ((ii>=0) && (ii<sub.globale.width) && (jj>=0) && (jj<sub.globale.height))
			result.iVector.append(ii + jj*sub.globale.width);
		}
	    X += sub.stepX;
	}
	Y += sub.stepY;
    }
    randomVector1.clear();
    randomVector2.clear();
    zBuffer.clear();
    return result;
}

SubResult CFractalPixmap::calculateJulia(const SubDefinition &sub)
{
    SubResult result;
    return result;
}


void CFractalPixmap::doPostProcess(QVector<int> &result, const SubResult &sub)
{
    qDebug() << "Postprocess" << result.count();
    if (result.count()==0)
	result.resize(sub.subDefinition.globale.width * sub.subDefinition.globale.height);
    if (sub.subDefinition.globale.fractalType == BUDDHABROT)
    {
	for (int i=0; i<sub.iVector.count(); i++)
	    result[sub.iVector.at(i)]++;
    }
    else
    {
	qDebug() << "Width:" << sub.subDefinition.width << "Height:" << sub.subDefinition.height;
	int index=0, ii, jj;
	int offsetX = sub.subDefinition.offsetX;
	int offsetY = sub.subDefinition.offsetY;
	for (int j=0; j<sub.subDefinition.height; j++)
	    for (int i=0; i<sub.subDefinition.width; i++)
	    {
		ii = offsetX + i*sub.subDefinition.globale.subStepX;
		jj = offsetY + j*sub.subDefinition.globale.subStepY;
		result[ii + jj*sub.subDefinition.globale.width] += sub.iVector.at(index);
		index ++;
	    }
    }
    *sub.subDefinition.pDensityMap = result;
    qDebug() << "Postprocess end" << result.count();
}

void CFractalPixmap::doPostProcessNonStatic(QVector<int> &result, const SubResult &sub)
{
    result.resize(sub.subDefinition.globale.width * sub.subDefinition.globale.height);
    if (sub.subDefinition.globale.fractalType == BUDDHABROT)
//    if (settings.fractalType == BUDDHABROT)
    {
	for (int i=0; i<sub.iVector.count(); i++)
	    result[sub.iVector.at(i)]++;
    }
    else
    {
	int index=0;
	int ii, jj;
	int offsetX = sub.subDefinition.offsetX;
	int offsetY = sub.subDefinition.offsetY;
	for (int j=0; j<sub.subDefinition.height; j++)
	    for (int i=0; i<sub.subDefinition.width; i++)
	    {
		ii = offsetX + i*sub.subDefinition.globale.subStepX;
		jj = offsetY + j*sub.subDefinition.globale.subStepY;
		result[ii + jj*sub.subDefinition.globale.width] += sub.iVector.at(index);
		index ++;
	    }
    }
    //sub.iVector.clear();
    result.append(1);
}

void CFractalPixmap::updateProgressValue(const int value)
{
    progressValue = value;
    emit progressValueChanged(value);
    progressRefreshIsNeeded = true;
}

void CFractalPixmap::updateProgressRange(const int minimum, const int maximum)
{
    if (minimum < maximum)
    {
	progressMin = minimum;
	progressMax = maximum;
	emit progressRangeChanged(minimum, maximum);
    }
}

void CFractalPixmap::stop()
{

}

void CFractalPixmap::pause()
{
    updateProgressValue(++progressValue);
}

void CFractalPixmap::resume()
{

}

void CFractalPixmap::cancel()
{

}

void CFractalPixmap::updateAllFinished()
{
    progressRefreshIsNeeded=false;
    emit allFinished();
}

QImage CFractalPixmap::scale(const QString &imageFileName, int size)
{
    QImage image(imageFileName);
    return image.scaled(QSize(PREVIEW_SIZE, PREVIEW_SIZE), Qt::IgnoreAspectRatio, Qt::SmoothTransformation);
}

QImage CFractalPixmap::scale(const QImage *image, int size)
{
    return image->scaled(QSize(PREVIEW_SIZE, PREVIEW_SIZE), Qt::IgnoreAspectRatio, Qt::SmoothTransformation);
}
