#ifndef FRACTALPIXMAP_H
#define FRACTALPIXMAP_H

#include <complex>
#include <QFutureWatcher>
#include <QGraphicsPixmapItem>
#include <QMutex>
#include <QObject>
#include <QPixmap>
#include <QTime>
#include <QTimer>
#include <QVector>
#include "QtConcurrentRun"
#include "colors.h"
#include "settingsDialog.h"

typedef std::complex<double> point;
typedef QFutureWatcher<void> VoidWatcher;

enum EStatus{
    NOTSTARTED,
    RUNNING,
    PAUSED,
    CALCULATED,
    RENDERED,
    ERROR
};

struct SubDefinition
{
    int offsetX;
    int offsetY;
    int width;
    int height;
    double startX;
    double startY;
    double stepX;
    double stepY;
    FractalSettings globale;
    EStatus status;
    QVector<int> *pDensityMap;
};

struct SubResult
{
    SubDefinition subDefinition;
    QVector<int> iVector;
};

class CFractalPixmap : public QObject, public QGraphicsPixmapItem
{
    Q_OBJECT
    Q_PROPERTY(QPointF pos READ pos WRITE setPos)
public:
    CFractalPixmap(const FractalSettings fractalSettings);
    ~CFractalPixmap();


    int getSize() {return dataSize;}
    int getIndex() {return fractalIndex;}
    QSize size() {return pixmap.size();}
    QSize imSize() {return imageSize;}
    QStringList fractalTypeNames;
    QAbstractTransition *trans;
    FractalSettings settings;

public slots:
    void renderProgress();
    void renderFull();
    void showPreview(int width=PREVIEW_SIZE, int height=PREVIEW_SIZE);
    void showFull(int width=0, int height=0);
    void updateProgressValue(const int value);
    void updateProgressRange(const int minimum, const int maximum);
    void updateAllFinished();
    void start();
    void stop();
    void pause();
    void resume();
    void cancel();

signals:
    void allFinished();
    void finishedAt(int index);
    void progressValueChanged ( int progressValue);
    void progressRangeChanged (int minimum, int maximum);
    void clicked(int fractIndex);

private:
    void mousePressEvent(QGraphicsSceneMouseEvent *event);

    static SubResult doCalculate	 (const SubDefinition &sub);
    static SubResult calculateMandelbrot (const SubDefinition &sub);
    static SubResult calculateBuddhabrot (const SubDefinition &sub);
    static SubResult calculateJulia	 (const SubDefinition &sub);
    //void doPostProcess		 (QVector<int> &result, const SubResult &sub);
    //static void doPostProcess		 (QVector<int> &result, const QVector<int> &vector);
    static void doPostProcess		 (QVector<int> &result, const SubResult &sub);
    void doPostProcessNonStatic		 (QVector<int> &result, const SubResult &sub);

    static int fractalCounter;

    void initialize();
    void generateColorMap();
    QImage scale(const QString &imageFileName, int size=PREVIEW_SIZE);
    QImage scale(const QImage *image, int size=PREVIEW_SIZE);

    uint colorMap[COLORMAP_SIZE];
    QVector<SubDefinition> subFractals;
    QVector<int>    densityMap;

    QString	    fractalName;
    int             fractalIndex;
    bool	    hasVerticalSymmetry;
    bool	    isPreviewMode;
    int		    dataSize;
    EFractalType    fractalType;
    int		    progressValue;
    int		    progressMin;
    int		    progressMax;
    bool            progressRefreshIsNeeded;

    VoidWatcher     watcher;
    QFuture< QVector<int> > future;
    QPixmap	    pixmap;
//    QImage          *previewImage;
    QImage	    *fullImage;
    QSize	    imageSize;
    uint            *previewPixels;
    QTimer          timer;
    QMutex          mutex;
    QTime	    timeStart, timeStop;
    int             densityMax;
};

#endif // FRACTALPIXMAP_H
