#include <QtGui>
#include <QDialog>
#include "mainwindow.h"

int buttonSize = 20;
int typeButtonHalfDim = 6;
int typeButtonDim = typeButtonHalfDim*2;
QSize typeButtonSize = QSize(typeButtonDim,typeButtonDim);
QRect typeButtonRect = QRect(0,0,typeButtonDim,typeButtonDim);
int buttonRadius=3;


QImage scale(const QString &imageFileName, QSize size)
{
    QImage image(imageFileName);
    return image.scaled(size, Qt::KeepAspectRatio, Qt::SmoothTransformation);
}


void View::resizeEvent(QResizeEvent *event)
{
    QGraphicsView::resizeEvent(event);
    scene()->setSceneRect(this->rect());
    emit resized();
//    fitInView(sceneRect(), Qt::KeepAspectRatio);
}


Button::Button(const QPixmap &pixmap, const int buttonSize, int index, QGraphicsItem *parent)
    : QGraphicsWidget(parent), _pix(pixmap)
{
    setAcceptHoverEvents(true);
    setCacheMode(DeviceCoordinateCache);
    fractalTypeIndex = index;
    size = buttonSize;
}

QRectF Button::boundingRect() const
{
    return QRectF( -(size/2+buttonRadius), -(size/2+buttonRadius),
                     size+2*buttonRadius,    size+2*buttonRadius);
}

QPainterPath Button::shape() const
{
    QPainterPath path;
    path.addRoundedRect(boundingRect(), buttonRadius, buttonRadius);
    return path;
}

void Button::paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *)
{
    //bool down = option->state & QStyle::State_Sunken;
    QRectF r = boundingRect();
//    QLinearGradient grad(r.topLeft(), r.bottomRight());
//    grad.setColorAt(down ? 1 : 0, Qt::white);
//    grad.setColorAt(down ? 1 : 0, option->state & QStyle::State_MouseOver ? Qt::white : Qt::lightGray);
//    grad.setColorAt(down ? 0 : 1, Qt::lightGray);
//    painter->setPen(Qt::lightGray);
    //    painter->setBrush(grad);
    painter->setPen(Qt::white);
    painter->setBrush(Qt::white);
    painter->drawRoundedRect(r, buttonRadius, buttonRadius);
//    QLinearGradient grad2(r.topLeft(), r.bottomRight());
//    grad2.setColorAt(down ? 1 : 0, Qt::lightGray);
//    grad2.setColorAt(down ? 0 : 1, Qt::white);
//    grad2.setColorAt(down ? 0 : 1, option->state & QStyle::State_MouseOver ? Qt::white : Qt::lightGray);
//    painter->setPen(Qt::NoPen);
//    painter->setBrush(grad2);
//    if (down)
//	painter->translate(2, 2);
    painter->drawRoundedRect(r.adjusted(5, 5, -5, -5), buttonRadius, buttonRadius);
    painter->drawPixmap(-_pix.width()/2, -_pix.height()/2, _pix);
}

void Button::mousePressEvent(QGraphicsSceneMouseEvent *mouseEvent)
{
    bool isLeftButton = (mouseEvent->buttons() == Qt::LeftButton);
    emit pressed(fractalTypeIndex, isLeftButton);

    update();
}

void Button::mouseReleaseEvent(QGraphicsSceneMouseEvent *mouseEvent)
{
    update();
}

void Button::hoverEnterEvent(QGraphicsSceneHoverEvent *)
{
    emit hoverEnter(fractalTypeIndex);
}

void Button::hoverLeaveEvent(QGraphicsSceneHoverEvent *)
{
    emit hoverLeave();
}


MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent)
{
    setSizePolicy(QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed));
    //scene.setBackgroundBrush(QBrush(Qt::black));
    scene.setSceneRect(0, 0, 800, 800);
    currentFractal=0;

    // Buttons
    buttonParent = new QGraphicsRectItem;

    // Buttons for Fractal types
    for (int i=0; i<FRACTAL_TYPE_COUNT; i++)
    {
        newFractalButtons[i]
		= new Button(QPixmap::fromImage(scale(QLatin1String(":/images/plus.png"), typeButtonSize)),
                         typeButtonDim, i, buttonParent);
	connect(newFractalButtons[i],   SIGNAL(pressed(int, bool)),
		this,			SLOT  (newFrac(int, bool)));
	connect(newFractalButtons[i],   SIGNAL(hoverEnter(int)),
		this,			SLOT  (showHint(int)));
	connect(newFractalButtons[i],   SIGNAL(hoverLeave()),
		this,			SLOT  (hideHint()));
    }

    scene.addItem(buttonParent);
    buttonParent->setPos(0, 0);
    buttonParent->setZValue(65);

    hintItem = new QGraphicsPixmapItem(hintPix);
    hintItem = scene.addPixmap(hintPix);

    QFont font;
    font.setPixelSize(12);
    font.setFixedPitch(true);

    hintText = new QGraphicsTextItem;
    hintText = scene.addText(QLatin1String(""),font);
    hintText->setDefaultTextColor(Qt::white);

    bottomLine = new QGraphicsLineItem;
    bottomLine = scene.addLine(20,scene.height()-50,scene.width()-20,scene.height()-50,QPen(Qt::white));
    bottomLine->setZValue(-1);
    //scene.addLine(bottomLine,QPen(Qt::white));

    // Ui
    view = new View(&scene);
    view->setMinimumSize(800, 400);
    connect(view, SIGNAL(resized()),
	    this, SLOT(redrawGUI()));
    connect(view, SIGNAL(resized()),
	    this, SLOT(repositionFractals()));
//    view->setHorizontalScrollBarPolicy(Qt::ScrollBarAsNeeded);
//    view->setVerticalScrollBarPolicy(Qt::ScrollBarAsNeeded);
    view->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    view->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    view->setWindowTitle(QT_TRANSLATE_NOOP(QGraphicsView, QLatin1String("fraQtal")));
    view->setViewportUpdateMode(QGraphicsView::BoundingRectViewportUpdate);
    view->setBackgroundBrush(QBrush(Qt::black));
    view->setCacheMode(QGraphicsView::CacheBackground);
    view->setRenderHints(QPainter::Antialiasing | QPainter::SmoothPixmapTransform);
    view->show();
}

void MainWindow::setProgressRange(const int minimum, const int maximum)
{

}

void MainWindow::setProgressValue(const int value)
{
}

void MainWindow::start()
{

}

void MainWindow::redrawGUI()
{
    for (int i=0; i<FRACTAL_TYPE_COUNT; i++)
	newFractalButtons[i]->setPos(20+i*25, scene.height()-20);
    bottomLine->setLine(10,scene.height()-40,scene.width()-10,scene.height()-40);
}

void MainWindow::showHint(int index)
{
    hintItem->setPixmap(QPixmap(QLatin1String(CSettings::iconFileNames[index])));
    int offsetX = newFractalButtons[FRACTAL_TYPE_COUNT-1]->pos().x() + 20;
    int offsetY = view->height()-hintItem->pixmap().height()-10;
    hintItem->setPos(offsetX, offsetY);
    hintItem->setVisible(true);

    QString text;
    if (index<2)
	text = tr("New <b>") % QLatin1String(CSettings::typeStrings[index]) % tr("</b>: <i>(Right-click to start with default settings...)</i>");
    else
	text = tr("<b>") % QLatin1String(CSettings::typeStrings[index]) % tr("</b>: <i> coming soon, thanks for your patience...</i>");
    offsetX += hintItem->pixmap().width() + 10;
    offsetY = view->height()-hintText->boundingRect().height()-5;
    hintText->setHtml(text);
    hintText->setPos(offsetX, offsetY);
    hintText->setVisible(true);
}

void MainWindow::hideHint()
{
    //hintItem->setPos(200, view->height()+200);
    hintItem->setVisible(false);
    hintText->setVisible(false);
}


void MainWindow::fractalClicked(int fractIndex)
{
    currentFractal = fractIndex;
    repositionFractals();
}

void MainWindow::repositionFractals()
{
    int fracCount = fractalList.count();
    int sizePreview = PREVIEW_SIZE;
    if (fracCount>1)
	sizePreview = qMin ((view->width()-20)/(fracCount-1)-20, static_cast<int>(PREVIEW_SIZE));

    int offset=sizePreview + 10;
    int pos=view->width()/2.0 - ((fracCount-1)*offset/2.0);
    int availableWidth = static_cast<int>(view->width())-40;
    int availableHeight= static_cast<int>(view->height())-200;
//    double fracWidth=fractalList[currentFractal]->settings.width;
//    double fracHeight=fractalList[currentFractal]->settings.height;
//    double correction = availableWidth * (qMax (availableWidth/availableHeight / (fracWidth/fracHeight) , 1.0) -1);

    for (int i=0; i<fracCount; i++)
    {
	if (i==currentFractal)
	{
	    fractalList[i]->showFull(availableWidth, availableHeight);
	    //scene.items()[i]->setPos(QPointF(20, 20));
	    fractalList[i]->setPos(QPointF(20+(availableWidth - fractalList[i]->imSize().width())/2, 20));
	    //stateList[state]->assignProperty(fractalList[i], "pos", QPointF(20, 20));
	}
	else
	{
	    fractalList[i]->showPreview(sizePreview, sizePreview);
	    fractalList[i]->setPos(QPointF(pos, view->height()-50 - sizePreview));
	    //scene.items()[i]->setPos(QPointF(pos, view->height()-50 - sizePreview));
	    //stateList[state]->assignProperty(fractalList[i], "pos", QPointF(pos, view->height()-50));
	    pos += offset;
	}
    }
    view->repaint();
}

void MainWindow::newFrac(const int typeIndex, const bool doShowDialog)
{
    FractalSettings settings;
    if (typeIndex>1) return; // Only the first 2 fractals are implemented yet...

    if (doShowDialog)
    {
	CSettings settingsDialog(defaultSettings[typeIndex]);
	settingsDialog.exec();
	settings=settingsDialog.getSettings();
    }
    else
	settings = defaultSettings[typeIndex];

    CFractalPixmap *item = new CFractalPixmap(settings);
    //item->setFlag(QGraphicsItem::ItemIsSelectable);
    int fracIndex = item->getIndex();
    currentFractal = fracIndex;
    connect (item, SIGNAL(progressValueChanged(int)),
	     this, SLOT(repositionFractals()));

    connect (item, SIGNAL(allFinished()),
	     this, SLOT(repositionFractals()));

    item->setOffset(0,0);
    //item->setPos(newFractalButtons[typeIndex]->pos());
    item->setZValue(fracIndex);
    fractalList << item;
    scene.addItem(item);

    item->start();

    repositionFractals();
}
