#include <QtGui>
#include "settingsDialog.h"

const char *CSettings::typeStrings[] = {
    QT_TR_NOOP("Mandelbrot Set"),
    QT_TR_NOOP("Buddhabrot Set"),
    QT_TR_NOOP("Julia Set"),
    QT_TR_NOOP("Clifford Attractor"),
    QT_TR_NOOP("Nova Set"),
    QT_TR_NOOP("Crab Set"),
    QT_TR_NOOP("Newton Set"),
    QT_TR_NOOP("Burning Ship Set"),
    QT_TR_NOOP("Tricorn Set")};

const char *CSettings::iconFileNames[] = {
    ":/images/icon_mandelbrot.png",
    ":/images/icon_buddhabrot.png",
    ":/images/icon_julia.png",
    ":/images/icon_clifford.png",
    ":/images/icon_nova.png",
    ":/images/icon_crab.png",
    ":/images/icon_newton.png",
    ":/images/icon_burningship.png",
    ":/images/icon_tricorn.png"};

const char *CSettings::bgFileName =
    ":/images/background.jpg";

CSettings::CSettings(FractalSettings settings, QWidget *parent)
    : QDialog(parent)

{
    initialSettings = settings;

    // Fractal type selection
    QLabel *typeLabel = new QLabel(tr("Fractal type:"));
    typeCombo = new QComboBox;
    typeCombo->setIconSize(QSize(ICON_SIZE, ICON_SIZE));
    //for (int i=0; i<FRACTAL_TYPE_COUNT; i++)
    for (int i=0; i<2; i++) // Only the first 2 fractals are implemented yet
        typeCombo->insertItem(i, QIcon(QLatin1String(iconFileNames[i])), tr(typeStrings[i]));
    typeLabel->setBuddy(typeCombo);
    connect(typeCombo,  SIGNAL(currentIndexChanged(int)),
            this,       SLOT(showSettingsForType(int)));

    QHBoxLayout *typeLayout = new QHBoxLayout;
    typeLayout->addWidget(typeLabel);
    typeLayout->addWidget(typeCombo);

    // x,y start,stop values
    QLabel *xStartLabel = new QLabel(tr("X start"));
    QLabel *xStopLabel	= new QLabel(tr("X stop"));
    QLabel *yStartLabel = new QLabel(tr("Y start"));
    QLabel *yStopLabel	= new QLabel(tr("Y stop"));
    xStartSpin	= new QDoubleSpinBox;
    xStopSpin	= new QDoubleSpinBox;
    yStartSpin	= new QDoubleSpinBox;
    yStopSpin	= new QDoubleSpinBox;

    connect(xStartSpin, SIGNAL(valueChanged (double)),
	    this,	SLOT  (xStartChanged(double)));
    connect(xStopSpin,  SIGNAL(valueChanged (double)),
	    this,	SLOT  (xStopChanged (double)));
    connect(yStartSpin, SIGNAL(valueChanged (double)),
	    this,	SLOT  (yStartChanged(double)));
    connect(yStopSpin,  SIGNAL(valueChanged (double)),
	    this,	SLOT  (yStopChanged (double)));

    // Setting up X start, stop
    xStartSpin->setDecimals	(PRECISION_DIGITS);
    xStopSpin->setDecimals	(PRECISION_DIGITS);
    xStartSpin->setSingleStep	(PRECISION*100);
    xStopSpin->setSingleStep	(PRECISION*100);
    xStartSpin->setMinimum	(-2.0);
    xStopSpin->setMaximum	(2.0);
    xStartSpin->setMaximum	(xStopSpin->value() -PRECISION);
    xStopSpin->setMinimum	(xStartSpin->value()+PRECISION);

    // Setting up Y start, stop
    yStartSpin->setDecimals	(PRECISION_DIGITS);
    yStopSpin->setDecimals	(PRECISION_DIGITS);
    yStartSpin->setSingleStep	(PRECISION*1000);
    yStopSpin->setSingleStep	(PRECISION*1000);
    yStartSpin->setMinimum	(-2.0);
    yStopSpin->setMaximum	(2.0);
    yStartSpin->setMaximum	(yStopSpin->value() -PRECISION);
    yStopSpin->setMinimum	(yStartSpin->value()+PRECISION);

    // All goes in a grid layout
    QGridLayout *xyLayout = new QGridLayout;
    xyLayout->addWidget(xStartLabel, 0,0);
    xyLayout->addWidget(xStartSpin,  0,1);
    xyLayout->addWidget(xStopLabel,  1,0);
    xyLayout->addWidget(xStopSpin,   1,1);
    xyLayout->addWidget(yStartLabel, 2,0);
    xyLayout->addWidget(yStartSpin,  2,1);
    xyLayout->addWidget(yStopLabel,  3,0);
    xyLayout->addWidget(yStopSpin,   3,1);


    // Image size settings
    QLabel *widthLabel	= new QLabel(tr("Image Width"));
    QLabel *heightLabel = new QLabel(tr("Image Height"));
    widthSpin = new QSpinBox;
    heightSpin = new QSpinBox;
    widthSpin->setRange(100, 10000);
    widthSpin->setSingleStep(100);
    heightSpin->setRange(100, 10000);
    heightSpin->setSingleStep(100);

    // All goes in a grid layout
    QGridLayout *sizeLayout = new QGridLayout;
    sizeLayout->addWidget(widthLabel,  0,0);
    sizeLayout->addWidget(widthSpin,   0,1);
    sizeLayout->addWidget(heightLabel, 1,0);
    sizeLayout->addWidget(heightSpin,  1,1);


    // Max iterations
    QLabel *iterLabel = new QLabel(tr("Maximal iterations"));
    iterSpin = new QSpinBox;
    iterSpin->setRange(1, 1e9);

    // All goes in a horizontal layout
    QHBoxLayout *iterLayout = new QHBoxLayout;
    iterLayout->addWidget(iterLabel);
    iterLayout->addWidget(iterSpin);


    // Dialog buttons
    dialogButtons = new QDialogButtonBox(QDialogButtonBox::Ok |
					 QDialogButtonBox::Cancel);
    connect(dialogButtons,  SIGNAL(accepted()),
	    this,	    SLOT(accept()));
    connect(dialogButtons,  SIGNAL(rejected()),
	    this,	    SLOT(cancel()));

    setSettings(settings);

    // Putting it all together
    QVBoxLayout *mainLayout = new QVBoxLayout;
    mainLayout->addLayout(typeLayout);
    mainLayout->addLayout(xyLayout);
    mainLayout->addLayout(sizeLayout);
    mainLayout->addLayout(iterLayout);
    mainLayout->addWidget(dialogButtons);
    setLayout(mainLayout);

    setWindowTitle(tr("Define fractal settings"));
    setFixedHeight(sizeHint().height());
    setFixedWidth(sizeHint().width());
}


void CSettings::showSettingsForType(int fracType)
{
    dialogSettings.fractalType=static_cast<EFractalType>(fracType);
}

FractalSettings CSettings::getSettings()
{
    double x1 = xStartSpin->value();
    double x2 = xStopSpin->value();
    double y1 = yStartSpin->value();
    double y2 = yStopSpin->value();
    dialogSettings.center  = QPointF((x1+x2)/2.0,(y1+y2)/2.0);
    dialogSettings.width   = widthSpin->value();
    dialogSettings.height  = heightSpin->value();
    dialogSettings.stepBetweenPoints = qMax((x2-x1)/dialogSettings.width, (y2-y1)/dialogSettings.width);

    double deltaX = dialogSettings.stepBetweenPoints * dialogSettings.width/2.0;
    x1 = dialogSettings.center.x() - deltaX;
    x2 = dialogSettings.center.x() + deltaX;
    double deltaY = dialogSettings.stepBetweenPoints * dialogSettings.height/2.0;
    y1 = dialogSettings.center.y() - deltaY;
    y2 = dialogSettings.center.y() + deltaY;

    dialogSettings.range=QRectF(QPointF(x1, y2),
				QPointF(x2, y1));

    dialogSettings.maxIterations = iterSpin->value();

    return dialogSettings;
}

void CSettings::setSettings(FractalSettings newSettings)
{
    typeCombo->setCurrentIndex	(static_cast<int>(newSettings.fractalType));
    xStartSpin->setValue	(newSettings.range.left());
    xStopSpin->setValue		(newSettings.range.right());
    yStartSpin->setValue	(newSettings.range.bottom());
    yStopSpin->setValue		(newSettings.range.top());
    widthSpin->setValue		(newSettings.width);
    heightSpin->setValue	(newSettings.height);
    iterSpin->setValue		(newSettings.maxIterations);
    dialogSettings=newSettings;
};

void CSettings::accept()
{
    this->close();
}

void CSettings::cancel()
{
    setSettings(initialSettings);
    this->close();
}
