/*
# Copyright (c) 2010 Louis du Verdier
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
*/

#ifndef GENERATOR_H
#define GENERATOR_H

/*!
 * \file Generator.h
 * \brief Thread gérant la génération des fractales Mandelbrot et Complex Map Buddhabrot.
 * \author Louis du Verdier
 * \version 1.0
 */

#include <QtGui>
#include <complex>

/*! \class Generator
 *  \brief La classe Generator correspond à un thread s'occupant de la génération des fractales.
 *  \image html Set.png
 */

class Generator : public QThread
{
	Q_OBJECT

	public slots:
	/*!
	*  \brief Fonction run() du thread appelant la génération d'une fractale.
	*/
	void run()
	{
		hasToStop = false;
		begin = QTime::currentTime();

		render.fill(QColor(0, 0, 0).rgb());

		if(set == "Mandelbrot")
		mandelbrot();
		if(set == "Complex Map Buddhabrot")
		buddhabrot();

		end = QTime::currentTime();
		emit genEnd();
	}
	/*!
	*  \brief Fonction de génération d'une fractale Mandelbrot.
	*/
	void mandelbrot()
	{
		int w = width;
		int h = height;

		render = QImage(QSize(w, h), QImage::Format_RGB32);

		double imagMax = (3.0) * h / w - 1.2;
		double realF = (3.0) / (w - 1);
		double imagF = (imagMax + 1.2) / (h - 1);

		for(int y = 0; y < h; y++)
		{
			std::complex<double> c;
			c.imag() = imagMax - y * imagF;
			QRgb *line = reinterpret_cast<QRgb *>(render.scanLine(y));
			for(int x = 0; x < w; x++)
			{
				c.real() = x * realF - 2.0;
				std::complex<double> z(c.real(), c.imag());
				double n;
				for(n = 0; n < it; n++)
				{
					std::complex<double> zn(z.real() * z.real(), z.imag() * z.imag());
					if(zn.real() + zn.imag() > 4) break;
					z.imag() = 2 * z.real() * z.imag() + c.imag();
					z.real() = zn.real() - zn.imag() + c.real();
				}
				double red = n / it * r;
				double green = n / it * g;
				double blue = n / it * b;
				if(n == it) line[x] = qRgb(0, 0, 0);
				else line[x] = qRgb(red, green, blue);
				if(hasToStop) return;
			}
		}
	}
	/*!
	*  \brief Fonction de génération d'une fractale Complex Map Buddhabrot.
	*/
	void buddhabrot()
	{
		int w = width;
		int h = height;

		render = QImage(QSize(w, h), QImage::Format_RGB32);

		double imagMax = (3.0) * h / w - 2.0;
		double realF = (3.0) / (w - 1);
		double imagF = (imagMax + 2.0) / (h - 1);

		int maxFreq = 0;

		int **table = new int *[h];
		for(int y = 0; y < h ; y++)
		{
			table[y] = new int[w];
			for(int x = 0; x < h ; x++)
			table[y][x] = 0;
		}
		for(int y = 0; y < h; y++)
		{
			std::complex<double> c;
			c.imag() = imagMax - y * imagF;
			for(int x = 0; x < w; x++)
			{
				c.real() = x * realF - 2.0;
				std::complex<double> z(c.real(), c.imag());
				double n;
				for(n = 0; n < it; n++)
				{
					std::complex<double> zn(z.real() * z.real(), z.imag() * z.imag());
					if(zn.real() + zn.imag() > 4) break;
					z.imag() = 2 * z.real() * z.imag() + c.imag();
					z.real() = zn.real() - zn.imag() + c.real();
					int y_ = qvariant_cast<int>(qAbs(z.imag() / imagF + h / 2));
					int x_ = qvariant_cast<int>(qAbs(z.real() / realF + w / 2));
					if(y_ < h && x_ < w)
					{
						table[y_][x_]++;
						if(table[y_][x_] > maxFreq)
						maxFreq = table[y_][x_];
					}
				}
				if(hasToStop) return;
			}
		}
		for(int y = 0; y < h; y++)
		{
			QRgb *line = reinterpret_cast<QRgb *>(render.scanLine(y));
			for(int x = 0; x < w; x++)
			{
				double red = qMin(r, qvariant_cast<int>(std::pow(qvariant_cast<double>(table[y][x]) / maxFreq, gamma) * r));
				double green = qMin(g, qvariant_cast<int>(std::pow(qvariant_cast<double>(table[y][x]) / maxFreq, gamma) * g));
				double blue = qMin(b, qvariant_cast<int>(std::pow(qvariant_cast<double>(table[y][x]) / maxFreq, gamma) * b));
				line[x] = qRgb(red, green, blue);
			}
			delete[] table[y];
		}
		delete[] table;
	}
	/*!
	*  \brief Retourne l'heure de début de la génération.
	*/
	QTime getBegin()
	{
		return begin;
	}
	/*!
	*  \brief Retourne l'heure de fin de la génération.
	*/
	QTime getEnd()
	{
		return end;
	}
	/*!
	*  \brief Retourne la fractale générée sous forme d'image.
	*/
	QImage getImage()
	{
		return render;
	}
	/*!
	*  \brief Définit les couleurs à utiliser lors de la génération.
	*  \param red : Couleur rouge à utiliser.
	*  \param green : Couleur verte à utiliser.
	*  \param blue : Couleur bleue à utiliser.
	*/
	void setColor(int red, int green, int blue)
	{
		r = red;
		g = green;
		b = blue;
	}
	/*!
	*  \brief Définit la fractale à générer.
	*  \param str : Correspond au nom de la fractale à générer.
	*/
	void setCurrentSet(QString str)
	{
		set = str;
	}
	/*!
	*  \brief Définit le gamma de l'image.
	*  Valeur inile à la génération d'une fractale Mandelbrot.
	*/
	void setGamma(double value)
	{
		gamma = value;
	}
	/*!
	*  \brief Définit le nombre d'itérations à effectuer.
	*  \param nb : Nombre d'itérations.
	*/
	void setItNum(int nb)
	{
		it = nb;
	}
	/*!
	*  \brief Définit les dimensions de l'image générée.
	*  \param w : Longueur de l'image.
	*  \param h : Hauteur de l'image.
	*/
	void setSize(int w, int h)
	{
		width = w;
		height = h;
		if(height > width) height = width;
		if(height < width) width = height;
	}
	/*!
	*  \brief Indique au thread de cesser la génération par le biais d'un boléen.
	*/
	void stop()
	{
		hasToStop = true;
	}

	signals:
	/*!
	*  \brief Signal envoyé lors de la fin de la génération.
	*/
	void genEnd();

	private:
	QTime begin;
	QTime end;
	QImage render;
	QString set;
	bool hasToStop;
	double gamma;
	int height;
	int width;
	int b;
	int g;
	int it;
	int r;
};

#endif
