/*
# Copyright (c) 2010 Louis du Verdier
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
*/

#include "MainWindow.h"

/*!
 * \file MainWindow.cpp
 * \brief Rassemble les fonctions de la classe MainWindow.
 * \author Louis du Verdier
 * \version 1.0
 */

MainWindow::MainWindow()
{
	initWindow();
	createMenus();
	createDialog();

	generator = new Generator;

	connect(this, SIGNAL(generationStateChanged(bool)), this, SLOT(showMessage(bool)));
	connect(generator, SIGNAL(genEnd()), this, SLOT(readyShowMandelbrot()));
}

void MainWindow::readyShowMandelbrot()
{
	scene->showMandelbrot(generator->getImage());

	QTime beginTime = generator->getBegin();
	QTime endTime = generator->getEnd();
	QTime totalTime = QTime(0, 0, 0, 0).addMSecs(beginTime.msecsTo(endTime));

	QString txt = "Statistiques de la dernire gnration (" + setArea->currentText() + ") :<br /><br />"
		"<strong>Dbut : </strong>" + beginTime.toString("HH") + " h " + beginTime.toString("mm")
		+ " min " + beginTime.toString("ss") + " sec " + beginTime.toString("zzz") + " msec<br />"
		"<strong>Fin : </strong>" + endTime.toString("HH") + " h " + endTime.toString("mm") + " min "
		+ endTime.toString("ss") + " sec " + endTime.toString("zzz") + " msec<br />"
		"<strong>Total : </strong>" + totalTime.toString("HH") + " h " + totalTime.toString("mm")
		+ " min " + totalTime.toString("ss") + " sec " + totalTime.toString("zzz") + " msec";

	stats->setText(txt);
	tabWidget->setCurrentIndex(statsTab);

	generateButton->setEnabled(true);
	generation = false;
	emit generationStateChanged(false);
}

void MainWindow::showMessage(bool on)
{
	if(on)
	{
		waitDialog->show();
		statusBar()->showMessage("Gnration...", 10000);
		setWindowTitle("Fractales - Gnration en cours [" + setArea->currentText() + "]");
	}
	else
	{
		statusBar()->showMessage("Termin !");
		QTimer::singleShot(1500, statusBar(), SLOT(clearMessage()));
		setWindowTitle("Fractales");
		waitDialog->hide();
	}
}

void MainWindow::initWindow()
{
	scene = new GraphicsScene(this);
	view = new GraphicsView(this);

	scene->setBackgroundBrush(Qt::black);

	view->setScene(scene);
	view->setMinimumSize(400, 300);
	view->setMouseTracking(true);

	tabWidget = new QTabWidget;
	QWidget *datas = new QWidget;
	QWidget *size_ = new QWidget;
	stats = new QLabel;
	setInfo = new QLabel;

	stats->setContentsMargins(10, 10, 10, 10);
	stats->setAlignment(Qt::AlignTop);
	stats->setWordWrap(true);
	setInfo->setContentsMargins(10, 10, 10, 10);
	setInfo->setAlignment(Qt::AlignTop);
	setInfo->setWordWrap(true);

	setArea = new QComboBox;
	setArea->addItem("Mandelbrot");
	setArea->addItem("Complex Map Buddhabrot");

	intelColo = new QCheckBox("Tenter une coloration intelligente de l'arrire-plan de la scne");
	intelColo->setChecked(false);

	nbIterationsArea = new QSpinBox;
	nbIterationsArea->setRange(0, 100000000);
	nbIterationsArea->setValue(50);

	widthArea = new QSpinBox;
	heightArea = new QSpinBox;

	windowSizeArea = new QRadioButton("Dimensions selon la taille de la fentre");
	userSizeArea = new QRadioButton("Dimensions dfinies");

	redArea = new QDial;
	greenArea = new QDial;
	blueArea = new QDial;

	redArea->setRange(0, 255);
	greenArea->setRange(0, 255);
	blueArea->setRange(0, 255);

	redArea->setNotchesVisible(true);
	greenArea->setNotchesVisible(true);
	blueArea->setNotchesVisible(true);

	redLabel = new QLabel(tr("Rouge (") + QString::number(redArea->value()) +")");
	greenLabel = new QLabel(tr("Vert (") + QString::number(greenArea->value()) +")");
	blueLabel = new QLabel(tr("Bleu (") + QString::number(blueArea->value()) +")");

	redLabel->setAlignment(Qt::AlignCenter);
	greenLabel->setAlignment(Qt::AlignCenter);
	blueLabel->setAlignment(Qt::AlignCenter);

	generateButton = new QPushButton(tr("Gnrer !"));

	QGridLayout *layoutToolsArea = new QGridLayout;
	layoutToolsArea->addWidget(redArea, 0, 0, 1, 1);
	layoutToolsArea->addWidget(greenArea, 0, 1, 1, 1);
	layoutToolsArea->addWidget(blueArea, 0, 2, 1, 1);
	layoutToolsArea->addWidget(redLabel, 1, 0, 1, 1);
	layoutToolsArea->addWidget(greenLabel, 1, 1, 1, 1);
	layoutToolsArea->addWidget(blueLabel, 1, 2, 1, 1);
	layoutToolsArea->addWidget(new QLabel(tr("Fractale  gnrer :")), 2, 0, 1, 1);
	layoutToolsArea->addWidget(setArea, 2, 1, 1, 2);
	layoutToolsArea->addWidget(new QLabel(tr("Nombre d'itrations :")), 3, 0, 1, 1);
	layoutToolsArea->addWidget(nbIterationsArea, 3, 1, 1, 2);
	layoutToolsArea->addWidget(intelColo, 4, 0, 1, 3);
	layoutToolsArea->addWidget(generateButton, 5, 0, 1, 3);
	layoutToolsArea->addItem(new QSpacerItem(1, 10, QSizePolicy::Minimum, QSizePolicy::Expanding), 6, 0, 1, 3);

	QGridLayout *layoutSizeArea = new QGridLayout;
	layoutSizeArea->addWidget(windowSizeArea, 0, 0, 1, 2);
	layoutSizeArea->addWidget(userSizeArea, 1, 0, 1, 2);
	layoutSizeArea->addWidget(new QLabel("Longueur :"), 2, 0, 1, 1);
	layoutSizeArea->addWidget(widthArea, 2, 1, 1, 1);
	layoutSizeArea->addWidget(new QLabel("Largeur :"), 3, 0, 1, 1);
	layoutSizeArea->addWidget(heightArea, 3, 1, 1, 1);
	layoutSizeArea->addItem(new QSpacerItem(1, 10, QSizePolicy::Minimum, QSizePolicy::Expanding), 4, 0, 1, 2);

	toolsArea = new ToolsArea(this);
	toolsArea->setWidget(tabWidget);

	tabWidget->setMinimumWidth(350);
	tabWidget->addTab(datas, tr("Informations"));
	tabWidget->addTab(size_, tr("Dimensions"));
	statsTab = tabWidget->addTab(stats, tr("Statistiques"));
	tabWidget->addTab(setInfo, tr("Fractale"));

	datas->setLayout(layoutToolsArea);
	size_->setLayout(layoutSizeArea);

	connect(view, SIGNAL(mouseEnter()), toolsArea, SLOT(setEffect()));
	connect(view, SIGNAL(mouseLeave()), toolsArea, SLOT(unsetEffect()));
	connect(userSizeArea, SIGNAL(toggled(bool)), widthArea, SLOT(setEnabled(bool)));
	connect(userSizeArea, SIGNAL(toggled(bool)), heightArea, SLOT(setEnabled(bool)));
	connect(widthArea, SIGNAL(editingFinished()), this, SLOT(sizeChanges()));
	connect(heightArea, SIGNAL(editingFinished()), this, SLOT(sizeChanges()));
	connect(redArea, SIGNAL(valueChanged(int)), this, SLOT(assignValue(int)));
	connect(greenArea, SIGNAL(valueChanged(int)), this, SLOT(assignValue(int)));
	connect(blueArea, SIGNAL(valueChanged(int)), this, SLOT(assignValue(int)));
	connect(generateButton, SIGNAL(clicked()), this, SLOT(generate()));
	connect(setArea, SIGNAL(currentIndexChanged(QString)), this, SLOT(displayInfo(QString)));

	redArea->setValue(255);
	greenArea->setValue(255);
	blueArea->setValue(255);

	windowSizeArea->setChecked(true);
	widthArea->setEnabled(false);
	widthArea->setValue(width());
	widthArea->setRange(400, 10000);
	heightArea->setEnabled(false);
	heightArea->setValue(height());
	heightArea->setRange(300, 10000);

	setCentralWidget(view);

	addDockWidget(Qt::RightDockWidgetArea, toolsArea);

	positionArea = new QLabel;
	positionArea->setFixedWidth(125);
	positionArea->setAlignment(Qt::AlignRight);
	positionArea->setFrameShape(QFrame::NoFrame);
	statusBar()->addPermanentWidget(positionArea, 2);

	displayInfo(setArea->currentText());
}

void MainWindow::createMenus()
{
	QMenu *file = menuBar()->addMenu(tr("&Fichier"));
	QMenu *window = menuBar()->addMenu(tr("&Fentre"));
	QMenu *scene_m = menuBar()->addMenu(tr("&Scne"));
	QMenu *help = menuBar()->addMenu(tr("&?"));

	QAction *saveAct = file->addAction(tr("Enregistrer la scne"), this, SLOT(saveScene()));
	QAction *exitAct = file->addAction(tr("Quitter"), qApp, SLOT(quit()));
	QAction *toolsAct = window->addAction(tr("Zone d'outils"));
	QAction *aboutAct = help->addAction(tr("A propos"), this, SLOT(about()));

	scene_m->addAction(tr("Nettoyer la fentre"), scene, SLOT(clear()));

	toolsAct->setCheckable(true);
	toolsAct->setChecked(toolsArea->isVisible());

	saveAct->setShortcut(QKeySequence("Ctrl+S"));
	exitAct->setShortcut(QKeySequence("Ctrl+Q"));
	aboutAct->setShortcut(QKeySequence("F1"));

	connect(toolsArea, SIGNAL(visibilityChanged(bool)), toolsAct, SLOT(setChecked(bool)));
	connect(toolsAct, SIGNAL(triggered(bool)), toolsAct, SLOT(setChecked(bool)));
	connect(toolsAct, SIGNAL(triggered(bool)), toolsArea, SLOT(setVisible(bool)));
}

void MainWindow::createDialog()
{
	waitDialog = new QProgressDialog("Gnration en cours...", 0, 0, 0);
	waitDialog->setWindowFlags(Qt::SubWindow|Qt::CustomizeWindowHint);
	waitDialog->setWindowTitle("Gnration");
	waitDialog->setMinimumDuration(0);
	QProgressBar *bar = new QProgressBar(waitDialog);
	bar->setRange(0, 0);
	bar->setStyleSheet("text-align: center;");
	waitDialog->setBar(bar);
}

void MainWindow::displayInfo(QString set)
{
	if(set == "Mandelbrot")
	{
		setInfo->setText("Informations concernant la fractale Mandelbrot :<br /><br />"
		"<strong>Suite :</strong> z<sub>n+1</sub> = zn<sup>2</sup> + c<br/>"
		"<strong>Itrations conseilles :</strong> 50<br />"
		"<strong>Temps de gnration<sup>(*)</sup> :</strong> Moins d'une seconde<br />"
		"<strong>Coloration intelligente efficace :</strong> Oui<br /><br />"
		"<sup>(*)</sup> Avec une machine rcente pour une image de 400x400px, "
		"gnre selon le nombre d'itrations conseilles.");
	}
	if(set == "Complex Map Buddhabrot")
	{
		setInfo->setText("Informations concernant la fractale Complex Map Buddhabrot :<br /><br />"
		"<strong>Suite :</strong> z<sub>n+1</sub> = zn<sup>2</sup> + c<br/>"
		"<strong>Itrations conseilles :</strong> 500<br />"
		"<strong>Temps de gnration<sup>(*)</sup> :</strong> Moins de cinq secondes<br />"
		"<strong>Coloration intelligente efficace :</strong> Non<br /><br />"
		"<sup>(*)</sup> Avec une machine rcente pour une image de 400x400px, "
		"gnre selon le nombre d'itrations conseilles.");
	}
}

void MainWindow::assignValue(int value)
{
	QDial *dial = qobject_cast<QDial *>(sender());
	if(!dial) return;
	if(dial == redArea)
	redLabel->setText("Rouge (" + QString::number(value) +")");
	if(dial == greenArea)
	greenLabel->setText("Vert (" + QString::number(value) +")");
	if(dial == blueArea)
	blueLabel->setText("Bleu (" + QString::number(value) +")");
}

void MainWindow::about()
{
	QMessageBox::information(this, tr("A propos"),
		tr("Ce logiciel a t ralis par Louis du Verdier, alias Amnell, "
		"pour le premier dfi Qt de <a href='http://qt.developpez.com/'>Developpez.com</a>.<br /><br />"
		"<strong>Compilateur :</strong> GNU GCC 4.4.0 (port MinGW)<br />"
		"<strong>Version de Qt :</strong> Qt 4.6.0<br /><br />"
		"Consultez le fichier README.txt pour plus d'informations."));
}

void MainWindow::sizeChanges()
{
	resized = true;
	int w = widthArea->value();
	int h = heightArea->value();
	QList<QGraphicsItem *> list = scene->items();
	scene->setSceneRect(QRect(QPoint(-1 * w / 2, -1 * h / 2), QSize(w, h)));
	for(int i = 0; i < list.size(); i++)
	list[i]->setPos(list[i]->pos());
}

void MainWindow::generate()
{
	generateButton->setEnabled(false);
	scene->clear();

	resized = false;
	generation = true;
	emit generationStateChanged(true);

	generator->setColor(redArea->value(), greenArea->value(), blueArea->value());
	generator->setCurrentSet(setArea->currentText());
	generator->setGamma(0.8);
	generator->setItNum(nbIterationsArea->value());
	generator->setSize(widthArea->value(), heightArea->value());
	generator->start();
}

void MainWindow::saveScene()
{
	QString path = QFileDialog::getSaveFileName(this, tr("Enregistrer"),
		QDesktopServices::storageLocation(QDesktopServices::DocumentsLocation),
		tr("PNG (*.png);;JPEG (*.jpg);; Fichier (*.)"));

	if(path.isEmpty()) return;

	QPixmap pixmap(scene->width(), scene->height());
	QPainter painter(&pixmap);
	painter.setRenderHint(QPainter::HighQualityAntialiasing);
	scene->render(&painter);

	(pixmap.save(path)) ? QMessageBox::information(this, windowTitle(), tr("Sauvegarde effectue avec succs !"))
		: QMessageBox::critical(this, tr("Erreur"), tr("Impossible d'enregistrer la scne."));
}

void MainWindow::closeEvent(QCloseEvent *event)
{
	generator->stop();
	waitDialog->close();
	event->accept();
}

void MainWindow::resizeEvent(QResizeEvent *)
{
	int w = view->width() - 10;
	int h = view->height() - 10;
	widthArea->setMinimum(w);
	heightArea->setMinimum(h);
	if(!windowSizeArea->isChecked()) return;
	resized = true;
	QList<QGraphicsItem *> list = scene->items();
	scene->setSceneRect(QRect(QPoint(-1 * w / 2, -1 * h / 2), QSize(w, h)));
	for(int i = 0; i < list.size(); i++)
	list[i]->setPos(list[i]->pos());
}

MainWindow::~MainWindow()
{
	delete scene;
	delete view;
}
