/*
# Copyright (c) 2010 Louis du Verdier
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
*/

#ifndef MAINWINDOW_H
#define MAINWINDOW_H

#include <QtGui>
#include <complex>
#include "Generator.h"

/*!
 * \file MainWindow.h
 * \brief Rassemble toutes les déclarations de classe.
 * \author Louis du Verdier
 * \version 1.0
 */

class ToolsArea;
class GraphicsScene;
class GraphicsView;
class MainWindow;
class Generator;

/*! \class ToolsArea
 *  \brief La classe ToolsArea correspond à la fenêtre d'outils.
 */

class ToolsArea : public QDockWidget
{
	Q_OBJECT

	public:
	/*!
	*  \brief Constructeur.
	*  Initialise les variables nécessaires au bon fonctionnement de la classe.
	*/
	ToolsArea(MainWindow *p = 0);

	public slots:
	/*!
	*  \brief Slot appliquant l'effet de colorisation sur l'instance de la classe.
	*/
	void setEffect();
	/*!
	*  \brief Slot retirant l'effet de colorisation de l'instance de la classe.
	*/
	void unsetEffect();

	private:
	MainWindow *parent;
	QGraphicsColorizeEffect *effect;
	QGraphicsColorizeEffect *_effect;
};

/*! \class GraphicsScene
 *  \brief La classe GraphicsScene, dérivée de QGraphicsScene, fournit une scène à la fenêtre principale.
 */

class GraphicsScene : public QGraphicsScene
{
	Q_OBJECT

	public:
	/*!
	*  \brief Constructeur.
	*  Initialise les variables nécessaires au bon fonctionnement de la classe.
	*/
	GraphicsScene(MainWindow *p = 0);
	MainWindow *parent; /*!< Instance parente*/
	/*!
	*  \brief Fonction appelant l'affichage d'une fractale Mandelbrot.
	*  \param set : Image représentant la fractale.
	*/
	void showMandelbrot(QImage set);
	/*!
	*  \brief Fonction s'occupant de la coloration dite intelligente de la scène.
	*  Fournit à la scène une ergonomie d'autant meilleure.
	*  \param set : Image représentant la fractale, nécessaire à la coloration.
	*/
	void changeSceneBgColor(QImage set);
	QGraphicsPixmapItem *renderItem; /*!< Instance de QGraphicsPixmapItem ayant pour utilité d'afficher la fractale*/

	protected:
	/*!
	*  \brief Evènement de mouvement de la souris.
	*  Permet d'actualiser les informations du parent concernant la position de la souris dans la scène.
	*/
	void mouseMoveEvent(QGraphicsSceneMouseEvent *);
};

/*! \class GraphicsView
 *  \brief La classe GraphicsView, dérivée de QGraphicsView, fournit une vue à la fenêtre principale.
 *  Elle est utilisée en harmonie avec la classe ToolsArea et avec la classe MainWindow, signalant une entrée et une sortie de la souris.
 */

class GraphicsView : public QGraphicsView
{
	Q_OBJECT

	public:
	/*!
	*  \brief Constructeur.
	*  Initialise les variables nécessaires au bon fonctionnement de la classe.
	*/
	GraphicsView(MainWindow *p = 0);

	signals:
	/*!
	*  \brief Signal envoyé lors d'une entrée de la souris.
	*/
	void mouseEnter();
	/*!
	*  \brief Signal envoyé lors d'une sortie de la souris.
	*/
	void mouseLeave();

	protected:
	/*!
	*  \brief Evènement d'entrée de la souris.
	*  Appelle le signal mouseEnter().
	*/
	void enterEvent(QEvent *);
	/*!
	*  \brief Evènement de sortie de la souris.
	*  Appelle le signal mouseLeave().
	*/
	void leaveEvent(QEvent *);
	/*!
	*  \brief Evènement ouvrant un menu contextuel selon la valeur de la variable GraphicsScene::generation.
	*/
	void contextMenuEvent(QContextMenuEvent *);

	private:
	MainWindow *parent;
};

/*! \class MainWindow
 *  \brief La classe MainWindow, dérivée de QMainWindow, correspond à la fenêtre principale.
 */

class MainWindow : public QMainWindow
{
	Q_OBJECT

	public:
	/*!
	*  \brief Constructeur.
	*  Initialise les variables nécessaires au bon fonctionnement de la classe, et initialise la fenêtre elle-même.
	*/
	MainWindow();
	/*!
	*  \brief Destructeur.
	*/
	~MainWindow();
	/*!
	*  \brief Fonction retournant si la fenêtre principale a été fermée.
	*/
	bool resized; /*!< Précise si la fenêtre principale vient d'être redimensionnée*/
	bool generation; /*!< Booléen mis à true si la génération est en cours*/
	bool closed; /*!< Booléen mis à true si la fenêtre principale est fermée*/
	GraphicsScene *scene; /*!< Correspond à la scène*/
	GraphicsView *view; /*!< Correspond à la vue*/
	QLabel *positionArea; /*!< Label renseignant la position de la souris dans la scène quand cette dernière la survole*/
	QLabel *messageArea; /*!< Label renseignant si l'état de la génération*/
	QComboBox *setArea; /*!< Instance de QComboBox permettant d'indiquer quelle fractale générer*/
	QCheckBox *intelColo; /*!< Instance de QCheckBox permettant d'indiquer si une coloration intelligente de l'arrière-plan de la scène doit être faite*/
	QRadioButton *windowSizeArea; /*!< Instance de QRadioButton permettant d'indiquer que l'image générée doit être en accord avec les coordonnées de la fenêtre*/
	QRadioButton *userSizeArea; /*!< Instance de QRadioButton permettan d'indiquer que l'image générée doit être aux dimensions de l'utilisateur*/
	QSpinBox *widthArea; /*!< Instance de QSpinBox permettant à l'utilisateur de renseigner la longueur de l'image générée*/
	QSpinBox *heightArea; /*!< Instance de QSpinBox permettant à l'utilisateur de renseigner la hauteur de l'image générée*/

	signals:
	/*!
	*  \brief Signal envoyé lorsque l'état de la génération change (début et fin).
	*/
	void generationStateChanged(bool);

	public slots:
	/*!
	*  \brief Slot utilisé pour lancer la génération.
	*/
	void generate();
	/*!
	*  \brief Fonction appelée lors de la fin de la génération de la fractale Mandelbrot.
	*/
	void readyShowMandelbrot();

	private slots:
	void assignValue(int);
	void displayInfo(QString);
	void showMessage(bool);
	void sizeChanges();
	void about();
	void saveScene();

	private:
	int statsTab;
	void initWindow();
	void createMenus();
	void createDialog();
	ToolsArea *toolsArea;
	Generator *generator;
	QSpinBox *nbIterationsArea;
	QDial *redArea;
	QDial *greenArea;
	QDial *blueArea;
	QLabel *redLabel;
	QLabel *greenLabel;
	QLabel *blueLabel;
	QLabel *stats;
	QLabel *setInfo;
	QProgressDialog *waitDialog;
	QPushButton *generateButton;
	QTabWidget *tabWidget;

	protected:
	/*!
	*  \brief Evènement de fermeture.
	*  Agit de manière à mettre un terme à l'exécution de l'application.
	*/
	void closeEvent(QCloseEvent *);
	/*!
	*  \brief Evènement de redimension.
	*  Modifie la taille de la scène en fonction des nouvelles dimensions de la fenêtre.
	*/
	void resizeEvent(QResizeEvent *);
	/*!
	*  \brief Evènement d'affichage.
	*  Centre la fenêtre à l'écran.
	*/
	void showEvent(QShowEvent *)
	{
		int x = QApplication::desktop()->rect().center().x() - width() / 2;
		int y = QApplication::desktop()->rect().center().y() - height() / 2;
		move(x, y);
	}
};

#endif
