/* Copyright (c) 2010(Hennequin Thibault) Permission is hereby granted, free of charge,
to any person obtaining a copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation the rights to use, copy,
modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. */

#include "brotdisplay.h"
#include <QPainter>
#include <QMessageBox>
#include "fractal.h"

const double ZoomFactor=0.8;				

/**
 *
 * \param parent du Widget
 * \return 
 */
BrotDisplay::BrotDisplay(QWidget *parent)
: QGLWidget(parent)
{
	SetFractal(Fractal_MandelBrot);
	m_fZoom = 0.005;
	m_fPosX = 0;
	m_fPosY = 0;
	m_iIteration = 100;
	m_bDrag = false;
	m_iRenderFinishCount = 0;
	m_fPrecision = 1.0;
	m_iMinIteration = 0;
	m_fReal = 0.0;
	m_fImaginary = 0.0;
	m_fDilatation = 1.0;
	m_iTexture = 0;
	m_bAnimatedRender = true;
	m_bSmoothColors = true;
	m_bWaitRender = true;

	m_fCurrentZoom = m_fZoom;

	connect(&m_Timer, SIGNAL(timeout()), this, SLOT(OnTimer()));

	SetupThread(1);
}

/**
 *
 * \return 
 */
BrotDisplay::~BrotDisplay()
{
#if QT_VERSION >= 0x040600
	delete m_pShader[0];
	delete m_pFragmentShader[0];
	delete m_pShader[1];
	delete m_pFragmentShader[1];
	delete m_pVertexShader;
#endif

	SetupThread(0);
}

/** Fonction de dessin du widget
 *
 * \param QPaintEvent* 
 */
void BrotDisplay::paintEvent(QPaintEvent*)
{
	if (m_iFractal<Fractal_MandelBrot_OpenGL)
	{
		QPainter painter(this);
		m_ImageMutex.lock();
		painter.setBrush(Qt::SolidPattern);
		painter.drawRect(0,0,width(),height());
		if (!m_bWaitRender)
		{ 
			painter.drawImage(0,0,m_Image);	
		}else{
			float ratio = (float)width()/height();

			float x = width()/2.0-(m_fRenderZoom/m_fCurrentZoom)*width()/2.0;
			float y = height()/2.0-(m_fRenderZoom/m_fCurrentZoom)*height()/2.0;
			x += (m_fRenderPosX-m_fPosX)/m_fCurrentZoom*(width()/512.0/ratio);
			y += (m_fRenderPosY-m_fPosY)/m_fCurrentZoom*(height()/512.0);
			int w = width()*(m_fRenderZoom/m_fCurrentZoom);
			int h = height()*(m_fRenderZoom/m_fCurrentZoom);
			painter.drawImage(QRect(x, y, w, h), m_Image);
		}
		m_ImageMutex.unlock();
	}else{
		updateGL();
	}
}

/** Fonction d'vnement du timer pour l'animation du zoom
 *
 */
void BrotDisplay::OnTimer()
{	
	double val = (m_fZoom-m_fCurrentZoom)/3.0;
	if (abs(val)<0.00000000000000001 || m_iZoomCount>=20)
	{
		m_fCurrentZoom = m_fZoom;
		m_Timer.stop();
	}else{
		m_fCurrentZoom+=val;
	}
	m_iZoomCount++;
	repaint();
}

/** Fonction de redimensionnement du Widget
 *
 * \param *event 
 */
void BrotDisplay::resizeEvent( QResizeEvent *event )
{
	Cancel();
	QMutexLocker locker(&m_ImageMutex);
	m_Image = QImage(width(),height(),QImage::Format_RGB32);
	QGLWidget::resizeEvent(event);
}

/** Fonction dvnement de la molette
 *
 * \param *event 
 */
void BrotDisplay::wheelEvent(QWheelEvent *event)
{
	if (event->delta()>0)
	{
		SetZoom(m_fZoom*ZoomFactor);
	}else{
		SetZoom(m_fZoom/ZoomFactor);
	}
}

/** Fonction dvnement du clic de la souris
 *
 * \param *event 
 */
void BrotDisplay::mousePressEvent(QMouseEvent *event)
{
	if (m_bWaitRender)
	{
		m_bDrag = true;
		m_MousePoint = event->globalPos();
		m_fOrigX = m_fPosX;
		m_fOrigY = m_fPosY;
	}
}

/** Fonction dvnement du lacher du clic de la souris
 *
 * \param QMouseEvent* 
 */
void BrotDisplay::mouseReleaseEvent(QMouseEvent*)
{
	m_bDrag = false;
	if (m_bWaitRender && (m_fPosX!=m_fOrigX || m_fPosY!=m_fOrigY))
	{
		emit PosXChanged(m_fPosX);
		emit PosYChanged(m_fPosY);
	}
}

/** Fonction dvnement du mouvement de la souris
 *
 * \param *event 
 */
void BrotDisplay::mouseMoveEvent(QMouseEvent *event)
{
	if (m_bDrag && m_bWaitRender)
	{
		QPoint temp = m_MousePoint - event->globalPos();
		m_fPosX = m_fOrigX + temp.x()*m_fZoom;
		m_fPosY = m_fOrigY + temp.y()*m_fZoom;
		repaint();
	}
}

/** Initialise OpenGL ainsi que les shader du MandelBrot et Julia
 *
 */
void BrotDisplay::initializeGL()
{
#if QT_VERSION >= 0x040600
	glEnable(GL_TEXTURE_2D);

	m_pVertexShader = new QGLShader(QGLShader::Vertex, this);
	m_pVertexShader->compileSourceCode(LoadResTxt("default.vert"));

	m_pFragmentShader[0] = new QGLShader(QGLShader::Fragment, this);
	m_pFragmentShader[0]->compileSourceCode(LoadResTxt("mandelbrot.frag"));

	m_pShader[0] = new QGLShaderProgram(this);
	m_pShader[0]->addShader(m_pVertexShader);
	m_pShader[0]->addShader(m_pFragmentShader[0]);
	m_pShader[0]->link();

	m_pFragmentShader[1] = new QGLShader(QGLShader::Fragment, this);
	m_pFragmentShader[1]->compileSourceCode(LoadResTxt("julia.frag"));

	m_pShader[1] = new QGLShaderProgram(this);
	m_pShader[1]->addShader(m_pVertexShader);
	m_pShader[1]->addShader(m_pFragmentShader[1]);
	m_pShader[1]->link();
#endif
}

/** Fonction de dessin en OpenGL
 *
 */
void BrotDisplay::paintGL()
{
	if (m_iFractal<Fractal_MandelBrot_OpenGL)
		return;
#if QT_VERSION >= 0x040600
	qglClearColor(QColor(0,0,0));
	glClear(/*GL_COLOR_BUFFER_BIT |*/ GL_DEPTH_BUFFER_BIT);

	int shaderId=0;
	if (m_iFractal==Fractal_MandelBrot_OpenGL)
	{
		shaderId=0;
	}else if (m_iFractal==Fractal_Julia_OpenGL)
	{
		shaderId=1;
	}

	m_pShader[shaderId]->bind();

	m_pShader[shaderId]->setUniformValue("width",width());
	m_pShader[shaderId]->setUniformValue("height",height());

	m_pShader[shaderId]->setUniformValue("zoom",(float)m_fZoom);
	m_pShader[shaderId]->setUniformValue("posx",(float)m_fPosX);
	m_pShader[shaderId]->setUniformValue("posy",(float)m_fPosY);

	m_pShader[shaderId]->setUniformValue("max_iter",m_iIteration);

	m_pShader[shaderId]->setUniformValue("texture",0);
	m_pShader[shaderId]->setUniformValue("smoothcolors",m_bSmoothColors);
	m_pShader[shaderId]->setUniformValue("dilatation",(float)m_fDilatation);

	if (m_iFractal==Fractal_Julia_OpenGL)
	{
		m_pShader[shaderId]->setUniformValue("real",(float)m_fReal);
		m_pShader[shaderId]->setUniformValue("imag",(float)m_fImaginary);
		/*
		m_pShader[shaderId]->setUniformValue("real",(float)0.285);
		m_pShader[shaderId]->setUniformValue("imag",(float)0.01);

		m_pShader[shaderId]->setUniformValue("real",(float)-0.4);
		m_pShader[shaderId]->setUniformValue("imag",(float)0.6);

		m_pShader[shaderId]->setUniformValue("real",(float)-0.8);
		m_pShader[shaderId]->setUniformValue("imag",(float)0.156);

		m_pShader[shaderId]->setUniformValue("real",(float)-0.835);
		m_pShader[shaderId]->setUniformValue("imag",(float)-0.2321);

		m_pShader[shaderId]->setUniformValue("real",(float)-0.70176);
		m_pShader[shaderId]->setUniformValue("imag",(float)0.3842);

		m_pShader[shaderId]->setUniformValue("real",(float)0.25);
		m_pShader[shaderId]->setUniformValue("imag",(float)0.528);
		*/
	}
	if (m_iTexture==0)
	{
		GenerateTexture();
	}

	glBindTexture(GL_TEXTURE_2D, m_iTexture);

	glBegin(GL_QUADS);
		glTexCoord2f(0,0);
		glVertex2f(0,0);

		glTexCoord2f(width(),0);
		glVertex2f(1,0);

		glTexCoord2f(width(),height());
		glVertex2f(1,1);

		glTexCoord2f(0,height());
		glVertex2f(0,1);
	glEnd();
#endif
}

/** Fonction de redimensionnement de la fenetre OpenGL
 *
 * \param width 
 * \param height 
 */
void BrotDisplay::resizeGL(int width, int height)
{
	m_fRatio = (float)width/height;
	glViewport(0,0,width,height);

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glOrtho(0, 1, 1, 0, -1.0, 1.0);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
}

/** Dfinie la position en horizontal du centre de la fractale
 *
 * \param x 
 */
void BrotDisplay::SetPosX(double x)
{
	if (m_fPosX!=x)
	{
		m_fPosX=x;
		emit PosXChanged(m_fPosX);
	}
}

/** Dfinie la position en vertical du centre de la fractale
 *
 * \param y 
 */
void BrotDisplay::SetPosY(double y)
{
	if (m_fPosY!=y)
	{
		m_fPosY=y;
		emit PosYChanged(m_fPosY);
	}
}

/** Dfinie le zoom appliqu  la fractale
 *
 * \param z 
 */
void BrotDisplay::SetZoom(double z)
{
	m_fZoom=z;
	//Limitation du zoom pour cause de prcision
	if (m_fZoom<0.0000000000000001)
		m_fZoom=0.0000000000000001;
	emit ZoomChanged(m_fZoom);
	repaint();
	// On anime le zoom sauf en mode OpenGL
	if (m_iFractal<Fractal_MandelBrot_OpenGL)
	{
		if (m_Timer.isActive()==false)
		{
			m_Timer.start(50);
			m_iZoomCount=0;
		}
	}
}


/** Dfinie le nombre d'itratio max  pour la fractale
 *
 * \param ite 
 */
void BrotDisplay::SetIteration(int ite)
{
	if (m_iIteration!=ite)
	{
		m_iIteration=ite;
		emit IterationChanged(m_iIteration);
	}
}

/** Dfinie le type de fractale  dssin
 *
 * \param frac 
 */
void BrotDisplay::SetFractal(Fractal frac)
{
	if (m_iFractal!=frac)
	{
		m_iFractal=(Fractal)frac;
		m_Image = QImage(width(),height(),QImage::Format_RGB32);
		/* On arrte le rendu dans les thread */
		Cancel();
		repaint();
	}
}

/** Dfinie le nombre de threads utiliss pour le rendue software
 *
 * \param count 
 */
void BrotDisplay::SetThreadCount( int count )
{
	if (m_RenderThreads.size()!=count)
	{
		SetupThread(count);
		emit ThreadCountChanged(count);
	}
}

/** Dfinie la prcision utilis pour le rendu du Buddhabrot
 *
 * \param precision 
 */
void BrotDisplay::SetPrecision( double precision )
{
	if (m_fPrecision!=precision)
	{
		m_fPrecision=precision;
		emit PrecisionChanged(precision);
	}
}

/** Dfinie le nombre minimum d'itration pour le Buddhabrot
 *
 * \param min 
 */
void BrotDisplay::SetMinIteration( int min )
{
	if (m_iMinIteration!=min)
	{
		m_iMinIteration=min;
		//emit MinIterationChanged(min);
	}
}

/** Dfinie la valeur rel du complex utilis pour le rendu du Julia
 *
 * \param real 
 */
void BrotDisplay::SetReal(double real)
{
	if (m_fReal!=real)
	{
		m_fReal=real;
		emit RealChanged(m_fReal);
		if (m_iFractal>=Fractal_MandelBrot_OpenGL)
		{
			repaint();
		}
	}
}

/** Dfinie la valeur imaginaire du complex utilis pour le rendu du Julia
 *
 * \param imaginary 
 */
void BrotDisplay::SetImaginary(double imaginary)
{
	if (m_fImaginary!=imaginary)
	{
		m_fImaginary=imaginary;
		emit ImaginaryChanged(m_fImaginary);
		if (m_iFractal>=Fractal_MandelBrot_OpenGL)
		{
			repaint();	
		}
	}
}

/** Dfinie le gradient utilis pour les couleurs des fractales
 *
 * \param stops 
 */
void BrotDisplay::SetGradient( const QGradientStops& stops )
{
	m_Gradient = stops;

	m_iTexture = 0;

	const int gradientWidth = 1024*16;
	/* On dssine le gradient dans une image pour viter de calculer le gradient pour chaque points */
	m_GradientImage = QImage(gradientWidth,1,QImage::Format_RGB32);
	QPainter painter(&m_GradientImage);
	QLinearGradient gradient(0, 0, gradientWidth, 0);
	for (int i=0; i<m_Gradient.size(); ++i)
	{
		QColor c = m_Gradient.at(i).second;
		gradient.setColorAt(m_Gradient.at(i).first, QColor(c.red(), c.green(), c.blue()));
	}
	painter.fillRect(0,0,gradientWidth, 1, gradient);

	if (m_iFractal>=Fractal_MandelBrot_OpenGL)
	{
		repaint();	
	}
}

/** Dfinie la dilatation du gradient
 *
 * \param dilatation 
 */
void BrotDisplay::SetDilatation( double dilatation )
{
	if (m_fDilatation!=dilatation)
	{
		m_fDilatation = dilatation;
		emit DilatationChanged(dilatation);
		if (m_iFractal>=Fractal_MandelBrot_OpenGL)
		{
			repaint();
		}
	}
}

/** Active ou dsactive le Normalized Iteration Count Algorithm
 *
 * \param smooth 
 */
void BrotDisplay::SetSmoothColors( bool smooth )
{
	if (m_bSmoothColors!=smooth)
	{
		m_bSmoothColors = smooth;
		emit SmoothColorsChanged(smooth);
		if (m_iFractal>=Fractal_MandelBrot_OpenGL)
		{
			repaint();
		}
	}
}

/** Active ou dsactive l'affichage du rendu en temps rel
 *
 * \param animated 
 */
void BrotDisplay::SetAnimatedRender( bool animated )
{
	if (m_bAnimatedRender!=animated)
	{
		m_bAnimatedRender = animated;
		emit AnimatedRenderChanged(animated);
	}
}

/** Initialise le nombre de threads utiliss
 *
 * \param count 
 */
void BrotDisplay::SetupThread( int count )
{
	for (int i = 0; i < m_RenderThreads.count(); ++i)
	{
		if (m_RenderThreads.at(i))
		{
			disconnect( m_RenderThreads.at(i),SIGNAL(Finish(int)), this, SLOT(OnThreadFinish(int)));
			disconnect( m_RenderThreads.at(i),SIGNAL(Progress(float, int)), this, SLOT(OnThreadProgress(float, int)));
			m_RenderThreads.at(i)->Finish();
		}
	}
	for (int i = 0; i < m_RenderThreads.count(); ++i)
	{
		if (m_RenderThreads.at(i))
		{			
			m_RenderThreads.at(i)->wait();
		}
		delete m_RenderThreads.at(i);
	}
	m_RenderThreads.clear();
	for (int i=0;i<count;i++)
	{
		RenderThread *thread = new RenderThread(this,i);
		connect( thread,SIGNAL(Finish(int)), this, SLOT(OnThreadFinish(int)));
		connect( thread,SIGNAL(Progress(float, int)), this, SLOT(OnThreadProgress(float, int)));
		m_RenderThreads.push_back(thread);
	}
	m_RenderProgress.resize(count);
}

/** Fonction d'vnement lorsque un thread est termin
 *
 * \param int 
 */
void BrotDisplay::OnThreadFinish(int)
{
	m_iRenderFinishCount++;
	if (m_iRenderFinishCount>=m_RenderThreads.count())
	{
		m_bWaitRender = true;
		repaint();
		emit Progress(1.0);
	}
}

/**  Fonction d'vnement lorsque un thread est en cours de rendu
 *
 * \param value 
 * \param id 
 */
void BrotDisplay::OnThreadProgress(float value, int id)
{
	if (id<m_RenderProgress.count())
	{
		m_RenderProgress[id]=value;
	}
	float total=0.0;
	for (int i = 0; i < m_RenderProgress.size(); ++i)
	{
		total+=m_RenderProgress[i];
	}

	if (m_bAnimatedRender==true && m_iLastPaint.elapsed()>=100)
	{
		m_iLastPaint.restart();
		repaint();
	}

	emit Progress(total/m_RenderProgress.count());
}


/** Fonction premettant de (re)initialis le rendu software ou OpenGL
 *
 */
void BrotDisplay::RedrawFractal()
{
	if (m_iFractal<Fractal_MandelBrot_OpenGL)
	{
		m_iRenderFinishCount = 0;

		m_bWaitRender = false;
		m_fRenderPosX = m_fPosX;
		m_fRenderPosY = m_fPosY;
		m_fRenderZoom = m_fZoom;
		m_iLastPaint.restart();
		m_Image = QImage(width(),height(),QImage::Format_RGB32);
		m_TempGradientImage = m_GradientImage;

		for (int i = 0; i < m_RenderThreads.count(); ++i)
		{
			m_RenderProgress[i]=0.0;	 
			if (m_RenderThreads.at(i))
			{
				m_RenderThreads.at(i)->SetRender(width(), height(),
					m_RenderThreads.count(),
					m_fPosX, m_fPosY,
					m_fPrecision, m_iMinIteration,
					m_fReal, m_fImaginary,
					m_fZoom, m_iIteration, m_iFractal,
					m_Gradient, &m_TempGradientImage, m_bSmoothColors, m_fDilatation,
					&m_Image, &m_ImageMutex);
			}
		}
	}else{
		m_bWaitRender = true;
		update();
	}
}

/** Charge un fichier texte inclut dans les ressources de l'application
 *
 * \param *file 
 * \return 
 */
QString BrotDisplay::LoadResTxt(const char *file)
{
	QFile inputFile(QString(":/")+file);
	inputFile.open(QIODevice::ReadOnly);

	QTextStream in(&inputFile);
	QString line = in.readAll();
	inputFile.close();
	return line;
}

/** Stope le rendu des threads
 *
 */
void BrotDisplay::Cancel()
{
	for (int i = 0; i < m_RenderThreads.count(); ++i)
	{
		if (m_RenderThreads.at(i))
		{
			m_RenderThreads.at(i)->Stop();
		}
	}
}

/** Sauvegarde le rendu dans une image PNG
 *
 * \param file 
 * \return 
 */
bool BrotDisplay::Save(QString file)
{
	if (m_iRenderFinishCount>=m_RenderThreads.count() || m_iFractal>=Fractal_MandelBrot_OpenGL)
	{
		if (m_iFractal>=Fractal_MandelBrot_OpenGL)
		{
			return grabFrameBuffer().save(file,"PNG");
		}else{
			return m_Image.save(file,"PNG");
		}
	}else{
		QMessageBox box;
		box.setText(tr("CantDuringProcess"));
		box.exec();
		return false;
	}
}

/** Copie l'image du gradient dans une texture OpenGL
 *
 */
void BrotDisplay::GenerateTexture()
{
#if QT_VERSION >= 0x040600
	/* On redimensionne la texture pour ne pas dpasser la taille max d'une texture sous OpenGL */
	QImage temp = m_GradientImage.scaled(4096,1);
	m_iTexture = bindTexture(temp, GL_TEXTURE_2D);	
#endif
}
