/* Copyright (c) 2010(Hennequin Thibault) Permission is hereby granted, free of charge,
to any person obtaining a copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation the rights to use, copy,
modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. */


#include "gradient.h"

ShadeWidget::ShadeWidget(ShadeType type, QWidget *parent)
: QWidget(parent), m_shade_type(type), m_alpha_gradient(QLinearGradient(0, 0, 0, 0))
{

	// Checkers background
	if (m_shade_type == RGBShade) {
		QPixmap pm(20, 20);
		QPainter pmp(&pm);
		pmp.fillRect(0, 0, 10, 10, Qt::lightGray);
		pmp.fillRect(10, 10, 10, 10, Qt::lightGray);
		pmp.fillRect(0, 10, 10, 10, Qt::darkGray);
		pmp.fillRect(10, 0, 10, 10, Qt::darkGray);
		pmp.end();
		QPalette pal = palette();
		pal.setBrush(backgroundRole(), QBrush(pm));
		setAutoFillBackground(true);
		setPalette(pal);

	} else {
		setAttribute(Qt::WA_NoBackground);

	}

	QPolygonF points;
	points << QPointF(0, sizeHint().height())
		<< QPointF(sizeHint().width(), 0);

	m_hoverPoints = new HoverPoints(this, HoverPoints::CircleShape);
	//     m_hoverPoints->setConnectionType(HoverPoints::LineConnection);
	m_hoverPoints->setPoints(points);
	m_hoverPoints->setPointLock(0, HoverPoints::LockToLeft);
	m_hoverPoints->setPointLock(1, HoverPoints::LockToRight);
	m_hoverPoints->setSortType(HoverPoints::XSort);

	setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Fixed);

	connect(m_hoverPoints, SIGNAL(pointsChanged(QPolygonF)), this, SIGNAL(colorsChanged()));
}


QPolygonF ShadeWidget::points() const
{
	return m_hoverPoints->points();
}


uint ShadeWidget::colorAt(int x)
{
	generateShade();

	QPolygonF pts = m_hoverPoints->points();
	for (int i=1; i < pts.size(); ++i) {
		if (pts.at(i-1).x() <= x && pts.at(i).x() >= x) {
			QLineF l(pts.at(i-1), pts.at(i));
			l.setLength(l.length() * ((x - l.x1()) / l.dx()));
			return m_shade.pixel(qRound(qMin(l.x2(), (qreal(m_shade.width() - 1)))),
				qRound(qMin(l.y2(), qreal(m_shade.height() - 1))));
		}
	}
	return 0;
}


void ShadeWidget::setGradientStops(const QGradientStops &stops)
{
	if (m_shade_type == RGBShade) {
		m_alpha_gradient = QLinearGradient(0, 0, width(), 0);

		for (int i=0; i<stops.size(); ++i) {
			QColor c = stops.at(i).second;
			m_alpha_gradient.setColorAt(stops.at(i).first, QColor(c.red(), c.green(), c.blue()));
		}

		m_shade = QImage();
		generateShade();
		update();
	}
}

void ShadeWidget::setPointsFromGradientStops(const QGradientStops &stops)
{
	QPolygonF points;
	qreal Height = height();

	for (int i=0; i<stops.size(); ++i)
	{
		qreal pos = stops.at(i).first;
		QRgb color = stops.at(i).second.rgba();
		int RGB = 0;
		if (m_shade_type == RedShade)
		{
			RGB = qRed(color);
		}else if (m_shade_type == GreenShade)
		{
			RGB = qGreen(color);
		}else if (m_shade_type == BlueShade)
		{
			RGB = qBlue(color);
		}
		points << QPointF(pos * width(), Height - RGB * Height / 255);
	}

	hoverPoints()->setPoints(points);
	hoverPoints()->setPointLock(0, HoverPoints::LockToLeft);
	hoverPoints()->setPointLock(points.size()-1, HoverPoints::LockToRight);
	update();
}


void ShadeWidget::paintEvent(QPaintEvent *)
{
	generateShade();

	QPainter p(this);
	p.drawImage(0, 0, m_shade);

	p.setPen(QColor(146, 146, 146));
	p.drawRect(0, 0, width() - 1, height() - 1);
}

void ShadeWidget::generateShade()
{
	if (m_shade.isNull() || m_shade.size() != size()) {

		if (m_shade_type == RGBShade) {
			m_shade = QImage(size(), QImage::Format_ARGB32_Premultiplied);
			m_shade.fill(0);

			QPainter p(&m_shade);
			p.fillRect(rect(), m_alpha_gradient);

			/*p.setCompositionMode(QPainter::CompositionMode_DestinationIn);
			QLinearGradient fade(0, 0, 0, height());
			fade.setColorAt(0, QColor(0, 0, 0, 255));
			fade.setColorAt(1, QColor(0, 0, 0, 0));
			p.fillRect(rect(), fade);*/

		} else {
			m_shade = QImage(size(), QImage::Format_RGB32);
			QLinearGradient shade(0, 0, 0, height());
			shade.setColorAt(1, Qt::black);

			if (m_shade_type == RedShade)
				shade.setColorAt(0, Qt::red);
			else if (m_shade_type == GreenShade)
				shade.setColorAt(0, Qt::green);
			else
				shade.setColorAt(0, Qt::blue);

			QPainter p(&m_shade);
			p.fillRect(rect(), shade);
		}
	}
}
