/* Copyright (c) 2010(Hennequin Thibault) Permission is hereby granted, free of charge,
to any person obtaining a copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation the rights to use, copy,
modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. */

#include "mainwindow.h"
#include <QEvent>

/**
 *
 * \param *parent 
 * \return 
 */
MainWindow::MainWindow(QWidget *parent) :
QMainWindow(parent)
{
	qApp->installTranslator(&m_Translator);

	setObjectName(QString::fromUtf8("MainWindow"));
	resize(800,600);

	QWidget* centralWidget = new QWidget(this);
	setCentralWidget(centralWidget);

	m_pMainLayout = new QHBoxLayout(centralWidget);

	m_pToolBox = new QToolBox(centralWidget);
	m_pParametersBox = new QWidget(m_pToolBox);

		m_pParametersLayout = new QFormLayout(m_pParametersBox);
			m_pParametersLayout->setSpacing(2);
			m_pParametersLayout->setContentsMargins(11, 11, 11, 11);
			m_pParametersLayout->setFieldGrowthPolicy(QFormLayout::AllNonFixedFieldsGrow);

			m_pZoomLabel = new QLabel(m_pParametersBox);
			m_pZoomSpin = new QDoubleSpinBox(m_pParametersBox);
			m_pZoomSpin->setObjectName(QString::fromUtf8("SpinZoom"));
			m_pZoomSpin->setDecimals(16);
			m_pZoomSpin->setSingleStep(0.0001);

			m_pPosXLabel = new QLabel(m_pParametersBox);
			m_pPosXSpin = new QDoubleSpinBox(m_pParametersBox);
			m_pPosXSpin->setObjectName(QString::fromUtf8("SpinPosX"));
			m_pPosXSpin->setRange(-99999,99999);
			m_pPosXSpin->setDecimals(5);
			m_pPosXSpin->setSingleStep(0.01);		

			m_pPosYLabel = new QLabel(m_pParametersBox);
			m_pPosYSpin = new QDoubleSpinBox(m_pParametersBox);
			m_pPosYSpin->setObjectName(QString::fromUtf8("SpinPosY"));
			m_pPosYSpin->setRange(-99999,99999);
			m_pPosYSpin->setDecimals(5);
			m_pPosYSpin->setSingleStep(0.0001);

			m_pIterationLabel = new QLabel(m_pParametersBox);
			m_pIterationSpin = new QSpinBox(m_pParametersBox);
			m_pIterationSpin->setObjectName(QString::fromUtf8("SpinPass"));
			m_pIterationSpin->setSingleStep(100);
			m_pIterationSpin->setMaximum(999999);

			m_pFractalLabel = new QLabel(m_pParametersBox);
			m_pFractalComboBox = new QComboBox(m_pParametersBox);
			m_pFractalComboBox->addItem("MandelBrot",Fractal_MandelBrot);
			m_pFractalComboBox->addItem("BuddhaBrot",Fractal_BuddhaBrot);
			m_pFractalComboBox->addItem("Julia",Fractal_Julia);
#if QT_VERSION >= 0x040600
			m_pFractalComboBox->addItem("MandelBrot - OpenGL",Fractal_MandelBrot_OpenGL);
			m_pFractalComboBox->addItem("Julia - OpenGL",Fractal_Julia_OpenGL);
#endif
			m_pFractalComboBox->setCurrentIndex(0);


		m_pBuddhaBrotParameters = new QWidget(m_pParametersBox);
			m_pBuddhaBrotParametersLayout = new QFormLayout(m_pBuddhaBrotParameters);
			m_pBuddhaBrotParametersLayout->setSpacing(6);
			m_pBuddhaBrotParametersLayout->setContentsMargins(11, 11, 11, 11);
			m_pBuddhaBrotParametersLayout->setFieldGrowthPolicy(QFormLayout::AllNonFixedFieldsGrow);

			m_pPrecisionLabel = new QLabel(m_pBuddhaBrotParameters);
			m_pPrecisionSpin = new QDoubleSpinBox(m_pBuddhaBrotParameters);
			m_pPrecisionSpin->setObjectName(QString::fromUtf8("SpinPrecision"));
			m_pPrecisionSpin->setRange(0.00001,1);
			m_pPrecisionSpin->setDecimals(5);
			m_pPrecisionSpin->setSingleStep(0.01);

			m_pMinIterationLabel = new QLabel(m_pBuddhaBrotParameters);
			m_pMinIterationSpin = new QSpinBox(m_pBuddhaBrotParameters);
				m_pMinIterationSpin->setObjectName(QString::fromUtf8("SpinMinIteration"));
				m_pMinIterationSpin->setRange(0,999999);
				m_pMinIterationSpin->setSingleStep(10);

			m_pBuddhaBrotParametersLayout->setWidget(0,QFormLayout::LabelRole,m_pPrecisionLabel);
			m_pBuddhaBrotParametersLayout->setWidget(0,QFormLayout::FieldRole,m_pPrecisionSpin);
			m_pBuddhaBrotParametersLayout->setWidget(1,QFormLayout::LabelRole,m_pMinIterationLabel);
			m_pBuddhaBrotParametersLayout->setWidget(1,QFormLayout::FieldRole,m_pMinIterationSpin);

		m_pJuliaParameters = new QWidget(m_pParametersBox);
			m_pJuliaParameters->setVisible(false);
			m_pJuliaParametersLayout = new QFormLayout(m_pJuliaParameters);
			m_pJuliaParametersLayout->setSpacing(6);
			m_pJuliaParametersLayout->setContentsMargins(11, 11, 11, 11);
			m_pJuliaParametersLayout->setFieldGrowthPolicy(QFormLayout::AllNonFixedFieldsGrow);

			m_pRealLabel = new QLabel(m_pJuliaParameters);
			m_pRealSpin = new QDoubleSpinBox(m_pJuliaParameters);
			m_pRealSpin->setObjectName(QString::fromUtf8("SpinReal"));
			m_pRealSpin->setRange(-99999,99999);
			m_pRealSpin->setDecimals(5);
			m_pRealSpin->setSingleStep(0.01);

			m_pImagLabel = new QLabel(m_pJuliaParameters);
			m_pImagSpin = new QDoubleSpinBox(m_pJuliaParameters);
			m_pImagSpin->setObjectName(QString::fromUtf8("SpinImaginary"));
			m_pImagSpin->setRange(-99999,99999);
			m_pImagSpin->setDecimals(5);
			m_pImagSpin->setSingleStep(0.01);

			m_pJuliaParametersLayout->setWidget(0,QFormLayout::LabelRole,m_pRealLabel);
			m_pJuliaParametersLayout->setWidget(0,QFormLayout::FieldRole,m_pRealSpin);
			m_pJuliaParametersLayout->setWidget(1,QFormLayout::LabelRole,m_pImagLabel);
			m_pJuliaParametersLayout->setWidget(1,QFormLayout::FieldRole,m_pImagSpin);

		m_pParametersLayout->setWidget(0, QFormLayout::LabelRole, m_pZoomLabel);
		m_pParametersLayout->setWidget(0, QFormLayout::FieldRole, m_pZoomSpin);
		m_pParametersLayout->setWidget(1, QFormLayout::LabelRole, m_pPosXLabel);
		m_pParametersLayout->setWidget(1, QFormLayout::FieldRole, m_pPosXSpin);
		m_pParametersLayout->setWidget(2, QFormLayout::LabelRole, m_pPosYLabel);
		m_pParametersLayout->setWidget(2, QFormLayout::FieldRole, m_pPosYSpin);
		m_pParametersLayout->setWidget(3, QFormLayout::LabelRole, m_pIterationLabel);
		m_pParametersLayout->setWidget(3, QFormLayout::FieldRole, m_pIterationSpin);
		m_pParametersLayout->setWidget(4, QFormLayout::LabelRole, m_pFractalLabel);
		m_pParametersLayout->setWidget(4, QFormLayout::FieldRole, m_pFractalComboBox);

		m_pParametersLayout->setWidget(5, QFormLayout::SpanningRole, m_pBuddhaBrotParameters);
		m_pParametersLayout->setWidget(6, QFormLayout::SpanningRole, m_pJuliaParameters);

	m_pToolBox->addItem(m_pParametersBox, tr("Parameters"));

	m_pColorsBox = new QWidget(m_pToolBox);
		m_pColorsLayout = new QFormLayout(m_pColorsBox);
			m_pRedShade = new ShadeWidget(ShadeWidget::RedShade,m_pColorsBox);
			m_pGreenShade = new ShadeWidget(ShadeWidget::GreenShade,m_pColorsBox);
			m_pBlueShade = new ShadeWidget(ShadeWidget::BlueShade,m_pColorsBox);
			m_pRGBShade = new ShadeWidget(ShadeWidget::RGBShade,m_pColorsBox);
			m_pRGBShade->hoverPoints()->setDisabled(true);

			m_pDilatationLabel = new QLabel(m_pColorsBox);
			m_pDilatationSpin = new QDoubleSpinBox(m_pColorsBox);
				m_pDilatationSpin->setRange(0.0000001,99999999);

		m_pColorsLayout->setWidget(0, QFormLayout::SpanningRole, m_pRedShade);
		m_pColorsLayout->setWidget(1, QFormLayout::SpanningRole, m_pGreenShade);
		m_pColorsLayout->setWidget(2, QFormLayout::SpanningRole, m_pBlueShade);
		m_pColorsLayout->setWidget(3, QFormLayout::SpanningRole, m_pRGBShade);
		m_pColorsLayout->setWidget(4, QFormLayout::LabelRole, m_pDilatationLabel);
		m_pColorsLayout->setWidget(4, QFormLayout::FieldRole, m_pDilatationSpin);

	m_pToolBox->addItem(m_pColorsBox, tr("Colors"));

	m_pRenderBox = new QWidget(m_pToolBox);
		m_pRendersLayout = new QFormLayout(m_pRenderBox);
			m_pRendersLayout->setSpacing(6);
			m_pRendersLayout->setContentsMargins(11, 11, 11, 11);
			m_pRendersLayout->setFieldGrowthPolicy(QFormLayout::AllNonFixedFieldsGrow);

			m_pThreadCountLabel = new QLabel(m_pRenderBox);
			m_pThreadCountSpin = new QSpinBox(m_pRenderBox);
				m_pThreadCountSpin->setRange(1,512);

			m_pWidthLabel = new QLabel(m_pRenderBox);
			m_pWidthSpin = new QSpinBox(m_pRenderBox);
				m_pWidthSpin->setRange(1,999999);

			m_pHeightLabel = new QLabel(m_pRenderBox);
			m_pHeightSpin = new QSpinBox(m_pRenderBox);
				m_pHeightSpin->setRange(1,999999);

			m_pSmoothLabel = new QLabel(m_pRenderBox);
			m_pSmoothCheckBox = new QCheckBox(m_pRenderBox);

			m_pAnimatedLabel = new QLabel(m_pRenderBox);
			m_pAnimatedCheckBox = new QCheckBox(m_pRenderBox);

		m_pRendersLayout->setWidget(0, QFormLayout::LabelRole, m_pThreadCountLabel);
		m_pRendersLayout->setWidget(0, QFormLayout::FieldRole, m_pThreadCountSpin);
		m_pRendersLayout->setWidget(1, QFormLayout::LabelRole, m_pWidthLabel);
		m_pRendersLayout->setWidget(1, QFormLayout::FieldRole, m_pWidthSpin);
		m_pRendersLayout->setWidget(2, QFormLayout::LabelRole, m_pHeightLabel);
		m_pRendersLayout->setWidget(2, QFormLayout::FieldRole, m_pHeightSpin);
		m_pRendersLayout->setWidget(3, QFormLayout::LabelRole, m_pSmoothLabel);
		m_pRendersLayout->setWidget(3, QFormLayout::FieldRole, m_pSmoothCheckBox);
		m_pRendersLayout->setWidget(4, QFormLayout::LabelRole, m_pAnimatedLabel);
		m_pRendersLayout->setWidget(4, QFormLayout::FieldRole, m_pAnimatedCheckBox);
	m_pToolBox->addItem(m_pRenderBox, tr("Render"));

	m_pDisplay = new BrotDisplay();
		m_pDisplay->setObjectName(QString::fromUtf8("BrotDisplay"));

	m_pDisplayScroll = new QScrollArea(centralWidget);
		m_pDisplayScroll->setAlignment(Qt::AlignCenter);
		m_pDisplayScroll->setWidget(m_pDisplay);

	m_pMainLayout->addWidget(m_pToolBox,0,Qt::AlignLeft);
	m_pMainLayout->addWidget(m_pDisplayScroll,1);
	setLayout(m_pMainLayout);

	m_pMenuBar = new QMenuBar(this);
	m_pMenuBar->setObjectName(QString::fromUtf8("menuBar"));
	m_pMenuFile = new QMenu(tr("File"),m_pMenuBar);
		m_pActionSave = new QAction(tr("SaveImage"),m_pMenuFile);
		m_pActionExit = new QAction(tr("Exit"),m_pMenuFile);
	m_pMenuFile->addAction(m_pActionSave);
	m_pMenuFile->addSeparator();
	m_pMenuFile->addAction(m_pActionExit);
	m_pMenuOptions = new QMenu(tr("Options"),m_pMenuBar);
		m_pMenuLanguage =  new QMenu(tr("Language"),m_pMenuOptions);
			m_pActionLanguageFrench = new QAction("Franais",m_pMenuLanguage);
				m_pActionLanguageFrench->setCheckable(true);
			m_pActionLanguageEnglish = new QAction("English",m_pMenuLanguage);
				m_pActionLanguageEnglish->setCheckable(true);

		m_pMenuLanguage->addAction(m_pActionLanguageFrench);
		m_pMenuLanguage->addAction(m_pActionLanguageEnglish);
	m_pMenuOptions->addMenu(m_pMenuLanguage);
	m_pActionAbout = new QAction(tr("About"),m_pMenuBar);

	m_pMenuBar->addMenu(m_pMenuFile);
	m_pMenuBar->addMenu(m_pMenuOptions);
	setMenuBar(m_pMenuBar);

	setStatusBar(new QStatusBar(this));
	
	m_pRenderButton = new QPushButton(statusBar());
		m_pRenderButton->resize(75,30);
	m_pProgressBar = new QProgressBar(statusBar());
	    m_pProgressBar->setRange(0,100);
	    m_pProgressBar->resize(250,30);
		m_pProgressBar->setTextVisible(false);
	m_pProgressLabel = new QLabel(statusBar());
		m_pProgressLabel->resize(75,30);
	m_pCancelButton = new QPushButton(statusBar());
		m_pCancelButton->resize(75,30);
	statusBar()->addWidget(m_pRenderButton,0);
	statusBar()->addWidget(m_pProgressBar,0);
	statusBar()->addWidget(m_pProgressLabel,0);	
	statusBar()->addWidget(m_pCancelButton,0);

	m_Translator.load("quickbrot_fr");
	m_pActionLanguageFrench->setChecked(true);
	m_pActionLanguageEnglish->setChecked(false);	

	TranslateUI();

	connect(m_pZoomSpin, SIGNAL(valueChanged(double)), m_pDisplay, SLOT(SetZoom(double)));
	connect(m_pPosXSpin, SIGNAL(valueChanged(double)), m_pDisplay, SLOT(SetPosX(double)));
	connect(m_pPosYSpin, SIGNAL(valueChanged(double)), m_pDisplay, SLOT(SetPosY(double)));
	connect(m_pIterationSpin, SIGNAL(valueChanged(int)), m_pDisplay, SLOT(SetIteration(int)));
	connect(m_pFractalComboBox, SIGNAL(activated(int)), this, SLOT(SetFractal(int)));

	connect(m_pPrecisionSpin, SIGNAL(valueChanged(double)), m_pDisplay, SLOT(SetPrecision(double)));
	connect(m_pRealSpin, SIGNAL(valueChanged(double)), m_pDisplay, SLOT(SetReal(double)));
	connect(m_pImagSpin, SIGNAL(valueChanged(double)), m_pDisplay, SLOT(SetImaginary(double)));

	connect(m_pRedShade, SIGNAL(colorsChanged()), this, SLOT(colorsChanged()));
	connect(m_pGreenShade, SIGNAL(colorsChanged()), this, SLOT(colorsChanged()));
	connect(m_pBlueShade, SIGNAL(colorsChanged()), this, SLOT(colorsChanged()));
	connect(m_pDilatationSpin, SIGNAL(valueChanged(double)), m_pDisplay, SLOT(SetDilatation(double)));

	connect(m_pThreadCountSpin, SIGNAL(valueChanged(int)), m_pDisplay, SLOT(SetThreadCount(int)));
	connect(m_pWidthSpin, SIGNAL(valueChanged(int)), this, SLOT(SetRenderWidth(int)));
	connect(m_pHeightSpin, SIGNAL(valueChanged(int)), this, SLOT(SetRenderHeight(int)));
	connect(m_pSmoothCheckBox, SIGNAL(stateChanged(int)), this, SLOT(SetSmoothColors(int)));
	connect(m_pAnimatedCheckBox, SIGNAL(stateChanged(int)), this, SLOT(SetAnimatedRender(int)));

	connect(m_pDisplay, SIGNAL(ZoomChanged(double)), m_pZoomSpin, SLOT(setValue(double)));
	connect(m_pDisplay, SIGNAL(PosXChanged(double)), m_pPosXSpin, SLOT(setValue(double)));
	connect(m_pDisplay, SIGNAL(PosYChanged(double)), m_pPosYSpin, SLOT(setValue(double)));
	connect(m_pDisplay, SIGNAL(IterationChanged(int)), m_pIterationSpin, SLOT(setValue(int)));
	connect(m_pDisplay, SIGNAL(ThreadCountChanged(int)), m_pThreadCountSpin, SLOT(setValue(int)));

	connect(m_pDisplay, SIGNAL(PrecisionChanged(double)), m_pPrecisionSpin, SLOT(setValue(double)));
	connect(m_pDisplay, SIGNAL(RealChanged(double)), m_pRealSpin, SLOT(setValue(double)));
	connect(m_pDisplay, SIGNAL(ImaginaryChanged(double)), m_pImagSpin, SLOT(setValue(double)));

	connect(m_pDisplay, SIGNAL(DilatationChanged(double)), m_pDilatationSpin, SLOT(setValue(double)));

	connect(m_pDisplay, SIGNAL(Progress(float)), this, SLOT(OnProgress(float)));	

	connect(m_pRenderButton, SIGNAL(clicked(bool)), m_pDisplay, SLOT(RedrawFractal()));
	connect(m_pCancelButton, SIGNAL(clicked(bool)), this, SLOT(Cancel(bool)));

	connect(m_pActionSave, SIGNAL(triggered()), this, SLOT(SaveImage()));
	connect(m_pActionExit, SIGNAL(triggered()), this, SLOT(close()));

	connect(m_pActionLanguageFrench, SIGNAL(triggered()), this, SLOT(LanguageFrench()));
	connect(m_pActionLanguageEnglish, SIGNAL(triggered()), this, SLOT(LanguageEnglish()));
	
	QMetaObject::connectSlotsByName(this);

	if (m_pDisplay)
	{
		m_pDisplay->SetZoom(0.005);
		m_pDisplay->SetPosX(0.0);
		m_pDisplay->SetPosY(0.0);
		m_pDisplay->SetIteration(1000);
		m_pDisplay->SetPrecision(0.1);
		m_pDisplay->SetReal(-0.4);
		m_pDisplay->SetImaginary(0.6);
		m_pDisplay->SetThreadCount(QThread::idealThreadCount());
		m_pDisplay->SetDilatation(20.0);
		SetRenderWidth(640);
		SetRenderHeight(480);
	}
	SetFractal(Fractal_MandelBrot);
	SetSmoothColors(2);
	SetAnimatedRender(2);

   /* m_Gradient << QGradientStop(0.00, QColor::fromRgba(0));
    m_Gradient << QGradientStop(0.04, QColor::fromRgba(0xff131360));
    m_Gradient << QGradientStop(0.08, QColor::fromRgba(0xff202ccc));
    m_Gradient << QGradientStop(0.42, QColor::fromRgba(0xff93d3f9));
    m_Gradient << QGradientStop(0.51, QColor::fromRgba(0xffb3e6ff));
    m_Gradient << QGradientStop(0.73, QColor::fromRgba(0xffffffec));
    m_Gradient << QGradientStop(0.92, QColor::fromRgba(0xff5353d9));
    m_Gradient << QGradientStop(0.96, QColor::fromRgba(0xff262666));
    m_Gradient << QGradientStop(1.00, QColor::fromRgba(0));

    m_Gradient << QGradientStop(0.00, QColor::fromRgba(0));
    m_Gradient << QGradientStop(0.10, QColor::fromRgba(0xffe0cc73));
    m_Gradient << QGradientStop(0.17, QColor::fromRgba(0xffc6a006));
    m_Gradient << QGradientStop(0.46, QColor::fromRgba(0xff600659));
    m_Gradient << QGradientStop(0.72, QColor::fromRgba(0xff0680ac));
    m_Gradient << QGradientStop(0.92, QColor::fromRgba(0xffb9d9e6));
    m_Gradient << QGradientStop(1.00, QColor::fromRgba(0));*/

	/*m_Gradient << QGradientStop(0.0, QColor(255,0,0));
    m_Gradient << QGradientStop(0.5, QColor(0,255,0));
    m_Gradient << QGradientStop(1.0, QColor(255,0,0));*/

	/*m_Gradient << QGradientStop(0.0, QColor(0,0,128));
    m_Gradient << QGradientStop(0.25, QColor(32,32,192));
	m_Gradient << QGradientStop(0.5, QColor(0,255,0));
	m_Gradient << QGradientStop(0.75, QColor(0,255,0));
    m_Gradient << QGradientStop(1.0, QColor(0,0,128));*/

	m_Gradient << QGradientStop(0.0, QColor(255,255,0));
    m_Gradient << QGradientStop(0.25, QColor(0,0,0));
	m_Gradient << QGradientStop(0.4, QColor(0,0,0));
	m_Gradient << QGradientStop(0.6, QColor(32,32,128));
	m_Gradient << QGradientStop(0.8, QColor(192,0,0));
    m_Gradient << QGradientStop(1.0, QColor(255,255,0));

	m_pRedShade->setPointsFromGradientStops(m_Gradient);	
	m_pGreenShade->setPointsFromGradientStops(m_Gradient);
	m_pBlueShade->setPointsFromGradientStops(m_Gradient);

	m_pRGBShade->setGradientStops(m_Gradient);

	m_pDisplay->SetGradient(m_Gradient);

	m_pProgressBar->setVisible(false);
	m_pProgressLabel->setVisible(false);
	m_pCancelButton->setVisible(false);
	m_pRenderButton->setVisible(true);
}

/**
 *
 * \return 
 */
MainWindow::~MainWindow()
{

}

/** Traduit l'intgralit de l'interface
 *
 */
void MainWindow::TranslateUI()
{
	m_pToolBox->setItemText(0,tr("Parameters"));
	m_pToolBox->setItemText(1,tr("Colors"));
	m_pToolBox->setItemText(2,tr("Render"));

	m_pZoomLabel->setText(tr("Zoom"));
	m_pPosXLabel->setText(tr("PosX"));
	m_pPosYLabel->setText(tr("PosY"));
	m_pIterationLabel->setText(tr("Iteration"));
	m_pFractalLabel->setText(tr("Fractal"));

	m_pPrecisionLabel->setText(tr("Precision"));
	m_pMinIterationLabel->setText(tr("MinIteration"));

	m_pRealLabel->setText(tr("Real"));
	m_pImagLabel->setText(tr("Imaginary"));

	m_pDilatationLabel->setText(tr("Dilatation"));

	m_pThreadCountLabel->setText(tr("ThreadCount"));
	m_pWidthLabel->setText(tr("Width"));
	m_pHeightLabel->setText(tr("Height"));
	m_pSmoothLabel->setText(tr("SmoothColors"));
	m_pAnimatedLabel->setText(tr("AnimatedRender"));

	m_pMenuFile->setTitle(tr("File"));
		m_pActionSave->setText(tr("SaveImage"));
		m_pActionExit->setText(tr("Exit"));
	m_pMenuOptions->setTitle(tr("Options"));
		m_pMenuLanguage->setTitle(tr("Language"));
	m_pActionAbout->setText(tr("About"));

	m_pRenderButton->setText(tr("StartRender"));
	m_pCancelButton->setText(tr("Cancel"));

	m_pParametersBox->update();
}

/** Fonction d'vnements
 *
 * \param *e 
 */
void MainWindow::changeEvent(QEvent *e)
{
	QMainWindow::changeEvent(e);
	switch (e->type()) {
	case QEvent::LanguageChange:
		TranslateUI();
		break;
	default:

		break;
	}
}

/** Fonction appel lors de progrs du rendu
 *
 * \param value 
 */
void MainWindow::OnProgress( float value )
{
	if (m_pProgressBar)
	{
		m_pProgressBar->setValue(value*100);
		QString str;
		str.sprintf("%.2f %%",value*100);
		m_pProgressLabel->setText(str);
		if (value>=1.0)
		{
			if (m_pProgressBar->isVisible()==true)
			{
				m_pProgressBar->setVisible(false);
				m_pProgressLabel->setVisible(false);
				m_pCancelButton->setVisible(false);
				m_pRenderButton->setVisible(true);
			}
		}else{
			if (m_pProgressBar->isVisible()==false)
			{
				m_pProgressBar->setVisible(true);
				m_pProgressLabel->setVisible(true);
				m_pCancelButton->setVisible(true);
				m_pRenderButton->setVisible(false);
			}
		}
	}
}

/** Dfinie le type de fractale et affiche les options correspondant
 *
 * \param fractal 
 */
void MainWindow::SetFractal(int fractal)
{
	m_pJuliaParameters->setVisible(false);
	m_pBuddhaBrotParameters->setVisible(false);

	switch (fractal)
	{
	case Fractal_BuddhaBrot:
		m_pBuddhaBrotParameters->setVisible(true);
		break;
	case Fractal_Julia:
	case Fractal_Julia_OpenGL:
		m_pJuliaParameters->setVisible(true);
		break;
	};

	if (m_pDisplay)
	{
		m_pDisplay->SetFractal((Fractal)fractal);
	}
}

/** Dfinie la largeur de la fenetre de rendu
 *
 * \param width 
 */
void MainWindow::SetRenderWidth(int width)
{
	if (m_pDisplay && m_pDisplay->width()!=width)
	{
		m_pWidthSpin->setValue(width);
		m_pDisplay->resize(width, m_pDisplay->height());
	}
}

/** Dfinie la hauteur de la fenetre de rendu
 *
 * \param height 
 */
void MainWindow::SetRenderHeight(int height)
{
	if (m_pDisplay && m_pDisplay->height()!=height)
	{
		m_pHeightSpin->setValue(height);
		m_pDisplay->resize(m_pDisplay->width(), height);
	}
}

/** Dfinie si la fractale doit etre affich durant le rendu
 *
 * \param animated 
 */
void MainWindow::SetAnimatedRender(int animated)
{
	if (m_pDisplay)
	{
		m_pDisplay->SetAnimatedRender(animated==0?false:true);
		if (m_pAnimatedCheckBox->checkState()!=(Qt::CheckState)animated)
		{
			m_pAnimatedCheckBox->setCheckState((Qt::CheckState)animated);
		}
	}
}

/** Dfinie si on utilise le Normalized Iteration Count Algorithm pour lisser les couleurs
 *
 * \param smooth 
 */
void MainWindow::SetSmoothColors(int smooth)
{
	if (m_pDisplay)
	{
		m_pDisplay->SetSmoothColors(smooth==0?false:true);
		if (m_pSmoothCheckBox->checkState()!=(Qt::CheckState)smooth)
		{
			m_pSmoothCheckBox->setCheckState((Qt::CheckState)smooth);
		}
	}
}

/** Annule le rendu en cours
 *
 * \param bool 
 */
void MainWindow::Cancel(bool)
{
	if (m_pDisplay)
	{
		m_pDisplay->Cancel();
	}
}

/** Sauvegarde le rendu dans une image png
 *
 * \param void 
 */
void MainWindow::SaveImage(void)
{
	if (m_pDisplay)
	{
		QFileDialog dialog(this);
		dialog.setFileMode(QFileDialog::AnyFile);
		dialog.setAcceptMode(QFileDialog::AcceptSave);
		dialog.setNameFilter(tr("Images (*.png)"));

		if (dialog.exec())
		{			
			QString filename = dialog.selectedFiles()[0];
			if (filename.right(4).toLower()!=".png")
			{
				filename+=".png";
			}
			if (m_pDisplay->Save(filename))
			{
				statusBar()->showMessage(tr("ImageSaved"), 5000);
			}else{
				statusBar()->showMessage(tr("ImageSavedError"), 5000);
			}
		}
	}
}

inline static bool x_less_than(const QPointF &p1, const QPointF &p2)
{
    return p1.x() < p2.x();
}


/** Rgnre le gradient en fonction en fonction des widget Red, Green et Blue
 *
 * \param void 
 */
void MainWindow::colorsChanged(void)
{
	qreal w = m_pRGBShade->width();
	m_Gradient.clear();
    QPolygonF points;
	
    points += m_pRedShade->points();
    points += m_pGreenShade->points();
    points += m_pBlueShade->points();

    qSort(points.begin(), points.end(), x_less_than);

    for (int i=0; i<points.size(); ++i) {
        qreal x = int(points.at(i).x());
        if (i < points.size() - 1 && x == points.at(i+1).x())
            continue;
        QColor color((0x00ff0000 & m_pRedShade->colorAt(int(x))) >> 16,
                     (0x0000ff00 & m_pGreenShade->colorAt(int(x))) >> 8,
                     (0x000000ff & m_pBlueShade->colorAt(int(x))));

        if (x / w > 1)
            return;

        m_Gradient << QGradientStop(x / w, color);
    }

    m_pRGBShade->setGradientStops(m_Gradient);

	m_pDisplay->SetGradient(m_Gradient);
}

/** Slctionne une langue pour l'interface
 *
 * \param lang 
 */
void MainWindow::LanguageChanged( int lang )
{
	m_pActionLanguageFrench->setChecked(false);
	m_pActionLanguageEnglish->setChecked(false);

	if (lang==0)
	{
		m_Translator.load("quickbrot_fr");
		m_pActionLanguageFrench->setChecked(true);
		
	}else if (lang==1)
	{
		m_Translator.load("quickbrot_en");
		m_pActionLanguageEnglish->setChecked(true);
		
	}
	TranslateUI();
}

/** Slctionne la langue franaise
 *
 */
void MainWindow::LanguageFrench()
{
	LanguageChanged(0);
}

/** Slctionne la langue anglaise
 *
 */
void MainWindow::LanguageEnglish()
{
	LanguageChanged(1);
}
