/* Copyright (c) 2010(Hennequin Thibault) Permission is hereby granted, free of charge,
to any person obtaining a copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation the rights to use, copy,
modify, merge, publish, distribute, sublicense, and/or sell copies of the Software,
and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. */

#include "renderthread.h"
#include <QColor>
#include <math.h>
#include <map>

#define log2(x) (log(x)/log((double)2.0))

QVector<unsigned long> RenderThread::m_Map;
int RenderThread::m_iThreadFinish;
unsigned long RenderThread::m_iMax;

/**
 *
 * \param *parent 
 * \param id 
 * \return 
 */
RenderThread::RenderThread(QObject *parent, int id)
: QThread(parent)
{
	m_iId = id;
	m_bRestart = false;
	m_bStop = false;
	m_bFinish = false;

	m_iThreadCount = 0;
	m_iProgress = 0;   
	m_iW = 0;
	m_iH = 0;
	m_fCenterX = 0;
	m_fCenterY = 0;
	m_fZoom = 1;
	m_iIteration = 0;
	m_fPrecision = 0;
	m_iMinIteration = 0;
	m_fReal = 0;
	m_fImaginary = 0;
	m_pImage = NULL;
	m_pImageMutex = NULL;
	m_iFractal = Fractal_MandelBrot;
}

/**
 *
 * \return 
 */
RenderThread::~RenderThread()
{
	Finish();
}

/** Initialise le rendu pour le thread
 *
 * \param w : largeur de la fenetre de rendu
 * \param h : hauteur de la fenetre de rendu
 * \param threadCount : nombre de threads utilis pour le rendu
 * \param centerx : coordone en x du centre de la fractale
 * \param centery : coordone en y du centre de la fractale
 * \param precision : prcision utilis pour le rendu du Buddhabrot
 * \param minIteration : nombre minimum d'iteration pour le rendu du Buddhabrot
 * \param real : valeur rel du complex utilis pour la fractale Julia
 * \param imaginary : valeur imaginaire du complex utilis pour la fractale Julia
 * \param zoom : zoom utilis sur la fractale
 * \param iteration : nombre d'itration maximum pour le rendu de la fractale
 * \param fractal : type de fractale  calcul
 * \param gradient : gradient utilis pour la slction des couleurs (non utilis)
 * \param *gradientImage : image contenant une reprsentation du gradient pour la slction des couleurs
 * \param smooth : utilisation du Normalized Iteration Count Algorithm pour lisser les couleurs
 * \param *image : image du rendu finale
 * \param *imageMutex : mutex de l'image du rendu finale
 */
void RenderThread::SetRender(int w, int h,
							 int threadCount,
							 double centerx, double centery,
							 double precision, int minIteration,
							 double real, double imaginary,
							 double zoom, int iteration, Fractal fractal,
							 const QGradientStops& gradient, QImage *gradientImage, bool smooth, float dilatation,
							 QImage *image, QMutex *imageMutex)
{
	m_Mutex.lock();
	m_iThreadCount = threadCount;
	m_iProgress = 0;   
	m_iW = w;
	m_iH = h;
	m_fCenterX = centerx;
	m_fCenterY = centery;
	m_fZoom = zoom;
	m_iIteration = iteration;
	m_fPrecision = precision;
	m_iMinIteration = minIteration;
	m_fReal = real;
	m_fImaginary = imaginary;
	m_pImage = image;
	m_pImageMutex = imageMutex;
	m_iFractal = fractal;
	m_Gradient = gradient;
	m_pGradientImage = gradientImage;
	m_bSmoothColors = smooth;
	m_fDilatation = dilatation;

	if (m_iId==0)
	{
		m_Map.resize(m_iW*m_iH);
		for (int i=0;i<m_iW*m_iH;i++)
		{
			m_Map[i] = 0;
		}
		m_iThreadFinish = 0;
		m_iMax = 0;
	}


	if (!isRunning())
	{
		start(LowPriority);
	}else{
		m_bRestart = true;
		m_Condition.wakeOne();
	}
	m_Mutex.unlock();
}

/** Envoie le signal d'arret et d'attente au thread
 *
 */
void RenderThread::Stop()
{
	QMutexLocker locker(&m_Mutex);
	m_bStop = true;
}

/** Envoie du signale de fin du thread
 *
 */
void RenderThread::Finish()
{
	m_Mutex.lock();
	m_bFinish = true;
	m_Condition.wakeOne();
	m_Mutex.unlock();
}

/** Fonction pricipale du thread
 *
 */
void RenderThread::run()
{
	while(!m_bFinish)
	{
		switch (m_iFractal)
		{
		case Fractal_MandelBrot:
			RenderMandelBrot();
			break;
		case Fractal_BuddhaBrot:
			RenderBuddhabrot();
			break;
		case Fractal_Julia:
			RenderJulia();
			break;
		default:
			break;
		};

		emit Finish(m_iId);

		if (!m_bRestart && !m_bFinish)
		{
			m_Condition.wait(&m_Mutex);
		}
		m_bRestart = false;
		m_bStop = false;
		m_Mutex.unlock();
	}
	emit Finish(m_iId);
	exit(0);
}

/** Fonction de rendu d'un MandelBrot
 *
 */
void RenderThread::RenderMandelBrot()
{
	m_Mutex.lock();
	double zoom = m_fZoom;
	double cx = m_fCenterX;
	double cy = m_fCenterY;
	int width=m_iW;
	int height=m_iH;
	int max_iter=m_iIteration;
	bool smoothColors=m_bSmoothColors;
	float dilatation=m_fDilatation;
	m_Mutex.unlock();

	QTime LastEventTime;
	LastEventTime.start();
	double x;
	double y;
	int iter;
	double xtemp;

	float ratio = (float)width/height;

	double newWidth = ratio*512.0;
	double newHeight = 512.0;
	double x0b;
	double y0b;

	QVector<uint> line;
	line.resize(width);

	for (int y0=m_iId;y0<height && !m_bRestart && !m_bStop;y0+=m_iThreadCount)
	{
		for (int x0=0;x0<width && !m_bRestart && !m_bStop;x0++)
		{
			if (m_bFinish)
			{
				return;
			}

			x0b=ratio*512.0*x0/width;
			y0b=512.0*y0/height;

			x = 0;
			y = 0;
			iter = 0;
			while (x*x + y*y <= (2*2) && iter < max_iter && !m_bRestart && !m_bStop && !m_bFinish)
			{
				xtemp = x*x - y*y + (x0b-newWidth/2.0)*zoom+cx;
				y = 2*x*y + (y0b-newHeight/2.0)*zoom+cy;
				x = xtemp;
				iter++;
			}

			if ( iter == max_iter )
			{
				line[x0] = 0;
			}else{
				/* Normalized Iteration Count Algorithm */
				/* Permet de lisser les couleurs */
				float val = iter;
				if (smoothColors==true)
				{
					val = val-log2(log2(sqrt(x*x+y*y)));
				}
				if (m_pGradientImage)
				{
					line[x0] = m_pGradientImage->pixel((sin(val/dilatation)+1.0)*0.5*m_pGradientImage->width(),0);
				}else{		
					val = (sin(val/dilatation)+1.0)*0.5*255;
					line[x0] = qRgb(val,val,val);
				}
			}

			if (LastEventTime.elapsed()>=50)			
			{
				LastEventTime.restart();
				float progress = (y0*width+x0)/((float)width*height);
				emit Progress(progress,m_iId);
			}
		}
		if (m_pImageMutex)
		{
			m_pImageMutex->lock();
			if (!m_bRestart && !m_bStop && !m_bFinish)
			{
				for (int j=0;j<width;j++)
				{
					m_pImage->setPixel(j,y0,line[j]);
				}
			}
			m_pImageMutex->unlock();
		}
	}
}

/** Fonction de rendu d'un Buddhabrot
 *
 */
void RenderThread::RenderBuddhabrot()
{
	m_Mutex.lock();
	double zoom = m_fZoom;
	double cx = m_fCenterX;
	double cy = m_fCenterY;
	int width = m_iW;
	int height = m_iH;
	int max_iter = m_iIteration;
	double precision = m_fPrecision;
	int minIteration = m_iMinIteration;
	m_Mutex.unlock();

	QTime LastEventTime;
	LastEventTime.start();

	double x;
	double y;
	int iter;
	double xtemp;

	std::map<unsigned long, unsigned long> map;

	float ratio = (float)width/height;

	double newWidth = ratio*512.0;
	double newHeight = 512.0;
	double x0b;
	double y0b;

	for (double y0=m_iId*precision;y0<height && !m_bRestart && !m_bStop;y0+=m_iThreadCount*precision)
	{
		map.clear();
		for (double x0=0;x0<width && !m_bRestart && !m_bStop;x0+=precision)
		{
			if (m_bFinish)
			{
				return;
			}
			x = 0;
			y = 0;
			x0b=ratio*512.0*x0/width;
			y0b=512.0*y0/height;
			iter = 0;
			while (x*x + y*y <= (2*2) && iter < max_iter && !m_bRestart && !m_bStop && !m_bFinish)
			{
				xtemp = x*x - y*y + (x0b-newWidth/2.0)*zoom+cx;
				y = 2*x*y + (y0b-newHeight/2.0)*zoom+cy;
				x = xtemp;

				iter++;
			}
			if (iter<max_iter)
			{
				x = 0;
				y = 0;
				iter = 0;
				while (x*x + y*y <= (2*2) && iter < max_iter && !m_bRestart && !m_bStop && !m_bFinish)
				{
					xtemp = x*x - y*y + (x0b-newWidth/2.0)*zoom+cx;
					y = 2*x*y + (y0b-newHeight/2.0)*zoom+cy;
					x = xtemp;

					int xpix, ypix;
					xpix = width*(x-cx+newWidth*zoom*0.5)/zoom/(ratio*512.0);
					ypix = height*(y-cy+newHeight*zoom*0.5)/zoom/512.0;

					if (xpix>=0 && xpix<width && ypix>=0 && ypix<height && iter>minIteration)
					{
						if (map.find(xpix+ypix*width)==map.end())
						{
							map[xpix+ypix*width]=1;
						}else{
							map[xpix+ypix*width]+=1;
						}
					}

					iter++;
				}
			}

			if (LastEventTime.elapsed()>=50)			
			{
				LastEventTime.restart();
				float progress = (y0*width+x0)/((float)width*height);
				emit Progress(progress,m_iId);
			}
		}

		std::map<unsigned long, unsigned long>::iterator ite;

		if (m_pImageMutex)
		{
			m_pImageMutex->lock();
		}
		m_Mutex.lock();
		for (ite = map.begin();ite!=map.end();ite++)
		{
			if (m_Map[ite->first]+ite->second > m_iMax)
			{
				m_iMax = m_Map[ite->first]+ite->second;
			}
			m_Map[ite->first]+=ite->second;
			if (m_pImage)
			{
				if (m_pGradientImage)
				{
					m_pImage->setPixel(ite->first%width,floor((float)ite->first/width), 
						m_pGradientImage->pixel(m_Map[ite->first]/(float)m_iMax*m_pGradientImage->width(),0));
				}else{
					m_pImage->setPixel( ite->first%width,floor((float)ite->first/width),qRgb(255* m_Map[ite->first]/(float)m_iMax,0,0));
				}
			}
			ite->second=0;
		}
		m_Mutex.unlock();
		if (m_pImageMutex)
		{
			m_pImageMutex->unlock();
		}
		m_iThreadFinish++;
	}

	/* Seul le premier thread crit la valeur finale dans l'image */
	if (m_iId==0)
	{
		m_Mutex.lock();
		while (m_iThreadFinish<m_iThreadCount && !m_bRestart && !m_bStop && !m_bFinish)
		{
			m_Mutex.unlock();
		}

		if (!m_bRestart && !m_bStop && !m_bFinish)
		{
			unsigned long max=1;

			for (int y0=0;y0<height && !m_bRestart && !m_bStop && !m_bFinish;y0++)
			{
				for (int x0=0;x0<width && !m_bRestart && !m_bStop && !m_bFinish;x0++)
				{
					if (m_Map[x0+y0*width]>max)
					{
						max = m_Map[x0+y0*width];
					}
				}
			}

			for (int y0=0;y0<height && !m_bRestart && !m_bStop && !m_bFinish;y0++)
			{
				if (m_pImageMutex)
				{
					m_pImageMutex->lock();
					for (int x0=0;x0<width && !m_bRestart && !m_bStop && !m_bFinish;x0++)
					{
						if (m_pGradientImage)
						{
							m_pImage->setPixel(x0,y0, 
								m_pGradientImage->pixel(m_Map[x0+y0*width]/(float)max*m_pGradientImage->width(),0));
						}else{
							m_pImage->setPixel(x0,y0,qRgb(255*m_Map[x0+y0*width]/max,0,0));
						}
					}
					m_pImageMutex->unlock();
				}
			}
		}
	}
}

/** Fonction de rendu d'un Julia
 *
 */
void RenderThread::RenderJulia()
{
	m_Mutex.lock();
	double zoom = m_fZoom;
	double cx = m_fCenterX;
	double cy = m_fCenterY;
	int width=m_iW;
	int height=m_iH;
	int max_iter=m_iIteration;
	bool smoothColors=m_bSmoothColors;
	float dilatation=m_fDilatation;
	m_Mutex.unlock();

	QTime LastEventTime;
	LastEventTime.start();

	double x;
	double y;
	int iter;
	double xtemp;

	float ratio = (float)width/height;

	double newWidth = ratio*512.0;
	double newHeight = 512.0;
	double x0b;
	double y0b;

	QVector<uint> line;
	line.resize(width);

	for (int y0=m_iId;y0<height && !m_bRestart && !m_bStop && !m_bFinish;y0+=m_iThreadCount)
	{
		for (int x0=0;x0<width && !m_bRestart && !m_bStop && !m_bFinish;x0++)
		{
			x0b = ratio*512.0*x0/width;
			y0b = 512.0*y0/height;
			x = (x0b-newWidth/2.0)*zoom+cx;
			y = (y0b-newHeight/2.0)*zoom+cy;
			iter = 0;
			while (x*x + y*y <= (2*2) && iter < max_iter && !m_bRestart && !m_bStop && !m_bFinish)
			{
				xtemp = x*x - y*y + m_fReal;
				y = 2*x*y + m_fImaginary;
				x = xtemp;
				iter++;
			}

			if ( iter == max_iter )
			{
				line[x0] = 0;
			}else{
				/* Normalized Iteration Count Algorithm */
				/* Permet de lisser les couleurs */
				float val = iter;
				if (smoothColors==true)
				{
					val = val-log2(log2(sqrt(x*x+y*y)));
				}
				if (m_pGradientImage)
				{
					line[x0] = m_pGradientImage->pixel((sin(val/dilatation)+1.0)*0.5*m_pGradientImage->width(),0);
				}else{
					val = (sin(val/dilatation)+1.0)*0.5*255;
					line[x0] = qRgb(val,val,val);
				}
			}

			if (LastEventTime.elapsed()>=50)			
			{
				LastEventTime.restart();
				float progress = (y0*width+x0)/((float)width*height);
				emit Progress(progress,m_iId);
			}
		}
		if (m_pImageMutex)
		{
			m_pImageMutex->lock();
			if (!m_bRestart && !m_bStop)
			{
				for (int j=0;j<width;j++)
				{
					m_pImage->setPixel(j,y0,line[j]);
				}
			}
			m_pImageMutex->unlock();
		}
	}
}
