#ifndef __GFRACTALE_HPP__
#define __GFRACTALE_HPP__

/*
# Copyright (c) 2009 Yan Verdavaine
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
*/

#include "Commun.hpp"
/*!
 * \file GFractale.hpp
 * \brief Définition de la classe représantant une suite fractale et quelques exemples.
 * \author Yan Verdavaine
 * \version 1.0
 */

namespace Fractale
{

	/*! \class GFractale
	   * \brief Classe représentant la formule d'une suite fractale.
	   *
	   */
	struct GFractale
	{
		std::vector<point> sequence;        /*!< Séquence de points de la suite fractale*/
		unsigned int    sequenceSizeMax;    /*!< Taille maximal de la séquence de points*/

		/*!
		 *  \brief Constructeur.
		 *
		 *  \param sequenceSizeMax : taille maximale de la séquence de points.
		 */
		GFractale(unsigned int sequenceSizeMax = 1000)
		: sequenceSizeMax(sequenceSizeMax)
		{
		}

		/*!
		 *  \brief Destructeur
		 */
		virtual ~GFractale() {};

		/*!
		 *  \brief Initialisation de la fractale.
		 *
		 *  Méthode qui initialise la suite fractale pour la génération d'une séquence et retourne \f$ z_{0} \f$.
		 *
		 *  \param pInit : le point d'initialisation de la suite fractale.
		 *  \return \f$ z_{0} \f$.
		 */
		virtual point initialisation(const point & pInit) = 0;

		/*!
		 *  \brief Donne le point suivant de la fractale.
		 *
		 *  Methode qui génère le point suivant de la fractale.
		 *  \f$ z_{n+1} = f(z_{n}) \f$
		 *
		 *  \param Zn : le point courant (\f$ z_{n} \f$).
		 *  \return Le point suivant (\f$ z_{n+1} \f$).
		 */
		virtual point suivant(const point & Zn) = 0;

		/*!
		 *  \brief Condition d'arrêt de la fractale.
		 *
		 *
		 * Cette méthode correspond à la condition d'arrêt de la suite fractale.
		 * Elle doit être réimplémentée en fonction de la nature de la suite fractale.
		 *
		 *  \param Zn : le point courant (\f$ z_{n} \f$).
		 *  \return True si la génération de la fractale est finie.
		 */
		virtual bool fin(const point & Zn) = 0;

		/*!
		 *  \brief Génération de la séquence fractale.
		 *
		 *  À partir du point d'initialisation, va générer la séquence fractale.
		 *  \param pInit : le point d'initialisatione de la fractale.
		 *                  Utilisé pour générer \f$ z_{0} \f$ et l'accumulation de la séquence dans l'image.
		 *  \return La séquence de la suite fractale. Si nulle, la condition d'arrêt n'a pas été atteinte.
		 */
		virtual std::vector<point> & genererSequence(const point &pInit)
		{
			sequence.clear();
			//récuperation de Z0
			point z = initialisation(pInit);
			for ( unsigned int i = 0; !fin(z) && i < sequenceSizeMax ; ++i)
			{
				//z_(n+1) = f(z_n)
				z = suivant(z);
				sequence.push_back(z);
			}
			//si la séquence n'est pas terminée, alors la séquence est nulle
			if(!fin(z))
			{
				sequence.clear();
			}
			return sequence;
		}

	};

	/*! \class Mandelbrot
	 *  \brief Classe représentant la formule de Mandelbrot.
	 *
	 *  \f$ z_{n+1} = z_{n}^2 + c \f$
	 *  avec \f$ z_{0} = constant \f$  et \f$ c = point~d'initialisation \f$.
	 *
	 *  \image html Mandelbrot.png
	 */
	class Mandelbrot : public  GFractale
	{
		const point z0;
		point c;
	public:

		Mandelbrot(const point & z0 = point())
		:   z0(z0)
		{}
		point initialisation(const point & pInit)
		{
			c = pInit;
			return z0;
		}

		point suivant(const point &Zn)
		{
			return Zn * Zn + c;
		}
		/*!
		 *  \brief Condition d'arrêt de la fractale.
		 *
		 *  La méthode est une condition de divergence :
		 *  si le point est à l'extérieur du cercle de rayon 2, la suite diverge.
		 *  \copydetails GFractale::fin
		 */
		bool fin(const point & Zn)
		{
			return Zn.real() * Zn.real() + Zn.imag() * Zn.imag() > 4;
		}
	};
	/*! \class BateauEnFeu
	   * \brief Classe représentant la formule du BateauEnFeu.
	   *
	   *  \f$z_{n+1} = (|\Re(z_{n})|+ i*|\Im(z_{n}|)^{2} + c \f$
	   *  avec \f$ z_{0} = constant \f$  et \f$ c = point~d'initialisation \f$.
	   *
	   *  \image html BateauEnFeu.png
	   */
	struct BateauEnFeu : public  GFractale
	{
		point       c;
	public:

		point initialisation(const point & pInit)
		{
			c = pInit;
			return point();
		}
		point suivant(const point & Zn)
		{
			point Znt = point(fabs(Zn.real()), fabs(Zn.imag()));
			return Znt * Znt + c;
		}
		/*!
		 *  \copydoc Mandelbrot::fin
		 */
		bool fin(const point & Zn)
		{
			return Zn.real() * Zn.real() + Zn.imag() * Zn.imag() > 4;
		}
	};
	/*! \class Tricorn
	 * \brief Classe représentant la formule du Tricorn.
	 *
	 *  \f$z_{n+1} = \bar{z_{n}} ^{2} + c \f$
	 *  avec \f$ z_{0} = constant \f$  et \f$ c = point~d'initialisation \f$.
	 *
	 *  \image html Tricorn.png
	 */
	struct Tricorn  : public  GFractale
	{
			point c;
	public:

		point initialisation(const point & pInit)
		{
			c = pInit;
			return point();
		}

		point suivant(const point &Zn)
		{
			point Znt = point(-Zn.real(), Zn.imag());
			return Znt * Znt + c;
		}
		/*!
		 *  \copydoc Mandelbrot::fin
		 */
		bool fin(const point & Zn)
		{
			return Zn.real() * Zn.real() + Zn.imag() * Zn.imag() > 4;
		}
	};

	/*! \class Julian
	 * \brief Classe représentant la formule de Julian.
	 *
	 *  \f$ z_{n+1} = z_{n}^2 + c \f$
	 *  avec \f$ z_{0} = point~d'initialisation  \f$  et \f$ c = constant \f$.
	 *  \image html Julian.png
	 */
	class Julian : public  GFractale
	{
		const point               c;
	public:
		Julian( const point & c = point())
		:   c(c)
		{}

		point initialisation(const point & pInit)
		{
			return pInit;
		}
		point suivant(const point &Zn)
		{
			return Zn * Zn + c;
		}
		/*!
		 *  \copydoc Mandelbrot::fin
		 */
		bool fin(const point & Zn)
		{
			return Zn.real() * Zn.real() + Zn.imag() * Zn.imag() > 4;
		}

	};

	/*! \class Newton
	 * \brief Classe représentant une formule simple de Newton.
	 *
	 *  \f$ z_{n+1} = z_{n} - \frac{f(z_{n})}{f'(z_{n})} \f$
	 *  avec \f$ f(z)= z^3 - 1 \f$ et \f$ z_{0} = point~d'initialisation  \f$.
	 *
	 *  \image html Newton.png
	 */
	struct Newton : public  GFractale
	{
		point initialisation(const point & pInit)
		{
			return pInit;
		}
		point suivant(const point &Zn)
		{
			return Zn - (Zn * Zn * Zn - 1.) / (3. * Zn * Zn );
		}
		bool fin(const point & Zn)
		{
			return std::abs(Zn * Zn * Zn - 1.) < 0.01;
		}
	};

	/*! \class Nova
	 * \brief Classe représentant une formule de Nova.
	 *
	 * \f$ z_{n+1} = z_{n} - R * \frac{z_{n}^i}{i * z_{n}^{i-1}} + c \f$  avec \n
	 * \f$ R = 1. \f$,
	 * \f$ i = 3. \f$,
	 * \f$ z_{0} = 1. \f$ et
	 * \f$ c = constant \f$.
	 *
	 *  \image html Nova.png
	 */
	struct Nova : public GFractale
	{
		point old;
		point c;
	public:
		Nova(unsigned int sequenceSizeMax = 1000)
			:GFractale(sequenceSizeMax)
		{

		}
		point initialisation(const point & pInit)
		{
			old =point();
			c = pInit;
			return point(1.);
		}

		point suivant(const point &Zn)
		{
			old = Zn;
			return Zn - 2. * (Zn * Zn * Zn - 1.) / (3. * Zn * Zn ) + c;
		}

		/*!
		 *  \brief Condition d'arrêt de la fractale.
		 *
		 *  La méthode est une condition de convergance :
		 *  si \f$z_{n} \simeq z_{n-1}\f$, la suite a convergé.
		 *
		 *  \param z : le point courant (\f$z_{n}\f$).
		 *  \return True si la génération de la fractale est finie.
		 */
		bool fin(const point &z)
		{
			return std::abs(z - old) <0.05;
		}
	};

	/*! \class CliffordAttractors
	 * \brief Classe représentant une formule de CliffordAttractors.
	 *
	 *  \f$  \Re (z_{n+1}) = sin(a * \Im(z_{n})) + c * cos(a * \Re(z_{n})) \f$ \n
	 *  \f$  \Im (z_{n+1}) = sin(b * \Re(z_{n})) + d * cos(b * \Im(z_{n})) \f$ \n
	 *
	 *  avec \f$ z_{0} = point~d'initialisation \f$ et \f$ a,b,c,d \f$ des constantes.
	 *  \image html CliffordAttractors.png
	 */
	struct CliffordAttractors : public  GFractale
	{
		const double a;
		const double b;
		const double c;
		const double d;
		CliffordAttractors(double a =-1.4, double b = 1.6, double c = 1.0,double d = 0.7)
		:GFractale(10000),a(a),b(b),c(c),d(d){}

		point initialisation(const point & pInit)
		{
			return pInit;
		}

		point suivant(const point &Zn)
		{

			return point
					(
						sin(a * Zn.imag()) + c * cos(a * Zn.real()),
						sin(b * Zn.real()) + d * cos(b * Zn.imag())
					);
		}

		/*!
		 *  \brief Condition d'arrêt de la fractale.
		 *
		 *  Cette fractale n'a aucune condition d'arrêt.
		 *
		 *  \param Zn : le point courant (\f$z_{n}\f$).
		 *  \return False.
		 */
		bool fin(const point &/*Zn*/)
		{
			return false;
		}

		virtual std::vector<point> & genererSequence(const point & c)
		{
			sequence.clear();
			sequence.push_back(c);

			point z = initialisation(c);
			for ( unsigned int i = 0; i < sequenceSizeMax ; ++i)
			{
				z = suivant(z);
				sequence.push_back(z);
			}
			return sequence;
		}
	};
}

#endif
