#ifndef __IMAGE_HPP__
#define __IMAGE_HPP__

/*
# Copyright (c) 2009 Yan Verdavaine
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
*/

#include <vector>
#include <exception>

/*!
 * \file GFractale.hpp
 * \brief Définition de la classe représantant une suite fractal et quelques exemples.
 * \author Yan Verdavaine
 * \version 1.0
 */

namespace Fractale
{

	/*! \class MauvaiseCoordonnee
	* \brief Exception correspondant à l'accès à un pixel non dans l'image.
	*
	*/
	class MauvaiseCoordonnee : public std::exception
	{
	public :
		const char * 	what () const throw ()
		{
			return "les coordonnees i,j ne sont pas dans l'image";
		}
	};

	/*! \class MauvaiseTailleImage
	* \brief Exception correspondant à la manipulation de deux images dont la taille est différente.
	*
	*/
	class MauvaiseTailleImage : public std::exception
	{
	public :
		const char * 	what () const throw ()
		{
			return "les images n'ont pas la meme taille";
		}
	};


	/*! \class Image
	* \brief Classe template correspondant à une image.
	* \tparam PIX Type du pixel.
	*
	*/
	template<typename PIX>
	class Image
	{


		unsigned int m_h; /*!< Hauteur de l'image*/
		unsigned int m_l; /*!< Largeur de l'image*/
		std::vector<PIX>  m_buffer;/*!< Buffer 1D des pixels de l'image*/

		/*!
		 *  \brief Prédicat qui vérifie que la coordonnée (i,j) est dans l'image.
		 *
		 *  Émet une exception si cette coordonnée n'est pas dans l'image.
		 */
		void PredIJ(unsigned int i,unsigned int j) const 
		{
			if( !( i < m_h && j < m_l))
			{
				throw MauvaiseCoordonnee();
			};
		}

		/*!
		 *  \brief Prédicat qui vérifie que la taille des deux images est identique.
		 *
		 *  Émet une exception si la taille est différente.
		 */
		void PredImgTaille(const Image &I1,const Image & I2)
		{
			if( I1.m_h != I2.m_h || I1.m_l != I2.m_l)
			{
				throw MauvaiseTailleImage();
			};
		}
	public :
		/*! \var typedef  std::vector<PIX>::iterator        iterator
		 *  \brief Itérateur qui permet de parcourir tous les pixels de l'image sous forme 1D.
		*/
		typedef  typename std::vector<PIX>::iterator        iterator;

		/*! \var typedef  std::vector<PIX>::const_iterator   const_iterator
		 *  \brief Itérateur constant qui permet de parcourir tous les pixels de l'image sous forme 1D.
		*/
		typedef  typename std::vector<PIX>::const_iterator   const_iterator;
		
		/*!
		 *  \brief Constructeur
		 */
		Image(void)
			:m_h(0),m_l(0),m_buffer(0)
		{
		}

		/*!
		 *  \brief Constructeur
		 *
		 *  \param h : hauteur de l'image.
		 *  \param l : largeur de l'image.
		 */
		Image(unsigned int h,unsigned int l)
			:m_h(h),m_l(l),m_buffer(h*l)
		{
		}
		/*!
		 *  \brief Constructeur de copie.
		 */
		Image(const Image & I)
			:m_h(I.m_h),m_l(I.m_l),m_buffer(I.m_buffer)
		{
		}

		 /*!
		 *  \brief Retaille l'image
		 *
		 *  \param h : hauteur de l'image.
		 *  \param l : largeur de l'image.
		 */
		void resize(unsigned int h,unsigned int l)
		{
			m_h = h;
			m_l = l;
			m_buffer.resize(h * l);
		}

		/*!
		 *  \brief Vérifie la coordonnée image (i,j)
		 *
		 *  \return True si la coordonnée est valide.
		 */
		bool isValid(unsigned int i,unsigned int j)
		{
			return i < m_h && j < m_l;
		}

		/*!
		 *  \brief Accède au pixel (i,j)
		 *  \exception La coordonnée (i,j) n'est pas dans l'image.
		 *  \return Référence sur le pixel.
		 */
		PIX & pixel(unsigned int i, unsigned int j) 
		{
			PredIJ(i,j);
			return m_buffer[i*m_l + j];
		}

		/*!
		 *  \brief accès constant au pixel (i,j)
		 *  \exception La coordonnée (i,j) n'est pas dans l'image.
		 *  \return Référence constante sur le pixel.
		 */
		const PIX & pixel(unsigned int i, unsigned int j) const
		{
			PredIJ(i,j);
			return m_buffer[i*m_l + j];
		}

		/*!
		 *  \brief Donne la hauteur de l'image.
		 */
		unsigned int hauteur() const {return m_h;}

		/*!
		 *  \brief Donne la largeur de l'image.
		 */
		unsigned int largeur() const {return m_l;}

		/*!
		 *  \brief Donne l'itérateur de début du buffer pixel.
		 */
		iterator begin() {return m_buffer.begin();}

		/*!
		 *  \brief Donne l'itérateur de fin du buffer pixel.
		 */
		iterator end() {return m_buffer.end();}

		/*!
		 *  \brief Donne l'itérateur constant de début du buffer pixel.
		 */
		const_iterator begin()const {return m_buffer.begin();}

		/*!
		 *  \brief Donne l'itérateur constant de fin du buffer pixel.
		 */
		const_iterator end()const {return m_buffer.end();}

		/*!
		 *  \brief Somme pixel par pixel deux images.
		 *  \exception Les deux images n'ont pas la même taille.
		 *  \param I : seconde image de l'addition.
		 *  \return L'image résultante.
		 *
		 */
		Image<PIX> operator+  (const Image & I) const 
		{
			//verifie la taille.
			PredImgTaille(*this,I);
			Image<PIX> tmp(*this);
			tmp += I;
			return tmp;
		}

		/*!
		 *  \brief Ajoute les valeur des pixels d'une image dans les pixels correspondants.
		 *  \exception Les deux images n'ont pas la même taille.
		 *  \param I : l'image à additionner.

		 *  \return This.
		 *
		 */
		Image<PIX> & operator+= (const Image & I)
		{
			//verifie la taille.
			PredImgTaille(*this,I);

			typename Image<PIX>::iterator  it = begin();
			typename Image<PIX>::iterator  end = end();
			typename Image<PIX>::const_iterator   it1 = I.begin();
			while(it != end)
			{
				*it += *it1 ;
				++it;
				++it1;
			}
			return *this;
		}

	};
}

#endif
