/*
# Copyright (c) 2010 Alexandre LAURENT
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
*/

#include "ColoursDialog.hpp"

#include <QWidget>
#include <QVBoxLayout>
#include <QFrame>
#include <QGridLayout>
#include <QGroupBox>
#include <QPushButton>
#include <QFileDialog>
#include <QDir>
#include <QDataStream>

#include "PaletteDisplayer.hpp"
#include "GradientDialog.hpp"
#include "ColourDialog.hpp"

#ifndef QT_NO_DEBUG
	#include <iostream>
#endif

ColoursDialog :: ColoursDialog(const QVector<QRgb> table, QWidget* parent)
	:QDialog(parent),
	pMainLayout(NULL), pButtonFrame(NULL), pButtonLayout(NULL),
	pd(NULL),
	pRandomButton(NULL), pGradientButton(NULL), pCustomiseButton(NULL), pValidateButton(NULL), pCancelButton(NULL),
	pLoadPaletteButton(NULL), pSavePaletteButton(NULL),
	oldTable(table), table(table)
{
	pMainLayout = new QVBoxLayout(this);
	
	pd = new PaletteDisplayer(table,this);
	
	pMainLayout->addWidget(pd,0,Qt::AlignHCenter) ;
	
	pButtonFrame = new QFrame(this);
	pButtonLayout = new QGridLayout(pButtonFrame);
	pMainLayout->addWidget(pButtonFrame,1,0);
	
	pRandomButton = new QPushButton(tr("Randomise"), this);
	pGradientButton = new QPushButton(tr("Gradient"), this);
	pColoration1Button = new QPushButton(tr("Coloration1"), this);
	pColoration2Button = new QPushButton(tr("Coloration2"), this);
	pColoration3Button = new QPushButton(tr("Coloration3"), this);
	pCustomiseButton = new QPushButton(tr("Customise"), this);
	
	pApplyButton = new QPushButton(tr("Apply"), this);
	pValidateButton = new QPushButton(tr("Ok"), this);
	pCancelButton = new QPushButton(tr("Cancel"), this);
	
	pLoadPaletteButton = new QPushButton(tr("Load palette"), this);
	pSavePaletteButton = new QPushButton(tr("Save palette"), this);
	
	pButtonLayout->addWidget(pRandomButton, 1, 0);
	pButtonLayout->addWidget(pGradientButton, 1, 1);
	pButtonLayout->addWidget(pColoration1Button, 1, 2);
	pButtonLayout->addWidget(pColoration2Button, 1, 3);
	pButtonLayout->addWidget(pColoration3Button, 1, 4);
	pButtonLayout->addWidget(pCustomiseButton, 1, 5);
	pButtonLayout->addWidget(pApplyButton, 2, 0);
	pButtonLayout->addWidget(pValidateButton, 2, 1);
	pButtonLayout->addWidget(pCancelButton, 2, 2);
	pButtonLayout->addWidget(pLoadPaletteButton, 2, 3);
	pButtonLayout->addWidget(pSavePaletteButton, 2, 4);
	
	connect(pRandomButton, SIGNAL(clicked()), this, SLOT(randomColours()));
	connect(pGradientButton, SIGNAL(clicked()), this, SLOT(gradientColours()));
	connect(pColoration1Button, SIGNAL(clicked()), this, SLOT(coloration1()));
	connect(pColoration2Button, SIGNAL(clicked()), this, SLOT(coloration2()));
	connect(pColoration3Button, SIGNAL(clicked()), this, SLOT(coloration3()));
	connect(pCustomiseButton, SIGNAL(clicked()), this, SLOT(customiseTable()));
	connect(pApplyButton, SIGNAL(clicked()), this, SLOT(apply()));
	connect(pValidateButton, SIGNAL(clicked()), this, SLOT(validate()));
	connect(pCancelButton, SIGNAL(clicked()), this, SLOT(cancel()));
	connect(pLoadPaletteButton, SIGNAL(clicked()), this, SLOT(loadPalette()));
	connect(pSavePaletteButton, SIGNAL(clicked()), this, SLOT(savePalette()));

#ifndef QT_NO_DEBUG
	std::cout << "ColoursDialog created" << std::endl;
#endif		
}

ColoursDialog :: ~ColoursDialog(void)
{
	delete pSavePaletteButton;
	delete pLoadPaletteButton;
	delete pCancelButton;
	delete pValidateButton;
	delete pApplyButton;
	delete pColoration1Button;
	delete pColoration2Button;
	delete pColoration3Button;
	delete pCustomiseButton;
	delete pGradientButton;
	delete pRandomButton;
	
	delete pd;
	
	delete pButtonLayout;
	delete pButtonFrame;
	
	delete pMainLayout;

#ifndef QT_NO_DEBUG
	std::cout << "ColoursDialog deleted" << std::endl;
#endif
}

void ColoursDialog :: randomColours(void)
{
#ifndef QT_NO_DEBUG
	std::cout << "SLOT: ColoursDialog :: randomColours" << std::endl;
#endif
	
	for ( QVector<QRgb>::iterator it_colours = table.begin() ; it_colours != table.end() ; ++it_colours )
	{
		*it_colours = qRgb(qrand()%256, qrand()%256, qrand()%256);
	}
	
	pd->setPalette(table);
}
		
void ColoursDialog :: gradientColours(void)
{	
	GradientDialog gd(QColor(table[0]), QColor(table[255]),this);
	
	if ( gd.exec() == 0 )	// N'a pas annulé donc on fait la modification
	{
		QColor startingColor = gd.getStartingColor();
		QColor endingColor = gd.getEndingColor();
		
		// Ici nous calculons le pas, pour l'interpolation
		float rStep = (endingColor.red() - startingColor.red()) / 256.0f;
		float gStep = (endingColor.green() - startingColor.green()) / 256.0f;
		float bStep = (endingColor.blue() - startingColor.blue()) / 256.0f;
		
		// Comme nous avons des valeurs en float, et que nous voulons des valeurs en unsigned char ... le pas ne peut être ajouté directement
		// Donc nous utilisons une sorte de compteur ( ou pas accumulatif ) que nous allons ajouter à la couleur de depart
		float rCounter = 0;
		float gCounter = 0;
		float bCounter = 0;
		
		for ( QVector<QRgb>::iterator it_colours = table.begin() ; it_colours != table.end() ; ++it_colours )
		{
			*it_colours = qRgb(startingColor.red() + rCounter, startingColor.green() + gCounter, startingColor.blue() + bCounter);
			
			rCounter += rStep;
			gCounter += gStep;
			bCounter += bStep;
		}
	}
	
	pd->setPalette(table);
	
#ifndef QT_NO_DEBUG
	std::cout << "SLOT: ColoursDialog :: gradientColours" << std::endl;
#endif

}

void ColoursDialog :: coloration1(void)
{
	unsigned int counter = 1;
	//unsgined int step = 16777216 / 256;	// 2^24 / 256 -> 2
	
	for ( QVector<QRgb>::iterator it_colours = table.begin() ; it_colours != table.end() ; ++it_colours )
	{
		*it_colours = qRgb(256 - counter, 256 - counter * 2, 256 - counter * 4);
		counter ++;
	}
	
	pd->setPalette(table);
	
#ifndef QT_NO_DEBUG
	std::cout << "SLOT: ColoursDialog :: coloration1" << std::endl;
#endif
}

void ColoursDialog :: coloration2(void)
{
	unsigned int counter = 1;
	//unsgined int step = 16777216 / 256;	// 2^24 / 256 -> 2
	
	for ( QVector<QRgb>::iterator it_colours = table.begin() ; it_colours != table.end() ; ++it_colours )
	{
		*it_colours = qRgb(abs(128 - counter), (128 + counter) % 256, 255);
		counter ++;
	}
	
	pd->setPalette(table);
	
#ifndef QT_NO_DEBUG
	std::cout << "SLOT: ColoursDialog :: coloration2" << std::endl;
#endif
}

void ColoursDialog :: coloration3(void)
{
	unsigned int counter = 1;
	//unsgined int step = 16777216 / 256;	// 2^24 / 256 -> 2
	
	for ( QVector<QRgb>::iterator it_colours = table.begin() ; it_colours != table.end() ; ++it_colours )
	{
		*it_colours = qRgb(abs(256 - counter * 2), 0 + counter, abs(256 - counter * 8)%256);
		counter ++;
	}
	
	pd->setPalette(table);
	
#ifndef QT_NO_DEBUG
	std::cout << "SLOT: ColoursDialog :: coloration3" << std::endl;
#endif
}
		
void ColoursDialog :: customiseTable(void)
{
	ColourDialog cd(table, this);
	
	connect(&cd, SIGNAL(needsApply(QVector<QRgb>)), this, SLOT(applyNewTable(QVector<QRgb>)));
	
	if ( cd.exec() == 0 )	// C'est correctement passé
	{
		table = cd.getTable();
		pd->setPalette(table);
	}
	else
	{
		// On remet l'ancienne table dans le palette displayer
		pd->setPalette(table);
	}
	
	disconnect(&cd, SIGNAL(needsApply(QVector<QRgb>)), this, SLOT(applyNewTable(QVector<QRgb>)));
	
#ifndef QT_NO_DEBUG
	std::cout << "SLOT: ColoursDialog :: customiseTable" << std::endl;
#endif
}
	
void ColoursDialog :: apply(void)
{
	emit coloursTableChanged(table);
	
#ifndef QT_NO_DEBUG
	std::cout << "SLOT: ColoursDialog :: apply" << std::endl;
#endif
}

void ColoursDialog :: applyNewTable(QVector<QRgb> newTable)
{
	// On met à jour la palette displayer
	//pd->setPalette(newTable);
	// Pas utile car de toute façon, la fenêtre n'étant pas réaffiché ... pas de mise à jour
	
	emit coloursTableChanged(newTable);
	
#ifndef QT_NO_DEBUG
	std::cout << "SLOT: ColoursDialog :: applyNewTable" << std::endl;
#endif
}

void ColoursDialog :: validate(void)
{
	this->done(0);
	
#ifndef QT_NO_DEBUG
	std::cout << "SLOT: ColoursDialog :: validate" << std::endl;
#endif
}

void ColoursDialog :: cancel(void)
{
	table = oldTable;
	this->done(1);
	
#ifndef QT_NO_DEBUG
	std::cout << "SLOT: ColoursDialog :: cancel" << std::endl;
#endif
}

void ColoursDialog :: loadPalette(void)
{
	QString filename = QFileDialog::getOpenFileName(this, tr("Open palette file"), QDir::currentPath(), "Palette (*.pal)");
	
	if ( !filename.endsWith(QString(".pal")) )
	{
		filename = filename + QString(".pal");
	}
	
	if ( !filename.isEmpty() )
	{
		QFile paletteFile(filename);
		
		if ( paletteFile.open(QIODevice::ReadOnly) )
		{
			QVector<QRgb> table;
			QDataStream dataFile(&paletteFile);
			
			while (!dataFile.atEnd())
			{
				QRgb color;
				
				dataFile >> color;
				table.push_back(color);
			}

			this->table = table;
			pd->setPalette(table);
			
			paletteFile.close();
		}
	}
	
#ifndef QT_NO_DEBUG
	std::cout << "SLOT: ColoursDialog :: loadPalette" << std::endl;
#endif
}

void ColoursDialog :: savePalette(void)
{
	QString filename = QFileDialog::getSaveFileName(this, tr("Save palette file"), QDir::currentPath(), "Palette (*.pal)");
	
	if ( !filename.endsWith(QString(".pal")) )
	{
		filename = filename + QString(".pal");
	}
	
	if ( !filename.isEmpty() )
	{
		QFile paletteFile(filename);
		
		if ( paletteFile.open(QIODevice::WriteOnly) )
		{
			QDataStream dataFile(&paletteFile);
			
			// Pour la table entière
			for ( int i = 0 ; i < table.size()  ; i++ )
			{
				dataFile << table[i];
			}

			paletteFile.close();
		}
	}
	
#ifndef QT_NO_DEBUG
	std::cout << "SLOT: ColoursDialog :: savePalette" << std::endl;
#endif
}
