#ifndef __COLOURSDIALOG_HPP__
#define __COLOURSDIALOG_HPP__

/*
# Copyright (c) 2010 Alexandre LAURENT
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
*/

#include <QDialog>

#include <QVector>
#include <QColor>

class QWidget;
class QVBoxLayout;
class QFrame;
class QGridLayout;
class PaletteDisplayer;

/*! \class ColoursDialog
 * \brief Fenêtre pour la selection des couleurs de l'image.\n
 * Je propose plusieurs fois prédéfini, sinon on peut aussi charger à partir d'un fichier, ou alors définir un dégradé, ou encore choisir les couleurs une par une.
 */
class ColoursDialog : public QDialog
{
	Q_OBJECT
	
	private:
	
		QVBoxLayout* pMainLayout;		/*!< La grille pour l'agencement de la fenêtre */
		
		QFrame* pButtonFrame;			/*!< Cadre pour les boutton ( englobe la grille pour les boutons ) */
		QGridLayout* pButtonLayout;		/*!< Grille pour l'agencement des boutons */
		
		PaletteDisplayer* pd;			/*!< Objet pour afficher la table des couleurs */
	
		QPushButton* pRandomButton;		/*!< Bouton pour la coloration aléatoire */
		QPushButton* pGradientButton;	/*!< Bouton pour la coloration par dégradé */
		QPushButton* pColoration1Button;	/*!< Bouton pour la coloration prédéfinie 1 */
		QPushButton* pColoration2Button;	/*!< Bouton pour la coloration prédéfinie 2 */
		QPushButton* pColoration3Button;	/*!< Bouton pour la coloration prédéfinie 3 */
		QPushButton* pCustomiseButton;		/*!< Bouton pour la coloration entièrement manuelle */
		QPushButton* pApplyButton;			/*!< Bouton pour appliquer la nouvelle coloration */
		QPushButton* pValidateButton;		/*!< Bouton pour valider la nouvelle coloration */
		QPushButton* pCancelButton;			/*!< Bouton pour annuler */
		
		QPushButton* pLoadPaletteButton;	/*!< Bouton pour charger une palette */
		QPushButton* pSavePaletteButton;	/*!< Bouton pour sauvegarder la palette courante */
		
		QVector<QRgb> oldTable;				/*!< Table temporaire ( pour remettre correctement après annulation */
		QVector<QRgb> table;				/*!< Table de couleur */
		
	signals:
		/*!
		 * \brief Signal pour indiquer que la table de couleur à changer ( donc on va appliquer la nouvelle table à l'élément qui reçoit le signal )
		 */
		void coloursTableChanged(QVector<QRgb>);
		
	public slots:
	
		/*!
		 * \brief Appelée après un clic sur le bouton de coloration aléatoire.\n
		 * Change les couleurs aléatoirement.
		 */
		void randomColours(void);
		
		/*!
		 * \brief Appelée après un clic sur le bouton de coloration par dégradé.\n
		 * Ouvre une boite pour choisir deux couleurs.\n
		 * Change les couleurs selon un dégradé linéaire ( interpolation entre deux couleurs ).
		 */
		void gradientColours(void);
		
		/*!
		 * \brief Appelée après un clic sur le bouton de coloration prédéfinie 1.\n
		 * Change les couleurs selon la fonction prédifinie.
		 */
		void coloration1(void);
		
		/*!
		 * \brief Appelée après un clic sur le bouton de coloration prédéfinie 2.\n
		 * Change les couleurs selon la fonction prédifinie.
		 */
		void coloration2(void);
		
		/*!
		 * \brief Appelée après un clic sur le bouton de coloration prédéfinie 3.\n
		 * Change les couleurs selon la fonction prédifinie.
		 */
		void coloration3(void);
		
		/*!
		 * \brief Appelée après un clic sur le bouton de coloration manuelle.\n
		 * Ouvre une boite de dialogue pour que l'utilisateur crée une palette manuellement.
		 */
		void customiseTable(void);
		
		/*!
		 * \brief Appelée après un clic sur le bouton apply.\n
		 * Applique la palette courante à l'image.
		 */
		void apply(void);
		
		/*!
		 * \brief Appelée après la reception d'un signal venant de ColourDialog pour pouvoir appliqué la palette en cours de création.\n
		 */
		void applyNewTable(QVector<QRgb> newTable);

		/*!
		 * \brief Appelée après un clic sur le bouton Ok.\n
		 * Valide la palette courante.
		 */		
		void validate(void);
		
		/*!
		 * \brief Appelée après un clic sur le bouton cancel.\n
		 * Restore la palette du début.
		 */
		void cancel(void);
		
		/*!
		 * \brief Appelée après un clic sur le bouton load.\n
		 * Charge une palette à partir d'un fichier.
		 */
		void loadPalette(void);
		
		/*!
		 * \brief Appelée après un clic sur le bouton save.\n
		 * Sauvegarde la palette courante dans un fichier.
		 */
		void savePalette(void);
	
	public:
	
		/*!
		 *  \brief Constructeur
		 *
		 *  \param table : la table de couleur qui va être modifié.
		 *  \param parent : La fenêtre parente.
		 */
		ColoursDialog(const QVector<QRgb> table, QWidget* parent);
		
		/*!
		 * \brief Destructeur
		 */
		~ColoursDialog(void);
		
		/*!
		 *  \brief Getteur pour la table des couleurs.
		 *  \return La nouvelle table de couleur.
		 */
		QVector<QRgb> getColourTable(void)const { return table; };
};

#endif
