/*
# Copyright (c) 2010 Alexandre LAURENT
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
*/

#include "MainWindow.hpp"
#include "ProgressDialog.hpp"
#include "PromptGenerationDialog.hpp"
#include "PromptCurveDialog.hpp"
#include "PromptSettings.hpp"
#include "GammaDialog.hpp"
#include "ColoursDialog.hpp"
#include "WidgetImage.hpp"

#include "../Calculation/Generation.hpp"
#include "../Calculation/Curve.hpp"

#include <QApplication>
#include <QMainWindow>
#include <QWidget>
#include <QTabWidget>
#include <QFileDialog>
#include <QMenuBar>
#include <QMenu>
#include <QAction>
#include <QImageWriter>
#include <QImageReader>
#include <QTime>

#include <QString>
#include <QList>

#include <iostream>

#ifndef QT_NO_DEBUG
	#include <iomanip>
#endif

MainWindow :: MainWindow(const QString& windowName, const unsigned int width, const unsigned int height, QWidget* parent/*=0*/)
	:QMainWindow(parent)
{
	// Mise à zero des pointeurs
	pFileMenu = NULL;
	pGenerationMenu = NULL;
	pImageMenu = NULL;
	
	pGammaCorrectionAction = NULL;
	
	// Set the title and default size
	this->setWindowTitle(windowName);
	this->resize(width, height);
	
	this->createMenus();
	
	tabImage = new QTabWidget(this);
	
	tabImage->setTabsClosable(true);
	
	this->setCentralWidget(tabImage);
	
	connect(tabImage, SIGNAL(tabCloseRequested(int)), this, SLOT(closeTab(int)));
	
#ifndef QT_NO_DEBUG
	std::cout << "MainWindow constructed (" << width << ";" << height << ")" << std::endl;
#endif
}

MainWindow :: ~MainWindow(void)
{
	// Nettoyage des actions
	for ( QList<QAction*>::iterator it_pAction = actions.begin() ; it_pAction != actions.end() ; ++it_pAction )
	{
		//pFileMenu->removeAction(*it_pAction);
		delete (*it_pAction);
	}
	actions.clear();
	
	// Delete de NULL ne fait rien
	delete pImageMenu;
	delete pGenerationMenu;
	delete pFileMenu;
	
	delete tabImage;

#ifndef QT_NO_DEBUG
	std::cout << "MainWindow deleted" << std::endl;
#endif	
}

QAction* MainWindow :: addAction(QMenu* pParentMenu,const QString& entryName, const QString& keySequence, const QString& statusTipMessage, const QObject* pObject, const char* pSlot, const bool enabled, const bool checkable)
{	
	QAction* pAction = pParentMenu->addAction(entryName, pObject, pSlot, QKeySequence(keySequence));
	
	pAction->setStatusTip(statusTipMessage);
	pAction->setEnabled(enabled);
	pAction->setCheckable(checkable);
	pAction->setChecked(checkable);
	
	// Garde le QAction dans une liste pour pouvoir le nettoyer plus tard
	actions.push_back(pAction);
	
#ifndef QT_NO_DEBUG
	std::cout << "\tNew action added to a menu (" << entryName.toStdString() << "; Tip: " << statusTipMessage.toStdString() << "; Raccourci: " << keySequence.toStdString()  << ")" << std::endl;
#endif	
	
	return pAction;
}

void MainWindow :: createMenus(void)
{
	if ( pFileMenu != NULL )	// Le test pourrait être enlevé, delete NULL ne fait rien
	{
		delete pFileMenu;
	}
	// File
	/**
	 * - Load
	 * -------
	 * - Save
	 * -------
	 * - Quit
	 **/
	pFileMenu = new QMenu(tr("&File"), this);
	{
		pSaveAction = this->addAction(pFileMenu, tr("&Save"), tr("Ctrl+S"), tr("Save the fractale in a file"), this, SLOT(savePicture()));
		actions.push_back(pFileMenu->addSeparator());
		this->addAction(pFileMenu, tr("&Quit"), tr("Ctrl+Q"), tr("Quit the application"), qApp, SLOT(quit()));
	}
	
	this->menuBar()->addMenu(pFileMenu);
	
	
	if ( pGenerationMenu != NULL )
	{
		delete pGenerationMenu;
	}
	// Generation
	/**
	 * Generate...
	 * -----------
	 * Preferences
	 **/
	pGenerationMenu = new QMenu(tr("&Generation"), this);
	{
		this->addAction(pGenerationMenu, tr("&Generate..."), tr("Ctrl+G"), tr("Generate fractales"), this, SLOT(startGeneration()));
		this->addAction(pGenerationMenu, tr("&Generate curves..."), tr("Ctrl+F"), tr("Generate fractales based on curves"), this, SLOT(startCurve()));
		actions.push_back(pGenerationMenu->addSeparator());
		this->addAction(pGenerationMenu, tr("&Preferences"), tr("Ctrl+Alt+P"), tr("Where to change some settings on the generation"), this, SLOT(settingsBox()));
	}
	
	this->menuBar()->addMenu(pGenerationMenu);
	
	// Image
	/**
	 *
	 *	Gamma Correction
	 * 
	 **/
	pImageMenu = new QMenu(tr("&Picture"), this);
	{
		pGammaCorrectionAction = this->addAction(pImageMenu, tr("Gamma &correction..."), tr("Ctrl+C"), tr("Change the gamma of the picture"), this, SLOT(gammaCorrection()));
		pRandomColoursAction = this->addAction(pImageMenu, tr("&Change colours..."), tr("Ctrl+R"), tr("Change the colour of the picture"), this, SLOT(changeColours()));
	}
	this->desactivateImageMenu();
	
	this->menuBar()->addMenu(pImageMenu);
}

void MainWindow :: activateImageMenu(void)
{
	if ( pSaveAction != NULL )
		pSaveAction->setEnabled(true);
	
	if ( pGammaCorrectionAction != NULL )
		pGammaCorrectionAction->setEnabled(true);
		
	if ( pRandomColoursAction != NULL )
		pRandomColoursAction->setEnabled(true);
		
	if ( pImageMenu != NULL )
		pImageMenu->setEnabled(true);
}

void MainWindow :: desactivateImageMenu(void)
{
	if ( pSaveAction != NULL )
		pSaveAction->setEnabled(false);
	
	if ( pGammaCorrectionAction != NULL )
		pGammaCorrectionAction->setEnabled(false);
		
	if ( pRandomColoursAction != NULL )
		pRandomColoursAction->setEnabled(false);
		
	if ( pImageMenu != NULL )
		pImageMenu->setEnabled(false);
}




/// SLOTS
void MainWindow :: savePicture(void)
{
#ifndef QT_NO_DEBUG
	std::cout << "\tSLOT: savePicture() ";
#endif

	QString formatFilters;
	QString selectedFilter;
	QList<QByteArray> formatSupported = QImageWriter::supportedImageFormats();
	bool first = true;
	
	for ( QList<QByteArray>::iterator it_list = formatSupported.begin() ; it_list != formatSupported.end() ; ++it_list )
	{
		if ( first )
		{
			// Pour avoir un filtre selectionné par default
			selectedFilter += it_list->data();
			first = false;
		}
		else
		{
			// Separateur entre les différents filtres
			formatFilters += ";;";	
		}

#ifndef QT_NO_DEBUG		
		std::cout << "Supported: " << it_list->data() << std::endl;
#endif
		
		// Construction d'une liste de filtres basé sur ce qui est supporté
		formatFilters += "*.";
		formatFilters += it_list->data();
		
	}
	
	// On demande à l'utilisateur le nom du fichier
	QString fileName = QFileDialog::getSaveFileName(this, tr("Save fractale file"), QDir::currentPath(), formatFilters, &selectedFilter);
	if ( fileName != "" )	// L'utilisateur n'a pas annulé ou choisi un fichier sans nom
	{
		QString completeFileName;
		bool hasExtension = false;
		
		// On regarde si dans le nom du fichier, il a directement mit l'extension
		for ( QList<QByteArray>::iterator it_list = formatSupported.begin() ; it_list != formatSupported.end() ; ++it_list )
		{
			hasExtension |= fileName.endsWith(QString(it_list->data()));
		}
		
		if ( hasExtension )
		{
			// Si extension, on laisse comme ça
			completeFileName = fileName;
		}
		else
		{
			// Sinon, on construit le nom du fichier avec l'extension selectionnée
			completeFileName = fileName + QString(selectedFilter.right(4));
		}
		// Ajoute l'extension à la fin du nom mis par l'utilisation
		((WidgetImage*)(tabImage->widget(tabImage->currentIndex())))->getImage().save(completeFileName);
		
		#ifndef QT_NO_DEBUG
			std::cout << "(Fichier selectionné: '" << fileName.toStdString() << " ; Filtre selectionné:" << selectedFilter.toStdString() <<  " ; Fichier de sorite: " << completeFileName.toStdString() <<"')" << std::endl;
		#endif	
	}
}

void MainWindow :: startGeneration(void)
{
	PromptGenerationDialog prompt(this);
	int result = prompt.exec();
	
	if ( result == 0 )	// Le prompt a fonctionner et renvoie des valeurs correctes
	{
		// Utilisation des valeurs du prompt pour lancer l'algorithme
		Params params;
		
		params.lifct = prompt.getInitializor();
		params.lgfct = prompt.getGenerator();
		params.lafct = prompt.getAccumulator();
		
		params.iWidth = prompt.getImageW();
		params.iHeight = prompt.getImageH();
		
		params.zWidth = prompt.getZoneW();
		params.zHeight = prompt.getZoneH();
		params.zX = prompt.getZoneX();
		params.zY = prompt.getZoneY();
		
#ifndef QT_NO_DEBUG
		std::cout << "Initializor: " << params.lifct << std::endl;
		std::cout << "Generator: " << params.lgfct << std::endl;
		std::cout << "Accumulator: " << params.lafct << std::endl << std::endl;
		
		std::cout << "Image(" << params.iWidth << ";" << params.iHeight << ")" << std::endl;
		std::cout << "Zone(" << params.zX << ";" << params.zY << ";" << params.zWidth << ";" << params.zHeight << ")" << std::endl << std::endl;
#endif

		{
			// Construction d'une table par défault ... en niveau de gris
			QVector<QRgb> table;
			QTime* timeStamp = new QTime();
			
			for ( unsigned int i = 0 ; i < 256 ; i++ )
			{
				table.push_back(qRgb(i,i,i));
			}
			
			Generation* g = new Generation(params, table, this);
			
			generationThreads.push_back(g);
			timesGeneration.push_back(timeStamp);
			connect(g, SIGNAL(finished()), this, SLOT(deleteThreadGeneration()));
			
			// Demarrage du temps ( comme chrono )
			timeStamp->start();
			g->start();
		}
	}

#ifndef QT_NO_DEBUG
	std::cout << "\tSLOT: startGeneration()" << std::endl;
#endif
}

void MainWindow :: startCurve(void)
{
	PromptCurveDialog prompt(this);
	int result = prompt.exec();
	
	if ( result == 0 )	// Le prompt a fonctionner et renvoie des valeurs correctes
	{
		// Utilisation des valeurs du prompt pour lancer l'algorithme
		ParamsCurve params;
		
		params.lcfct = prompt.getCurve();
		
		params.nbPoints = prompt.getNbPoints();
		
		params.pX = prompt.getPointX();
		params.pY = prompt.getPointY();
		
		params.paramA = prompt.getPA();
		params.paramB = prompt.getPB();
		params.paramC = prompt.getPC();
		params.paramD = prompt.getPD();
		
		params.iWidth = prompt.getImageW();
		params.iHeight = prompt.getImageH();
		
		params.zWidth = prompt.getZoneW();
		params.zHeight = prompt.getZoneH();
		params.zX = prompt.getZoneX();
		params.zY = prompt.getZoneY();
		
#ifndef QT_NO_DEBUG
		std::cout << "Generator: " << params.lcfct << std::endl;

		std::cout << "Points (nb, X, Y)" << params.nbPoints << "; " << params.pX << "; " << params.pY << std::endl;
		
		std::cout << "Image(" << params.iWidth << ";" << params.iHeight << ")" << std::endl;
		std::cout << "Zone(" << params.zX << ";" << params.zY << ";" << params.zWidth << ";" << params.zHeight << ")" << std::endl << std::endl;
#endif

		{
			// Construction d'une table par défault ... en niveau de gris
			QVector<QRgb> table;
			QTime* timeStamp = new QTime();
			
			for ( unsigned int i = 0 ; i < 256 ; i++ )
			{
				table.push_back(qRgb(i,i,i));
			}
			
			Curve* c = new Curve(params, table, this);
			
			curveThreads.push_back(c);
			timesCurve.push_back(timeStamp);
			connect(c, SIGNAL(finished()), this, SLOT(deleteThreadCurve()));
			
			timeStamp->start();	// Chrono démarré
			c->start();
		}
	}

#ifndef QT_NO_DEBUG
	std::cout << "\tSLOT: startCurve()" << std::endl;
#endif
}

void MainWindow :: settingsBox(void)
{
#ifndef QT_NO_DEBUG
	std::cout << "\tSLOT: settingsBox()" << std::endl;
#endif	
	PromptSettings ps(this);
	ps.exec();
}

void MainWindow :: gammaCorrection(void)
{
#ifndef QT_NO_DEBUG
	std::cout << "\tSLOT: gammaCorrection()" << std::endl;
#endif	

	GammaDialog gd(this);
	int result;
	WidgetImage* wi = (WidgetImage*)tabImage->widget(tabImage->currentIndex());
	
	connect(&gd, SIGNAL(valueChanged(double)), wi , SLOT(gammaChanged(double)));
	result = gd.exec();
	
	if ( result == 1 )
	{
		wi->needRestore();
	}
	else
	{
		wi->needSave();
	}
	
	disconnect(&gd, SIGNAL(valueChanged(double)), wi , SLOT(gammaChanged(double)));
}

void MainWindow :: changeColours(void)
{
#ifndef QT_NO_DEBUG
	std::cout << "\tSLOT: changeColours()" << std::endl;
#endif	

	int result;
	WidgetImage* wi = (WidgetImage*)tabImage->widget(tabImage->currentIndex());
	ColoursDialog cd(wi->getColourTable(), this);
	
	connect(&cd, SIGNAL(coloursTableChanged(QVector<QRgb>)), wi , SLOT(setColourTable(QVector<QRgb>)));
	result = cd.exec();
	
	wi->setColourTable(cd.getColourTable());
	disconnect(&cd, SIGNAL(coloursTableChanged(QVector<QRgb>)), wi , SLOT(setColourTable(QVector<QRgb>)));
}

void MainWindow :: deleteThreadGeneration(void)
{
	unsigned int counter = 0;	// Le compteur permet de savoir qu'elle est la position l'élément actuel ... donc je récupère le bon élément aussi dans les timers
	
#ifndef QT_NO_DEBUG
	std::cout << "\tSLOT: deleteThreadGeneration()" << std::endl;
#endif	
	
	// Parcours de toute la liste à la recherche du thread à retirer.
	for ( QVector< Generation* >::iterator it_generation = generationThreads.begin() ; it_generation != generationThreads.end() ; )
	{
		if ( (*it_generation)->isFinished() )
		{
			WidgetImage* wi = new WidgetImage((*it_generation)->getFractale(), (*it_generation)->getTable(), (*it_generation)->getParams(), this);
			int newIndex = 0;
			
			newIndex = tabImage->addTab(wi, (*it_generation)->getTitle());
			tabImage->setCurrentIndex(newIndex);
			
			connect(wi, SIGNAL(zoom(QPoint, QPoint, Params, QVector<QRgb>)), this, SLOT(renderZoom(QPoint, QPoint, Params, QVector<QRgb>)));
			connect(wi, SIGNAL(unzoom(Params, QVector<QRgb>)), this, SLOT(renderUnZoom(Params, QVector<QRgb>)));
			
			// Ici nous pouvons activer le menu car nous avons enfin un onglet
			this->activateImageMenu();
			
			// On affiche le temps
			//std::cout << "
			qDebug("Generation time: %d ms", timesGeneration[counter]->elapsed());
			
			// On supprime aussi le QTime
			delete timesGeneration[counter];
			timesGeneration.remove(counter);
			delete (*it_generation);
			
			it_generation = generationThreads.erase(it_generation);
		}
		else
		{
			it_generation++;
		}
		
		counter++;
	}
}

void MainWindow :: deleteThreadCurve(void)
{
	unsigned int counter = 0;
	
#ifndef QT_NO_DEBUG
	std::cout << "\tSLOT: deleteThreadCurve()" << std::endl;
#endif	
	
	// Parcours de toute la liste à la recherche du thread à retirer.
	for ( QVector< Curve* >::iterator it_curves = curveThreads.begin() ; it_curves != curveThreads.end() ; )
	{
		if ( (*it_curves)->isFinished() )
		{
			WidgetImage* wi = new WidgetImage((*it_curves)->getFractale(), (*it_curves)->getTable(), (*it_curves)->getParams(), this);
			int newIndex = 0;
			
			newIndex = tabImage->addTab(wi, (*it_curves)->getTitle());
			tabImage->setCurrentIndex(newIndex);
			
			connect(wi, SIGNAL(zoom(QPoint, QPoint, ParamsCurve, QVector<QRgb>)), this, SLOT(renderZoom(QPoint, QPoint, ParamsCurve, QVector<QRgb>)));
			connect(wi, SIGNAL(unzoom(ParamsCurve, QVector<QRgb>)), this, SLOT(renderUnZoom(ParamsCurve, QVector<QRgb>)));
			
			// Ici nous pouvons activer le menu car nous avons enfin un onglet
			this->activateImageMenu();
			
			// On affiche le temps
			//std::cout << "
			qDebug("Generation time: %d ms", timesCurve[counter]->elapsed());
			
			// On supprime aussi le QTime
			delete timesCurve[counter];
			timesCurve.remove(counter);
			
			delete (*it_curves);
			
			it_curves = curveThreads.erase(it_curves);
		}
		else
		{
			it_curves++;
		}
		
		counter++;
	}
}

void MainWindow :: closeTab(int index)
{
#ifndef QT_NO_DEBUG
	std::cout << "\tSLOT: closeTab(" << index << ")" << std::endl;
#endif

	WidgetImage* wi = reinterpret_cast<WidgetImage*>(tabImage->widget(index));

	tabImage->removeTab(index);
	
	disconnect(wi, SIGNAL(zoom(QPoint, QPoint, Params, QVector<QRgb>)), this, SLOT(renderZoom(QPoint, QPoint, Params, QVector<QRgb>)));
	
	delete wi;
	
	if ( tabImage->count() == 0 ) // Car nous n'avons plus d'image à traiter
	{
		desactivateImageMenu();
	}
}

void MainWindow :: renderZoom(QPoint start, QPoint end, Params params, QVector<QRgb> table)
{
	// Nous calculons les nouveaux paramètres
	Params newParams = params;
	QTime* timeStamp = new QTime();
	
	// Nous mettons le start au début du rectangle
	if ( start.x() > end.x() )
	{
		int tmpX = start.x();
		start.setX(end.x());
		end.setX(tmpX);
	}
	
	if ( start.y() > end.y() )
	{
		int tmpY = start.y();
		start.setY(end.y());
		end.setY(tmpY);
	}
	
	newParams.zX =  ( ( start.x() / static_cast<double>(params.iWidth - 1)) * params.zWidth) + params.zX;
	newParams.zY =  ( ( start.y() / static_cast<double>(params.iHeight - 1)) * params.zHeight) + params.zY;
	newParams.zWidth =  (( end.x() - start.x() ) / static_cast<double>(params.iWidth - 1)) * params.zWidth ;
	newParams.zHeight =  (( end.y() - start.y() ) / static_cast<double>(params.iHeight - 1)) * params.zHeight ;
	
#ifndef QT_NO_DEBUG
		std::cout << "Start:(" << start.x() << ";" << start.y() << ")" << std::endl;
		std::cout << "End:(" << end.x() << ";" << end.y() << ")" << std::endl << std::endl;
#endif
	
#ifndef QT_NO_DEBUG
		std::cout << "Initializor: " << params.lifct << std::endl;
		std::cout << "Generator: " << params.lgfct << std::endl;
		std::cout << "Accumulator: " << params.lafct << std::endl << std::endl;
		
		std::cout << "Image(" << params.iWidth << ";" << params.iHeight << ")" << std::endl;
		std::cout << "Zone(" << params.zX << ";" << params.zY << ";" << params.zWidth << ";" << params.zHeight << ")" << std::endl << std::endl;
#endif

#ifndef QT_NO_DEBUG
		std::cout << "Initializor: " << newParams.lifct << std::endl;
		std::cout << "Generator: " << newParams.lgfct << std::endl;
		std::cout << "Accumulator: " << newParams.lafct << std::endl << std::endl;
		
		std::cout << "Image(" << newParams.iWidth << ";" << newParams.iHeight << ")" << std::endl;
		std::cout << "Zone(" << newParams.zX << ";" << newParams.zY << ";" << newParams.zWidth << ";" << newParams.zHeight << ")" << std::endl << std::endl;
#endif
	
	{
		Generation* g = new Generation(newParams, table, this);
				
		generationThreads.push_back(g);
		timesGeneration.push_back(timeStamp);
		connect(g, SIGNAL(finished()), this, SLOT(deleteThreadGeneration()));
				
		timeStamp->start();
		g->start();
	}
}

void MainWindow :: renderZoom(QPoint start, QPoint end, ParamsCurve params, QVector<QRgb> table)
{
	// Nous calculons les nouveaux paramètres
	ParamsCurve newParams = params;
	QTime* timeStamp = new QTime();
	
	// Nous mettons le start au début du rectangle
	if ( start.x() > end.x() )
	{
		int tmpX = start.x();
		start.setX(end.x());
		end.setX(tmpX);
	}
	
	if ( start.y() > end.y() )
	{
		int tmpY = start.y();
		start.setY(end.y());
		end.setY(tmpY);
	}
	
	newParams.zX =  ( ( start.x() / static_cast<double>(params.iWidth - 1)) * params.zWidth) + params.zX;
	newParams.zY =  ( ( start.y() / static_cast<double>(params.iHeight - 1)) * params.zHeight) + params.zY;
	newParams.zWidth =  (( end.x() - start.x() ) / static_cast<double>(params.iWidth - 1)) * params.zWidth ;
	newParams.zHeight =  (( end.y() - start.y() ) / static_cast<double>(params.iHeight - 1)) * params.zHeight ;
	
	{
		Curve* c = new Curve(newParams, table, this);
				
		curveThreads.push_back(c);
		timesCurve.push_back(timeStamp);
		connect(c, SIGNAL(finished()), this, SLOT(deleteThreadCurve()));
				
		timeStamp->start();
		c->start();
	}
	
#ifndef QT_NO_DEBUG
		std::cout << " Curve thread zoom" << std::endl << std::endl;
#endif
}


void MainWindow :: renderUnZoom(Params params, QVector<QRgb> table)
{
	// Nous calculons les nouveaux paramètres
	Params newParams = params;
	QTime* timeStamp = new QTime();
	
	// Le dezoom ne demande pas grand chose ... on ne fait qu'un dezoom qui représentera deux fois plus de choses
	newParams.zX =  params.zX * 2;
	newParams.zY =  params.zY * 2;;
	newParams.zWidth =  params.zWidth * 2;
	newParams.zHeight = params.zHeight * 2;
	
	{
		Generation* g = new Generation(newParams, table, this);
				
		generationThreads.push_back(g);
		timesGeneration.push_back(timeStamp);
		connect(g, SIGNAL(finished()), this, SLOT(deleteThreadGeneration()));
			
		timeStamp->start();	
		g->start();
	}
}

void MainWindow :: renderUnZoom(ParamsCurve params, QVector<QRgb> table)
{
	// Nous calculons les nouveaux paramètres
	ParamsCurve newParams = params;
	QTime* timeStamp = new QTime();
	
	// Le dezoom ne demande pas grand chose ... on ne fait qu'un dezoom qui représentera deux fois plus de choses
	newParams.zX =  params.zX * 2;
	newParams.zY =  params.zY * 2;;
	newParams.zWidth =  params.zWidth * 2;
	newParams.zHeight = params.zHeight * 2;
	
	{
		Curve* c = new Curve(newParams, table, this);
				
		curveThreads.push_back(c);
		timesCurve.push_back(timeStamp);
		connect(c, SIGNAL(finished()), this, SLOT(deleteThreadCurve()));
				
		timeStamp->start();
		c->start();
	}
	
#ifndef QT_NO_DEBUG
		std::cout << " Curve thread zoom" << std::endl << std::endl;
#endif
}
