#ifndef __MAINWINDOW_HPP__
#define __MAINWINDOW_HPP__

/*
# Copyright (c) 2010 Alexandre LAURENT
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
*/

#include <QMainWindow>

#include <QString>
#include <QList>

#include <QVector>
#include <QColor>

#include "global.hpp"

class QWidget;
class QTabWidget;
class QAction;
class QTime;

class Generation;
class Curve;

/*! \class MainWindow
 * \brief La fenêtre principale pour l'application
 */
class MainWindow : public QMainWindow
{
	Q_OBJECT
	
	private:
	
		// Variables
		
		QList < QAction* > actions;	/*!< Liste des actions ( évite des fuites )*/
		
		QMenu* pFileMenu;	/*!< Menu pour l'entrée "Fichier ( File )*/
		QMenu* pGenerationMenu;	/*!< Menu pour l'entrée "Generation ( Generation )*/
		QMenu* pImageMenu;	/*!< Menu pour l'entrée "Image ( Picture )*/
		
		QAction* pSaveAction;	/*!< Action correspondant au menu 'Save' */
		QAction* pGammaCorrectionAction;	/*!< Action correspondant au menu 'Gamma correction' */
		QAction* pRandomColoursAction;	/*!< Action correspondant au menu 'Random colours' */
		
		
		QVector < Generation* > generationThreads;	/*!< Liste de threads 'Génération' ( évite les fuites ) */
		QVector < Curve* > curveThreads;	/*!< Liste de threads 'Curve' ( évite les fuites ) */
		
		QVector < QTime* > timesGeneration;		/*!< Liste des temps de départ des generations ( marche comme un chrono ) */
		QVector < QTime* > timesCurve;			/*!< Liste des temps de départ des courbes ( marche comme un chrono ) */
		
		QTabWidget* tabImage;					/*!< Les images sont affiché dans un système d'onglet */
		
		// Functions
	
		/*!
		 * \brief - Crée une entrée dans un menu
		 * 		  - Assigne un raccourci
	 	 *	  	  - Assigne une description
	 	 * 		  - Défini si le menu est activé ou pas
		 * 		  - Assigne une fonction qui va être appelé lors d'un clique sur la nouvelle entrée
		 * N'intègre pas les icones
		 * 
		 * \param Le menu où l'on ajoute l'entrée
		 * \param Le nom de l'entrée
		 * \param La séquence de touche pour le raccourci
		 * \param La description affiché lors du passage sur l'entrée
		 * \param Le pointeur sur l'objet qui doit être utilisé lors du clique sur l'entrée
		 * \param La fonction appelé lorsque un clic est effectué sur l'entrée
		 * \param Définit si le menu est activé ou pas
		 * \param Définit si le menu est checkable
		 * \return QAction* Pointeur sur le QAction nouvellement crée
	 	 */
		QAction* addAction(	QMenu* parentMenu,
							const QString& entryName,
							const QString& keySequence,
							const QString& statusTipMessage,
							const QObject* object,
							const char* slot,
							const bool enabled = true,
							const bool checkable = false);
		
		/*!
		 * \brief Crée les menus de la fenêtre
		 */
		void createMenus(void);
		
		/*!
		 * \brief Active les menus Image ( que lors de la présence d'un onglet )
		 */
		void activateImageMenu(void);
		
		/*!
		 * \brief Désactive les menus Image ( que lors de la présence d'un onglet )
		 */
		void desactivateImageMenu(void);
		
	public slots:
	
		/*!
		 * \brief Appelée lors d'un clic sur le menu Save. Sauvegarde la fractale dans un fichier image.
		 */
		void savePicture(void);
		
		/*!
		 * \brief Appelée lors d'un clic sur le menu Generation. Démarrage de l'algorithme.
		 */
		void startGeneration(void);
		
		/*!
		 * \brief Appelée lors d'un clic sur le menu Curve. Démarrage de l'algorithme.
		 */
		void startCurve(void);
		
		/*!
		 * \brief Appelée lors d'un clic sur le menu Preferences. Démarrage d'une boite de dialogue pour changer les paramètres.
		 */
		void settingsBox(void);
		
		/*!
		 * \brief Appelée lors d'un clic sur le menu Gamme correction. Démarrage d'une boite de dialogue pour changer le gamma de l'image courrante.
		 */
		void gammaCorrection(void);
		
		/*!
		 * \brief Appelée lors d'un clic sur le menu Random colours.
		 */
		void changeColours(void);
		
		/*!
		 * \brief Appelée lorsqu'un thread 'Generation' se termine. Va détruire le thread et l'enlever de la liste
		 */
		void deleteThreadGeneration(void);
		
		/*!
		 * \brief Appelée lorsqu'un thread 'Curve' se termine. Va détruire le thread et l'enlever de la liste
		 */
		void deleteThreadCurve(void);
		
		/*!
		 * \brief Appelée lorsque l'utilisateur ferme une tabulation
		 */
		void closeTab(int index);
		
		/*!
		 * \brief Appelée lorsque l'utilisateur veut faire un zoom
		 */
		void renderZoom(QPoint start, QPoint end, Params params, QVector<QRgb> table);
		
		/*!
		 * \brief Appelée lorsque l'utilisateur veut faire un zoom (Curve version)
		 */
		void renderZoom(QPoint start, QPoint end, ParamsCurve params, QVector<QRgb> table);
		
		/*!
		 * \brief Appelée lorsque l'utilisateur veut faire un dézoom
		 */
		void renderUnZoom(Params params, QVector<QRgb> table);
		
		/*!
		 * \brief Appelée lorsque l'utilisateur veut faire un dézoom (Curve version)
		 */
		void renderUnZoom(ParamsCurve params, QVector<QRgb> table);
	
	public:
	
		/*!
		 *  \brief Constructeur
		 *
		 *  \param windowName : Le nom que la fenêtre va porter.
		 *  \param width : La taille en largeur de la fenêtre. ( Celle ci va être surement redimensionner par l'utilisateur )
		 *  \param height : La taille en hauteur de la fenêtre.
		 *  \param parent : La fenêtre parente.
		 */
		MainWindow(const QString& windowName, const unsigned int width, const unsigned int height, QWidget* parent=0);
		
		/*!
		 * \brief Destructeur
		 */
		~MainWindow(void);
};

#endif
