/*
# Copyright (c) 2010 Alexandre LAURENT
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
*/

#include "PixmapWidget.hpp"

#include <QPaintEvent>
#include <QPainter>
#include <QMouseEvent>
#include <QKeyEvent>

#ifndef QT_NO_DEBUG
	#include <iostream>
#endif

PixmapWidget :: PixmapWidget(QImage* image, QWidget* parent/*=0*/)
	:QWidget(parent),
	img(image),
	clicked(false)
{
	// Defini la taille minimum pour aider le QScrollArea à savoir quand on a besoin des scroll bar
	this->setMinimumSize( img->width(), img->height() );
	
	// Cette ligne permet d'avoir le focus lors d'un clic sur le widget.
	// Comme nous aurons le focus grace au clic, nous allons avoir les evenements du clavier sur ce widget, donc nous pourrons annulé un zoom
	this->setFocusPolicy ( Qt::ClickFocus );
	
#ifndef QT_NO_DEBUG
	std::cout << "PixmapWidget created" << std::endl;
#endif
}

PixmapWidget :: ~PixmapWidget(void)
{
	
#ifndef QT_NO_DEBUG
	std::cout << "PixmapWidget deleted" << std::endl;
#endif
}

	// Gestion de la souris
void PixmapWidget ::mousePressEvent(QMouseEvent* event)
{
#ifndef QT_NO_DEBUG
	std::cout << "\tPixmapWidget:: Mouse pressed (" << event->pos().x() << ";" << event->pos().y() << ")" << std::endl;
#endif	

	if ( event->button() == Qt::LeftButton )
	{
		startMousePosition = event->pos();	// Position relative au widget
		actualMousePosition = event->pos();
		clicked = true;
	
		repaint();
	}
	else if ( event->button() == Qt::RightButton )	// Bouton droit, on dézoome
	{
		emit unzoom();
	}
}

void PixmapWidget :: mouseMoveEvent(QMouseEvent* event)
{
	if ( clicked )	// Commençons par limité les difficulté.
	{
		#ifndef QT_NO_DEBUG
			std::cout << "\tPixmapWidget:: Mouse pressed" << std::endl;
		#endif	

		actualMousePosition = event->pos();
	
		repaint();
	}
}

void PixmapWidget :: mouseReleaseEvent(QMouseEvent* event)
{
#ifndef QT_NO_DEBUG
	std::cout << "\tPixmapWidget:: Mouse released (" << event->pos().x() << ";" << event->pos().y() << ")" << std::endl;
#endif	

	if ( clicked == true )
	{
		clicked = false;
	
		// Envoie du message
		if ( startMousePosition != event->pos())
		{
			emit zoom(startMousePosition, event->pos());
		}
	
		repaint();
	}
}

void PixmapWidget :: keyReleaseEvent(QKeyEvent* event)
{
#ifndef QT_NO_DEBUG
	std::cout << "\tPixmapWidget:: Key released (" << std::hex << event->key() << std::dec << ")" << std::endl;
#endif

	if ( event->key() == Qt::Key_Escape )
	{
		clicked = false;
	
		repaint();
	}
}

void PixmapWidget :: paintEvent(QPaintEvent* /*event*/)
{
	QPainter painter(this);
	
	// J'imagine que la génération de la pixmap comme ceci est loin d'être une bonne chose
	// Mais, comme ceci, je sais que la QPixmap sera détruit au bon moment
	// Il doit y avoir une meilleure methode... mais je me suis un peu perdu :s
	painter.drawPixmap(0, 0, QPixmap::fromImage(*img));
	
	if ( clicked )
	{
		painter.setPen(Qt::white);
		painter.drawRect(startMousePosition.x(), startMousePosition.y(), actualMousePosition.x() - startMousePosition.x(), actualMousePosition.y() - startMousePosition.y());
	}
}
