#ifndef __PROMPTCURVEDIALOG_HPP__
#define __PROMPTCURVEDIALOG_HPP__

/*
# Copyright (c) 2010 Alexandre LAURENT
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
*/

#include <QDialog>

#include "../global.hpp"

class QWidget;
class QGridLayout;
class QHBoxLayout;
class QGroupBox;
class QComboBox;
class QSpinBox;
class QDoubleSpinBox;
class QPushButton;
class QLabel;

/*! \class PromptDialog
 * \brief Surcharge du QDialog afin d'avoir une boite de paramètrages pour la generation de la fractale
 */
class PromptCurveDialog : public QDialog
{
	Q_OBJECT
	
	private:
	
		// Interface
		//QWidget* pCentralArea;
		QGridLayout* pMainGrid;		/*!< Grille pour l'agencement de la fenêtre */
		
		QGroupBox* pFunctionsBox;	/*!< Boite en relation avec les fonctions */
		QGroupBox* pZoneBox;		/*!< Boite en relation avec la zone */
		
		QGridLayout* pFunctionsGrid;	/*!< Grille pour l'agencement des éléments relatifs aux fonctions */
		QHBoxLayout* pZoneGrid;			/*!< Grille pour l'agencement des éléments relatifs à la zone */
		QHBoxLayout* pButtonsGrid;		/*!< Grille pour l'agencement des bouttons */
		
		// Listes
										// Ceci va définir les fonctions à utiliser pour la génération de la fractale
		QLabel* pLabelCurvesList;	QComboBox* pCurvesList;		/*!< Bouton et label pour la liste des courbes */
		
		// Zone
		QGroupBox* pPointBox;				/*!< Groupe regroupant les élements relatifs aux points */
		QGroupBox* pParamBox;				/*!< Groupe regroupant les élements relatifs aux paramètres */
		QGroupBox* pZoneSizeBox;			/*!< Groupe regroupant les élements relatifs à la taille */
		QGroupBox* pZoneGeneratorBox;		/*!< Groupe regroupant les élements relatifs aux generateurs */
		
		QGridLayout* pPointGrid;			/*!< Grille regroupant les élements relatifs aux points */
		QGridLayout* pParamGrid;			/*!< Grille regroupant les élements relatifs aux paramètres */
		QGridLayout* pZoneSizeGrid;			/*!< Grille regroupant les élements relatifs à la taille */
		QGridLayout* pZoneGeneratorGrid;	/*!< Grille regroupant les élements relatifs aux générateurs */
		
		// Point
		QLabel* pLabelNbPoints; QSpinBox* pNbPoints;	/*!< Nombre de point */
		QLabel* pLabelPointX; QDoubleSpinBox* pPointX;	/*!< Coordonnée X du point d'initialisation */
		QLabel* pLabelPointY; QDoubleSpinBox* pPointY;	/*!< Coordonnée Y du point d'initialisation */
		
		// Paramètres
		QLabel* pLabelPA; QDoubleSpinBox* pPA;		/*!< Paramètre 1 */
		QLabel* pLabelPB; QDoubleSpinBox* pPB;		/*!< Paramètre 2 */
		QLabel* pLabelPC; QDoubleSpinBox* pPC;		/*!< Paramètre 3 */
		QLabel* pLabelPD; QDoubleSpinBox* pPD;		/*!< Paramètre 4 */
		
		// Image
		QLabel* pLabelWidth; QSpinBox* pWidth;	/*!< Largeur de l'image */
		QLabel* pLabelHeight; QSpinBox* pHeight;	/*!< Hauteur de l'image */
		
		//Fractale
						 // Ceci va contenir les paramètres pour la zone de la fractale
		QLabel* pLabelX; QDoubleSpinBox* pX;	/*!< Valeur réelle la plus petite de la zone complexe */
		QLabel* pLabelY; QDoubleSpinBox* pY;	/*!< Valeur complexe la plus petite de la zone complexe */
		QLabel* pLabelW; QDoubleSpinBox* pW;	/*!< Largeur de la zone complexe */
		QLabel* pLabelH; QDoubleSpinBox* pH;	/*!< Hauteur de la zone complexe */
		
		// Bouttons
		QPushButton* pCancelButton;				/*! Bouton pour annuler */
		QPushButton* pDefaultButton;			/*! Bouton pour remettre les valeurs par default */
		QPushButton* pStartButton;				/*! Bouton pour démarrer la generation */
		
		// Creation de l'interface
		/*!
		 * \brief Crée les bouttons
		 */
		void createButtons(void);
		
		/*!
		 * \brief Crée les listes de fonctions
		 */
		void createListsFunctions(void);
		
		/*!
		 * \brief Crée les spin boxes
		 */
		void createSpinBoxes(void);
		
	public slots:
	
		/*!
		 * \brief Appelée lors d'un clic sur le bouton Start. Démarrage de l'algorithme de génération des fractales.
		 */
		void startGeneration(void);
		
		/*!
		 * \brief Appelée lors d'un clic sur le menu Cancel. Fermeture de la boite.
		 */
		void cancelSettings(void);
		
		/*!
		 * \brief Appelée lors de l'appuie sur la touche 'Echap'. Fermeture de la boite.
		 */
		void reject(void);
		
		/*!
		 * \brief Appelée lorsque la fonction selectionnée est modifiée. Va regarder si les valeurs de la zone ne sont pas défini, et va appelée applyDefaultValues dans ce cas.
		 */
		void functionSelectedChanged(int index);
		
		/*!
		 * \brief Appelée lors d'un clic sur le bouton 'Defaults'. Les valeurs des propriétés de la zone sont remises à zero.
		 */
		void applyDefaultValues(void);
	
	public:
	
		/*!
		 * \brief Constructeur
		 * \param parent : La fenêtre parente
		 */
		PromptCurveDialog(QWidget* parent = 0);
		
		/*!
		 * \brief Destructeur
		 */
		~PromptCurveDialog(void);
		
		/*!
		 * \brief Getteur pour la fonction de courbe
		 * \return La fonction de courbe
		 */
		ListCurveFCT getCurve(void)const;
		
		/*!
		 * \brief - Get le nombre de points à dessiner
		 * \return unsigned int le nombre de point à dessiner
	 	 */
		unsigned int getNbPoints(void)const;
		
		/*!
		 * \brief - Get la position sur l'axe des Y du point de départ
		 * \return double la position du point sur l'axe des Y
	 	 */
		double getPointX(void)const;
		
		/*!
		 * \brief - Get la position sur l'axe des Y du point de départ
		 * \return double la position du point sur l'axe des Y
	 	 */
		double getPointY(void)const;
		
		/*!
		 * \brief - Get le paramètre A
		 * \return double le paramètre A
	 	 */
		double getPA(void)const;
		
		/*!
		 * \brief - Get le paramètre B
		 * \return double le paramètre B
	 	 */
		double getPB(void)const;
		
		/*!
		 * \brief - Get le paramètre C
		 * \return double le paramètre C
	 	 */
		double getPC(void)const;
		
		/*!
		 * \brief - Get le paramètre D
		 * \return double le paramètre D
	 	 */
		double getPD(void)const;

		/*!
		 * \brief - Get la largeur de l'image à generer
		 * \return unsigned int la largeur de l'image à generer
	 	 */
		unsigned int getImageW(void)const;
		
		/*!
		 * \brief - Get la hauteur de l'image à generer
		 * \return unsigned int la hauteur de l'image à generer
	 	 */
		unsigned int getImageH(void)const;
		
		/*!
		 * \brief - Get la valeur réelle la plus petite de la zone complexe 
		 * \return unsigned int la valeur réelle la plus petite de la zone complexe 
	 	 */
		double getZoneX(void)const;
		
		/*!
		 * \brief - Get la valeur complexe la plus petite de la zone complexe
		 * \return unsigned int la valeur complexe la plus petite de la zone complexe
	 	 */
		double getZoneY(void)const;
		
		/*!
		 * \brief - Get la largeur de la zone complexe
		 * \return unsigned int la largeur de la zone complexe
	 	 */
		double getZoneW(void)const;
		
		/*!
		 * \brief - Get la hauteur de la zone complexe
		 * \return unsigned int la hauteur de la zone complexe
	 	 */
		double getZoneH(void)const;
};

#endif
