#include "WidgetImage.hpp"

/*
# Copyright (c) 2010 Alexandre LAURENT
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
*/

#include <QScrollArea>
#include <QPaintEvent>
#include <QPainter>

#include <PixmapWidget.hpp>

#include <cmath>

#ifndef QT_NO_DEBUG
	#include <iostream>
#endif

WidgetImage :: WidgetImage(const unsigned char* const img, const QVector<QRgb>& table, const Params params, QWidget* parent /*= 0*/)
	:QScrollArea(parent), params(params),gamma(1),saveGamma(1)	
{
	this->img = QImage(params.iWidth, params.iHeight, QImage::Format_Indexed8);
	
	// Copie de l'image
	{
		unsigned char* pPixels = this->img.bits();
		
		for ( unsigned int i = 0 ; i < params.iWidth * params.iHeight ; ++i )
		{
			pPixels[i] = img[i];
		}
	}
	this->img.setColorTable(table);
	
	this->saveImg = QImage(this->img);
	
	//pixmap = QPixmap::fromImage(this->img);
	
	pw = new PixmapWidget(&this->img, this);
	
	this->setAlignment(Qt::AlignCenter);
	
	this->setWidget(pw);
	
	connect(pw, SIGNAL(zoom(QPoint, QPoint)), this, SLOT(pushSignalZoom(QPoint, QPoint)));
	connect(pw, SIGNAL(unzoom(void)), this, SLOT(pushSignalUnZoom(void)));
	
	gamma = 0;
	
#ifndef QT_NO_DEBUG
	std::cout << "WidgetImage created" << std::endl;
#endif
}

WidgetImage :: WidgetImage(const unsigned char* const img, const QVector<QRgb>& table, const ParamsCurve params, QWidget* parent /*= 0*/)
	:QScrollArea(parent), paramsCurve(params),gamma(1),saveGamma(1)	
{
	this->img = QImage(params.iWidth, params.iHeight, QImage::Format_Indexed8);
	
	// Copie de l'image
	{
		unsigned char* pPixels = this->img.bits();
		
		for ( unsigned int i = 0 ; i < params.iWidth * params.iHeight ; ++i )
		{
			pPixels[i] = img[i];
		}
	}
	this->img.setColorTable(table);
	
	this->saveImg = QImage(this->img);
	
	//pixmap = QPixmap::fromImage(this->img);
	
	pw = new PixmapWidget(&this->img, this);
	
	this->setAlignment(Qt::AlignCenter);
	
	this->setWidget(pw);
	
	connect(pw, SIGNAL(zoom(QPoint, QPoint)), this, SLOT(pushSignalZoomCurve(QPoint, QPoint)));
	connect(pw, SIGNAL(unzoom(void)), this, SLOT(pushSignalUnZoomCurve(void)));
	
	gamma = 0;
	
#ifndef QT_NO_DEBUG
	std::cout << "WidgetImage created (curved)" << std::endl;
#endif
}

WidgetImage :: ~WidgetImage(void)
{
	disconnect(pw, SIGNAL(zoom(QPoint, QPoint)), this, SLOT(pushSignalZoom(QPoint, QPoint)));
	disconnect(pw, SIGNAL(unzoom()), this, SLOT(pushSignalUnZoom()));
	disconnect(pw, SIGNAL(zoom(QPoint, QPoint)), this, SLOT(pushSignalZoomCurve(QPoint, QPoint)));
	disconnect(pw, SIGNAL(unzoom()), this, SLOT(pushSignalUnZoomCurve()));
	
	delete pw;
	
#ifndef QT_NO_DEBUG
	std::cout << "WidgetImage deleted" << std::endl;
#endif	
}

void WidgetImage :: pushSignalZoom(QPoint start, QPoint end)
{
	emit zoom(start, end, params, img.colorTable());
}

void WidgetImage :: pushSignalZoomCurve(QPoint start, QPoint end)
{
	emit zoom(start, end, paramsCurve, img.colorTable());
}

void WidgetImage :: pushSignalUnZoom()
{
	emit unzoom(params, img.colorTable());
}

void WidgetImage :: pushSignalUnZoomCurve(void)
{
	emit unzoom(paramsCurve, img.colorTable());
}

void WidgetImage :: gammaChanged(double value)
{
#ifndef QT_NO_DEBUG
	std::cout << "\tSLOT: WidgetImage::valueChanged:" << value << std::endl;
#endif

	// Calculs comme le faisait Yan
	{
		// Je gère une image sur 8 bits
		unsigned char* data = img.bits();
		unsigned char max = 0;
		
		// Recherche du maximum
		for ( unsigned int i = 0 ; i < (unsigned)img.width() * img.height() ; i++ )
		{
			if ( max < data[i] )
			{
				max = data[i];
			}
		}
		
		{
			unsigned int tmpColor = 0;
					
			for ( unsigned int i = 0 ; i < (unsigned)img.width() * img.height() ; i++ )
			{
				tmpColor = 255.0 * std::pow(static_cast<double>(data[i]) / max, value) + 0.4;
				
				if ( tmpColor > 255 )
				{
					data[i] = 255;
				}			
				else
				{
					data[i] = static_cast<unsigned char>(tmpColor);
				}
			}
		}
		
	}

	// Mis à jour de l'image
	//pixmap = QPixmap::fromImage(img);
	pw->repaint();
	//dis->setPixmap(pixmap);

	// Mis à jour de la valeur lorcal
	this->gamma = value;
}

void WidgetImage :: needRestore(void)
{
#ifndef QT_NO_DEBUG
	std::cout << "\tSLOT: WidgetImage::needRestore" << std::endl;
#endif

	img = saveImg;
	gamma = saveGamma;
	
	//pixmap = QPixmap::fromImage(img);
	pw->repaint();
}

void WidgetImage :: needSave(void)
{
#ifndef QT_NO_DEBUG
	std::cout << "\tSLOT: WidgetImage::needSave(" << std::endl;
#endif

	saveImg = img;
	saveGamma = gamma;
	
	//pixmap = QPixmap::fromImage(img);
	pw->repaint();
}

void WidgetImage :: setColourTable(QVector<QRgb> newTable)
{
#ifndef QT_NO_DEBUG
	std::cout << "\tWidgetImage :: setColourTable" << std::endl;
#endif	

	img.setColorTable(newTable);
	
	//pixmap = QPixmap::fromImage(img);
	pw->repaint();
	
	saveImg = img;
}
