#ifndef __WIDGETIMAGE_HPP__
#define __WIDGETIMAGE_HPP__

/*
# Copyright (c) 2010 Alexandre LAURENT
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
*/

#include <QScrollArea>
#include <QImage>
#include <QPixmap>
#include <QLabel>
#include <QPoint>
#include <QVector>
#include <QColor>

#include "global.hpp"

class PixmapWidget;
class QPaintEvent;

/*! \class WidgetImage
 * \brief Surcharge du QWidget afin d'afficher une QImage
 */
class WidgetImage : public QScrollArea
{
	Q_OBJECT
	
	private:
	
		QImage saveImg;				/*!< Ancienne image ( pour annulation ) */
		QImage img;					/*!< Image ( permet la modification rapide des pixels ) */
		//QPixmap pixmap;				/*!< Permet l'affichage rapide de l'image */
		
		Params params;				/*!< Paramètre de la génération ( sauvegarde des paramètre, peuvent être nécessaire pour le zoom ) */
		ParamsCurve paramsCurve;	/*!< Paramètre de la courbe ( sauvegarde des paramètre, peuvent être nécessaire pour le zoom ) */
		
		PixmapWidget* pw;			/*!< Widget qui affiche l'image */
		//QLabel* dis;
		
		double gamma;				/*!< Valeur du gamma */
		double saveGamma;			/*!< Sauvegarde de la valeur du gamma ( pour l'annulation ) */
		
		
	public slots:
	
		/*!
		 * \brief Appelée lorsque l'utilisateur change la valeur du gamma pour l'image courante.
		 */
		void gammaChanged(double value);
		
		/*!
		 * \brief Appelée lorsque le signal du PixmapWidget zoom est reçu
		 */
		void pushSignalZoom(QPoint start, QPoint end);
		
		/*!
		 * \brief Appelée lorsque le signal du PixmapWidget unzoom est reçu
		 */
		void pushSignalUnZoom();
		
		/*!
		 * \brief Appelée lorsque le signal du PixmapWidget zoom est reçu (Curve version)
		 */
		void pushSignalZoomCurve(QPoint start, QPoint end);
		
		/*!
		 * \brief Appelée lorsque le signal du PixmapWidget unzoom est reçu
		 */
		void pushSignalUnZoomCurve();
		
		/*!
		 * \brief Appelée lorsque le signal du ColourWidget est reçu
		 */
		void setColourTable(QVector<QRgb> newTable);
		
	signals:
		/*!
		 * \brief Signal envoyer pour faire un zoom
		 */
		void zoom(QPoint start, QPoint end, Params params, QVector<QRgb> table);
		
		/*!
		 * \brief Signal envoyer pour faire un zoom ( Courbe )
		 */
		void zoom(QPoint start, QPoint end, ParamsCurve params, QVector<QRgb> table);
		
		/*!
		 * \brief Signal envoyer pour faire un dezoom
		 */
		void unzoom(Params params, QVector<QRgb> table);
		
		/*!
		 * \brief Signal envoyer pour faire un dezoom ( Courbe )
		 */
		void unzoom(ParamsCurve params, QVector<QRgb> table);
	
	public:
	
		/*!
		 * \brief Costructeur
		 * \param img : tableau de pixel de l'image
		 * \param table : table de couleur pour l'image
		 * \param params : paramètre de la génération
		 * \param parent : Fenêtre parente
		 */
		WidgetImage(const unsigned char* const img, const QVector<QRgb>& table, const Params params, QWidget* parent = 0);
		
		/*!
		 * \brief Costructeur
		 * \param img : tableau de pixel de l'image
		 * \param table : table de couleur pour l'image
		 * \param params : paramètre de la courbe
		 * \param parent : Fenêtre parente
		 */
		WidgetImage(const unsigned char* const img, const QVector<QRgb>& table, const ParamsCurve params, QWidget* parent = 0);
		
		/*!
		 * \brief Destructeur
		 */
		~WidgetImage(void);
		
		/*!
		 * \brief Getteur pour l'image
		 * \return image
		 */
		const QImage& getImage(void)const { return img; }
		
		/*!
		 * \brief Fonction de restauration de l'ancienne image
		 */
		void needRestore(void);
		
		/*!
		 * \brief Fonction de sauvegarde de la nouvelle image
		 */
		void needSave(void);
		
		/*!
		 * \brief Getteur pour la table de couleur
		 * \return la table des couleurs
		 */
		QVector<QRgb> getColourTable(void)const { return img.colorTable(); };
};

#endif
