/**
 \brief AiGenerator is a generator core
 \author Adrabi Abderrahim
 \date 2010-01-10
 \package AiGenerator
 \page License
    \paragraph <The Artistic License>
    The intent of this document is to state the conditions under which a Package may be copied, such that the Copyright Holder maintains some semblance of artistic control over the development of the package, while giving the users of the package the right to use and distribute the Package in a more-or-less customary fashion, plus the right to make reasonable modifications.

    Definitions:
    "Package" refers to the collection of files distributed by the Copyright Holder, and derivatives of that collection of files created through textual modification.
    "Standard Version" refers to such a Package if it has not been modified, or has been modified in accordance with the wishes of the Copyright Holder.
    "Copyright Holder" is whoever is named in the copyright or copyrights for the package.
    "You" is you, if you're thinking about copying or distributing this Package.
    "Reasonable copying fee" is whatever you can justify on the basis of media cost, duplication charges, time of people involved, and so on. (You will not be required to justify it to the Copyright Holder, but only to the computing community at large as a market that must bear the fee.)
    "Freely Available" means that no fee is charged for the item itself, though there may be fees involved in handling the item. It also means that recipients of the item may redistribute it under the same conditions they received it.

    1. You may make and give away verbatim copies of the source form of the Standard Version of this Package without restriction, provided that you duplicate all of the original copyright notices and associated disclaimers.
    2. You may apply bug fixes, portability fixes and other modifications derived from the Public Domain or from the Copyright Holder. A Package modified in such a way shall still be considered the Standard Version.
    3. You may otherwise modify your copy of this Package in any way, provided that you insert a prominent notice in each changed file stating how and when you changed that file, and provided that you do at least ONE of the following:
        a) place your modifications in the Public Domain or otherwise make them Freely Available, such as by posting said modifications to Usenet or an equivalent medium, or placing the modifications on a major archive site such as ftp.uu.net, or by allowing the Copyright Holder to include your modifications in the Standard Version of the Package.
        b) use the modified Package only within your corporation or organization.
        c) rename any non-standard executables so the names do not conflict with standard executables, which must also be provided, and provide a separate manual page for each non-standard executable that clearly documents how it differs from the Standard Version.
        d) make other distribution arrangements with the Copyright Holder.

    4. You may distribute the programs of this Package in object code or executable form, provided that you do at least ONE of the following:
        a) distribute a Standard Version of the executables and library files, together with instructions (in the manual page or equivalent) on where to get the Standard Version.
        b) accompany the distribution with the machine-readable source of the Package with your modifications.
        c) accompany any non-standard executables with their corresponding Standard Version executables, giving the non-standard executables non-standard names, and clearly documenting the differences in manual pages (or equivalent), together with instructions on where to get the Standard Version.
        d) make other distribution arrangements with the Copyright Holder.

    5. You may charge a reasonable copying fee for any distribution of this Package. You may charge any fee you choose for support of this Package. You may not charge a fee for this Package itself. However, you may distribute this Package in aggregate with other (possibly commercial) programs as part of a larger (possibly commercial) software distribution provided that you do not advertise this Package as a product of your own.
    6. The scripts and library files supplied as input to or produced as output from the programs of this Package do not automatically fall under the copyright of this Package, but belong to whomever generated them, and may be sold commercially, and may be aggregated with this Package.
    7. C or perl subroutines supplied by you and linked into this Package shall not be considered part of this Package.
    8. The name of the Copyright Holder may not be used to endorse or promote products derived from this software without specific prior written permission.
    9. THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.

    The End

 */

#ifndef AIZONE_HPP
#define AIZONE_HPP

#include <QtGui>

namespace AiGenerator
{
    /**
      \brief A custom class for creating a perfect complex zone
      \class AiZone
      \author Adrabi Abderrahim
      \date 2010-01-10
     */
    class AiZone : public QObject
    {
        Q_OBJECT

    public:

        /**
          \brief copy constructor
          \param zone : object reference
         */
        AiZone( const AiZone & zone );

        /**
          \brief optional constructor for zone initialization
          \param parent : object parent ( default = 0 )
          \param x : min X point in complex zone ( default = .0 )
          \param y : min Y point in complex zone ( default = .0 )
          \param zoneWidth : a zone width ( default = .0 )
          \param zoneHeight : a zone height ( default = .0 )
          \param imageWidth : a image width ( default = 0 )
          \param imageHeight : a image height ( default = 0 )
          \param startPoint : a image start point for fractal generation ( default = 0 )
          \param increment : increment next image point by "defined number" ( default = 0 )
          \param iteration : iteration numbers for generation ( default = 0 )
          \param zoom : zoom value ( default = 1. )
          \param viewx : zoom viewx value ( default = 0. )
          \param viewy : zoom viewy value ( default = 0. )
         */
        AiZone( QObject * parent = 0 ,
            double x = .0 , double y = .0 ,
            double zoneWidth = .0 , double zoneHeight = .0 ,
            int imageWidth = 0 , int imageHeight = 0 ,
            int starPoint = 0,int increment = 0 , int iteration = 0,
            double zoom = 1. , double viewx = 0. , double viewy = 0.);

        /**
          \brief get current complex zone image
          \return QImage
         */
        QImage  & image();

        /**
          \brief genaration status if True generation continue, if False generation canceled
         */
        volatile bool status;

        /**
          \brief zone objects serialisation
          \param stream : output "to QDataStream"
         */
        void serialize( QDataStream & stream );

        /**
          \brief zone objects deserialisation
          \param stream : input "from QDataStream"
         */
        void deserialize( QDataStream & stream );

        /**
          \brief opearator "assigne"
          \param rhs : zone source
         */
        AiZone & operator = ( const AiZone &rhs );

    public slots:
        /**
          \brief get a min X in complex zone
          \return double
         */
        double  getX();

        /**
          \brief get a min Y in complex zone
          \return double
         */
        double  getY();

        /**
          \brief get complex zone width
          \return double
         */
        double  getZoneWidth();

        /**
          \brief get complex zone height
          \return double
         */
        double  getZoneHeight();

        /**
          \brief get image width
          \return int
         */
        int     getImageWidth();

        /**
          \brief get image height
          \return int
         */
        int     getImageHeight();

        /**
          \brief get complex zone start point
          \return int
         */
        int     getStartPoint();

        /**
          \brief get complex zone increment point
          \return int
         */
        int     getIncrement();

        /**
          \brief get complex zone itearation
          \return int
         */
        int     getIteration();

        /**
          \brief get pixel at x,y from complex zone image
          \param x : image X
          \param y : image Y
          \return unsigned int
         */
        unsigned int getPixel(int x, int y);

        /**
          \brief set a pixel at x,y in image
          \param x : image X
          \param y : image Y
          \param rgb : rgb color
         */
        void    setPixel(int x, int y, unsigned int rgb);

        /**
          \brief set a pixel at x,y in image
          \param x : image X
          \param y : image Y
          \param red : color red
          \param green : color green
          \param blue : color blue
         */
        void    setPixel(int x, int y, int red, int green, int blue);

        /**
          \brief set pixels per line
          \param line : line index
          \param pix : array
         */
        void    setPixels( int line ,QList< int >  pix );

        /**
          \brief test for stopping all loops
          \return bool
         */
        bool    cancel();

        /**
          \brief set progress bar current value
          \param val : current value
         */
        void    progress(int val);

        /**
          \brief get zoom value
          \return double
         */
        double  getZoom();

        /**
          \brief get zoom view X value
          \return double
         */
        double  getViewX();

        /**
          \brief get zoom view Y
          \return double
         */
        double  getViewY();

        /**
          \brief test if point exists in image
          \param x : image X
          \param y : image Y
          \return bool
         */
        bool    valid(int x, int y);

        /**
          \brief send a sanpshot to view
         */
        void    sendSnapshot();

        /**
          \brief send a beep sound
         */
        void    beep();

        /**
          \brief iniialize a random generator
          \param j : initial value ( default = 0 )
         */
        void    initRandom(unsigned long long j = 0);

        /**
          \brief generate a random number
          \return unsigned long long
         */
        unsigned long long generateUInt64();

        /**
          \brief generate a random number
          \return double
         */
        double generateDouble();

        /**
          \brief generate a random number
          \return unsigned int
         */
        unsigned int generateUInt32();

        /**
          \brief return last X in width before serialization
          \return int
         */
        int getLastX() const;

        /**
          \brief return last Y in height before serialization
          \return int
         */
        int getLastY() const;

        /**
          \brief return last iteration before serialization
          \return int
         */
        int getLastIteration() const;

        /**
          \brief set last X and Y for serialization
          \param x : last x
          \param y : last y
          \param i : last iteration
         */
        void setLastXYI(int x, int y, int i);

    signals:
        /**
          \brief signal fired within progressbar value changed
          \param val : new progressbar value
         */
        void progressStatus(int val);

        /**
          \brief signal fired for getting a snapshot for current zone
          \param zone : current zone
         */
        void snapshot(AiGenerator::AiZone * zone);

    private:
        /**
          \brief min X in complex zone
         */
        double m_x;

        /**
          \brief min Y in complex zone
         */
        double m_y;

        /**
          \brief zone width
         */
        double m_zoneWidth;

        /**
          \brief zone height
         */
        double m_zoneHeight;

        /**
          \brief zone image width
         */
        int m_imageWidth;

        /**
          \brief zone image height
         */
        int m_imageHeight;

        /**
          \brief zone start point
         */
        int m_startPoint;

        /**
          \brief zone increment point
         */
        int m_increment;

        /**
          \brief iteartion count
         */
        int m_iteration;

        /**
          \brief zoom value
         */
        double m_zoom;

        /**
          \brief zoom view x value
         */
        double m_viewX;

        /**
          \brief zoom view y value
         */
        double m_viewY;

        /**
          \brief current zonz image
         */
        QImage m_image;

        /**
          \brief variable for random numbers
         */
        unsigned long long m_u, m_v, m_w;

        /**
          \brief last X before serialozation
         */
        int m_last_x;

        /**
          \brief last Y before serialization
         */
        int m_last_y;

        /**
          \brief last Itearation before serialization
         */
        int m_last_iteration;
    };
}

/**
  \brief Oops =) what's ? for registring Zone class as Qt object and manipulating it,
  \brief in script engine
 */
Q_DECLARE_METATYPE( AiGenerator::AiZone )

#endif // AIZONE_HPP
