/**
 \brief AiFractal is a UI project code name for Fractal Generator (AiGenerator)
 \author Adrabi Abderrahim
 \date 2009-12-18
 \package AiFractals
 \page License
    \paragraph <The Artistic License>
    The intent of this document is to state the conditions under which a Package may be copied, such that the Copyright Holder maintains some semblance of artistic control over the development of the package, while giving the users of the package the right to use and distribute the Package in a more-or-less customary fashion, plus the right to make reasonable modifications.

    Definitions:
    "Package" refers to the collection of files distributed by the Copyright Holder, and derivatives of that collection of files created through textual modification.
    "Standard Version" refers to such a Package if it has not been modified, or has been modified in accordance with the wishes of the Copyright Holder.
    "Copyright Holder" is whoever is named in the copyright or copyrights for the package.
    "You" is you, if you're thinking about copying or distributing this Package.
    "Reasonable copying fee" is whatever you can justify on the basis of media cost, duplication charges, time of people involved, and so on. (You will not be required to justify it to the Copyright Holder, but only to the computing community at large as a market that must bear the fee.)
    "Freely Available" means that no fee is charged for the item itself, though there may be fees involved in handling the item. It also means that recipients of the item may redistribute it under the same conditions they received it.

    1. You may make and give away verbatim copies of the source form of the Standard Version of this Package without restriction, provided that you duplicate all of the original copyright notices and associated disclaimers.
    2. You may apply bug fixes, portability fixes and other modifications derived from the Public Domain or from the Copyright Holder. A Package modified in such a way shall still be considered the Standard Version.
    3. You may otherwise modify your copy of this Package in any way, provided that you insert a prominent notice in each changed file stating how and when you changed that file, and provided that you do at least ONE of the following:
        a) place your modifications in the Public Domain or otherwise make them Freely Available, such as by posting said modifications to Usenet or an equivalent medium, or placing the modifications on a major archive site such as ftp.uu.net, or by allowing the Copyright Holder to include your modifications in the Standard Version of the Package.
        b) use the modified Package only within your corporation or organization.
        c) rename any non-standard executables so the names do not conflict with standard executables, which must also be provided, and provide a separate manual page for each non-standard executable that clearly documents how it differs from the Standard Version.
        d) make other distribution arrangements with the Copyright Holder.

    4. You may distribute the programs of this Package in object code or executable form, provided that you do at least ONE of the following:
        a) distribute a Standard Version of the executables and library files, together with instructions (in the manual page or equivalent) on where to get the Standard Version.
        b) accompany the distribution with the machine-readable source of the Package with your modifications.
        c) accompany any non-standard executables with their corresponding Standard Version executables, giving the non-standard executables non-standard names, and clearly documenting the differences in manual pages (or equivalent), together with instructions on where to get the Standard Version.
        d) make other distribution arrangements with the Copyright Holder.

    5. You may charge a reasonable copying fee for any distribution of this Package. You may charge any fee you choose for support of this Package. You may not charge a fee for this Package itself. However, you may distribute this Package in aggregate with other (possibly commercial) programs as part of a larger (possibly commercial) software distribution provided that you do not advertise this Package as a product of your own.
    6. The scripts and library files supplied as input to or produced as output from the programs of this Package do not automatically fall under the copyright of this Package, but belong to whomever generated them, and may be sold commercially, and may be aggregated with this Package.
    7. C or perl subroutines supplied by you and linked into this Package shall not be considered part of this Package.
    8. The name of the Copyright Holder may not be used to endorse or promote products derived from this software without specific prior written permission.
    9. THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.

    The End

 */

#include "aimainwindow.hpp"

using namespace AiFractals;
using namespace AiGenerator;

AiMainWindow::AiMainWindow()
        :m_saveFile(),m_selectedFractalIndex(0),m_currentThreadsNumber(1),m_startedThreads(1),
        m_firstCompositionModeIndex(QPainter::CompositionMode_Source), m_othersCompositionModeIndex(QPainter::CompositionMode_Screen),
        m_viewX( .0 ), m_viewY( .0 ), m_zoom( 1. ),
        m_serialize( false )
{
    //~ init

    //# menubar widget
    m_menuBar = new AiMenuBar( this );
    //# toolbar widget
    m_toolbar = new AiToolBar(this);
    //# statusbar widget
    m_statusbar = new AiStatusBar();
    //# view-area widget for displaying generated fractals and opened png files
    m_view = new AiViewerArea(this);
    //# tool-box dialog box maybe GIMP-like ;)
    m_config = new AiConfig(this);
    //#undo dialog
    m_undoDialog = new AiUndoDialog( m_view->view()->stack, this );

    //~ setup

    //# connect menubar signals to main-thread slots
        //* file menu
    this->connect( m_menuBar, SIGNAL(fileOpen()), this, SLOT(fileOpen()));
    this->connect( m_menuBar, SIGNAL(fileSave()), this, SLOT(fileSave()));
    this->connect( m_menuBar, SIGNAL(fileSaveAs()), this, SLOT(fileSaveAs()));
    this->connect( m_menuBar, SIGNAL(filePrint()), this, SLOT(filePrint()));
    this->connect( m_menuBar, SIGNAL(fileClear()), this, SLOT(fileClear()));
    this->connect( m_menuBar, SIGNAL(fileQuit()), this, SLOT(fileQuit()));

        //* zoom menu
    this->connect( m_menuBar, SIGNAL(zoomZoomIn()), this, SLOT(zoomZoomIn()));
    this->connect( m_menuBar, SIGNAL(zoomZoomOut()), this, SLOT(zoomZoomOut()));
    this->connect( m_menuBar, SIGNAL(zoomFitInWindow()), this, SLOT(zoomFitInWindow()));
    this->connect( m_menuBar, SIGNAL(zoomFitToWindow()), this, SLOT(zoomFitToWindow()));
    this->connect( m_menuBar, SIGNAL(zoom1600p()), this, SLOT(zoom1600p()));
    this->connect( m_menuBar, SIGNAL(zoom800p()), this, SLOT(zoom800p()));
    this->connect( m_menuBar, SIGNAL(zoom400p()), this, SLOT(zoom400p()));
    this->connect( m_menuBar, SIGNAL(zoom200p()), this, SLOT(zoom200p()));
    this->connect( m_menuBar, SIGNAL(zoom100p()), this, SLOT(zoom100p()));
    this->connect( m_menuBar, SIGNAL(zoom50p()), this, SLOT(zoom50p()));
    this->connect( m_menuBar, SIGNAL(zoom25p()), this, SLOT(zoom25p()));
    this->connect( m_menuBar, SIGNAL(zoom12_5p()), this, SLOT(zoom12_5p()));
    this->connect( m_menuBar, SIGNAL(zoom6_25p()), this, SLOT(zoom6_25p()));

        //* image menu
    this->connect( m_menuBar, SIGNAL(imageHorizontalMirror()), this, SLOT(imageHorizontalMirror()));
    this->connect( m_menuBar, SIGNAL(imageVerticalMirror()), this, SLOT(imageVerticalMirror()));
    this->connect( m_menuBar, SIGNAL(imageRotation90()), this, SLOT(imageRotation90()));
    this->connect( m_menuBar, SIGNAL(imageRotationNigative90()), this, SLOT(imageRotationNigative90()));
    this->connect( m_menuBar, SIGNAL(imageRotation180()), this, SLOT(imageRotation180()));

        //* complex zone
    this->connect( m_menuBar, SIGNAL(genRun()), this, SLOT(genRun()));
    this->connect( m_menuBar, SIGNAL(genStop()), this, SLOT(genStop()));
    this->connect( m_menuBar, SIGNAL(genThreadsNumbers()), this, SLOT(genThreadsNumbers()));
    this->connect( m_menuBar, SIGNAL(genChooseFractal()), this, SLOT(genChooseFractal()));
    this->connect( m_menuBar, SIGNAL(complexZone()), this, SLOT(complexZone()));
    this->connect( m_menuBar, SIGNAL(serialize()), this, SLOT(serialize()));
    this->connect( m_menuBar, SIGNAL(deserialize()), this, SLOT(deserialize()));

        //* help
    this->connect( m_menuBar, SIGNAL(helpAbout()), this, SLOT(helpAbout()));
    this->connect( m_menuBar, SIGNAL(helpHelp()), this, SLOT(helpHelp()));

        //* toolbar
    this->connect( m_toolbar, SIGNAL(toolbox(bool)),this,SLOT(showToolbox(bool)));
    this->connect( m_toolbar, SIGNAL(fileOpen()), this, SLOT(fileOpen()));
    this->connect( m_toolbar, SIGNAL(fileSave()), this, SLOT(fileSave()));
    this->connect( m_toolbar, SIGNAL(fileClear()), this, SLOT(fileClear()));
    this->connect( m_toolbar, SIGNAL(zoomIn()), this, SLOT(zoomZoomIn()));
    this->connect( m_toolbar, SIGNAL(zoomOut()), this, SLOT(zoomZoomOut()));
    this->connect( m_toolbar, SIGNAL(mirrorH()), this, SLOT(imageHorizontalMirror()));
    this->connect( m_toolbar, SIGNAL(mirrorV()), this, SLOT(imageVerticalMirror()));
    this->connect( m_toolbar, SIGNAL(complexZone()), this, SLOT(complexZone()));
    this->connect( m_toolbar, SIGNAL(helpAbout()), this, SLOT(helpAbout()));
    this->connect( m_toolbar ,SIGNAL(complexRun()),this,SLOT(runGenerator()));
    this->connect( m_toolbar ,SIGNAL(complexStop()),this,SLOT(stopGenerator()));
    this->connect( m_toolbar ,SIGNAL(undo()),this,SLOT(undo()));

        //* configuration
    this->connect(m_config ,SIGNAL(closed()),this,SLOT(checkedToolbox()));
    this->connect(m_config ,SIGNAL(zoomIn()),this,SLOT(zoomZoomIn()));
    this->connect(m_config ,SIGNAL(zoomOut()),this,SLOT(zoomZoomOut()));
    this->connect(m_config ,SIGNAL(zoomArea(bool)),this,SLOT(toolsZoomArea(bool)));
    this->connect(m_config ,SIGNAL(horizontalMirror()),this,SLOT(imageHorizontalMirror()));
    this->connect(m_config ,SIGNAL(verticalMirror()),this,SLOT(imageVerticalMirror()));
    this->connect(m_config ,SIGNAL(clear()),this,SLOT(fileClear()));
    this->connect(m_config ,SIGNAL(rotation(int)),this,SLOT(toolsRotation(int)));
    this->connect(m_config ,SIGNAL(refreshItems()),this,SLOT(refreshItems()));
    this->connect(m_config ,SIGNAL(debugFractal()),this,SLOT(debugFractal()));
    this->connect(m_config ,SIGNAL(currentIndexChanged(int)),this,SLOT(currentIndexChanged(int)));
    this->connect(m_config ,SIGNAL(currentThreads(int)),this,SLOT(currentThreads(int)));
    this->connect(m_config ,SIGNAL(firstCompositionMode(int)),this,SLOT(firstCompositionMode(int)));
    this->connect(m_config ,SIGNAL(othersCompositionMode(int)),this,SLOT(othersCompositionMode(int)));
    this->connect(m_config ,SIGNAL(runGenerator()),this,SLOT(runGenerator()));
    this->connect(m_config ,SIGNAL(stopGenerator()),this,SLOT(stopGenerator()));

    this->connect(m_view ,SIGNAL(zoneZooming(int,int,int,int)),this,SLOT(zoneZooming(int,int,int,int)));
    this->connect(m_view ,SIGNAL(enabledChilds(bool)),this,SLOT(enabledChilds(bool)));

    this->setMenuBar( m_menuBar );
    this->addToolBar( m_toolbar );
    this->setStatusBar( m_statusbar );
    this->setCentralWidget( m_view );


    this->enabledChilds( false );

    m_config->resize(200,400);
    this->resize(500,400);
}

void AiMainWindow::showToolbox(bool show)
{
    m_config->setShown( show );
}

void AiMainWindow::checkedToolbox()
{
    m_toolbar->checkedToolbox( false );
}

void AiMainWindow::fileOpen()
{
    try
    {
        //~ test if screen is not empty (no image opened or generated)
        if( !m_view->view()->emptyScreen() )
        {
            if( QMessageBox::Open != QMessageBox::warning(this, tr("Open...") , tr("The screen is not empty, you want really open another file in this screen ?"), QMessageBox::Open | QMessageBox::Cancel, QMessageBox::Cancel ) )
            {
                return;
            }
        }

        //~ open a png image
        QString fileName = QFileDialog::getOpenFileName(this, tr("Open Image"), "", tr("Image File (*.png)"));
        m_view->view()->makeUpdate( fileName );
        if( !fileName.isNull() && !fileName.isEmpty() )
        {
            this->enabledChilds( true );
        }
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::fileSave()
{
    try
    {
        //~ test if file name to save image is empty (in first time)
        //~ for not show again a file dialog to save image =)
        if( m_saveFile.isEmpty() )
        {
            m_saveFile = QFileDialog::getSaveFileName(this, tr("Save Image"), "", tr("Image File (*.png)"));
        }

        //~ save png image
        m_view->view()->saveAs( m_saveFile );
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::fileSaveAs()
{
    try
    {
        //~ save image as another name
        m_saveFile = QFileDialog::getSaveFileName(this, tr("Save Image"), "", tr("Image File (*.png)")) ;
        m_view->view()->saveAs( m_saveFile );
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::filePrint()
{
    try
    {
        //~ print an image
        m_view->view()->print();
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::fileClear()
{
    try
    {
        //~ test if screen is not empty
        if( !m_view->view()->emptyScreen() )
        {
            if( QMessageBox::Yes != QMessageBox::warning(this, tr("Clear...") , tr("The screen is not empty, you want really clear this screen ?"), QMessageBox::Yes | QMessageBox::No, QMessageBox::No ) )
            {
                return;
            }
        }
        m_view->view()->clear();
        this->enabledChilds( false );
        m_zoom = 1.;
        m_viewX = m_viewY = .0;
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::fileQuit()
{
    try
    {
        this->close();
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::zoomZoomIn()
{
    try
    {
        m_view->view()->scale(1.5,1.5);
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::zoomZoomOut()
{
    try
    {
        m_view->view()->scale(-.5,-.5);
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::zoomFitInWindow()
{
    try
    {
        m_view->view()->fillInWindow();
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::zoomFitToWindow()
{
    try
    {
        m_view->view()->fillToWindow();
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::zoom1600p()
{
    try
    {
        m_view->view()->clearScale();
        m_view->view()->scale(16.,16.);
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::zoom800p()
{
    try
    {
        m_view->view()->clearScale();
        m_view->view()->scale(8.,8.);
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::zoom400p()
{
    try
    {
        m_view->view()->clearScale();
        m_view->view()->scale(4.,4.);
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::zoom200p()
{
    try
    {
        m_view->view()->clearScale();
        m_view->view()->scale(2.,2.);
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::zoom100p()
{
    try
    {
        m_view->view()->clearScale();
        m_view->view()->scale(1.,1.);
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::zoom50p()
{
    try
    {
        m_view->view()->clearScale();
        m_view->view()->scale(.5,.5);
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::zoom25p()
{
    try
    {
        m_view->view()->clearScale();
        m_view->view()->scale(-.25,-.25);
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::zoom12_5p()
{
    try
    {
        m_view->view()->clearScale();
        m_view->view()->scale(0.125,.125);
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::zoom6_25p()
{
    try
    {
        m_view->view()->clearScale();
        m_view->view()->scale(.0625,.0625);
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::imageHorizontalMirror()
{
    try
    {
        m_view->view()->retate( 361. );
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::imageVerticalMirror()
{
    try
    {
        m_view->view()->retate( 362. );
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::imageRotation90()
{
    try
    {
        m_view->view()->retate( 90. );
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::imageRotationNigative90()
{
    try
    {
        m_view->view()->retate( -90. );
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::imageRotation180()
{
    try
    {
        m_view->view()->retate( 180. );
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::genRun()
{
    try
    {
        //~ set active tab and show toolbox if not visible,
        //finally check toolbox button
        m_config->activeTab( 3 ) ;
        m_toolbar->checkedToolbox( true );
        m_config->setShown( true );
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::genStop()
{
    try
    {
        this->genRun();
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::genThreadsNumbers()
{
    try
    {
        this->genRun();
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::genChooseFractal()
{
    try
    {
        //~ set active tab to "fractals tab" and show toolbox if not visible,
        //finally check toolbox button
        m_config->activeTab( 1 ) ;
        m_toolbar->checkedToolbox( true );
        m_config->setShown( true );
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::complexZone()
{
    try
    {
        //~ set active tab to "complex zone tab" and show toolbox if not visible,
        //finally check toolbox button
        m_config->activeTab( 2 ) ;
        m_toolbar->checkedToolbox( true );
        m_config->setShown( true );
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::serialize()
{
    try
    {
        //check if threads is running
        if( m_threads.size() > 0 )
        {
            if( m_threads[ 0 ]->isRunning() )
            {
                if( QMessageBox::Yes != QMessageBox::warning(this, tr("Serialization...") , tr("GOD MODE stop all current threads, you want continue?"), QMessageBox::Yes | QMessageBox::No, QMessageBox::No ) )
                {
                    return;
                }
                else
                {
                    //# magic bool
                    m_serialize = true;
                    //#

                    // stoping current threads
                    for( int index = 0 ; index < m_threads.size() ; index++ )
                    {
                        m_threads[ index ]->stop();
                        m_threads[ index ]->wait();
                    }

                }
            }
        }
        else
        {
            QMessageBox::warning(this, tr("Threads"), tr("No threads for using 'GOD MODE' =)!"), QMessageBox::Ok) ;
            return;
        }

        //get file to save serialisation
        QString toFile = QFileDialog::getSaveFileName(this, tr("Serialization"), "", tr("File (*.adrabi)")) ;
        if( !toFile.isNull() && !toFile.isEmpty() )
        {
            QFile file( toFile );
            file.open(QIODevice::WriteOnly);

            //serializing (hmm.. maybe as TCP-Packets with header and data)
                //# Header
            QDataStream stream( &file );
            stream.setVersion(QDataStream::Qt_4_6);
            stream  << QTime::currentTime()
                    << QDate::currentDate()
                    << (m_threads.count());

                //# Data

            for( int x = 0 ; x < m_threads.count() ; x++ )
            {
                AiZone * zone = m_threads[ x ]->getZone();
                zone->serialize( stream );

            }
        }

        if( QMessageBox::Yes != QMessageBox::warning(this, tr("Threads...") , tr("You want clean all threads now?"), QMessageBox::Yes | QMessageBox::No, QMessageBox::No ) )
        {
            return;
        }

        //~ delete them all
        for( int index = 0 ; index < m_threads.size() ; index++ )
        {
            delete m_threads[ index ];
        }
        m_threads.clear();
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::deserialize()
{
    try
    {
        //check if threads is running
        if( m_threads.size() > 0 )
        {
            if( m_threads[ 0 ]->isRunning() )
            {
                QMessageBox::warning(this, tr("Threads"), tr("Current threads is running, please stop them before running a new deserialization!"), QMessageBox::Ok) ;
                return;
            }
        }

        if( QMessageBox::Yes != QMessageBox::warning(this, tr("Deserialization...") , tr("Make sure you are choosing right script and composition modes?"), QMessageBox::Yes | QMessageBox::No, QMessageBox::No ) )
        {
            return;
        }

        QString fileName = QFileDialog::getOpenFileName(this, tr("Deserialization..."), "", tr("File (*.adrabi)"));
        if( fileName.isNull() || fileName.isEmpty() )
        {
            return;
        }

        //~ Just added for checking
        // stoping current threads
        for( int index = 0 ; index < m_threads.size() ; index++ )
        {
            m_threads[ index ]->stop();
            m_threads[ index ]->wait();
        }
        //~ delete them all
        for( int index = 0 ; index < m_threads.size() ; index++ )
        {
            delete m_threads[ index ];
        }
        m_threads.clear();
        //~ Just added for checking

        QFile file( fileName );
        file.open(QIODevice::ReadOnly);

        QDataStream stream( &file );
        stream.setVersion(QDataStream::Qt_4_6);
        QTime time; QDate date;
        stream  >> time     // creation time
                >> date     // creation date
                >> m_startedThreads;   // threads count
        
        //~ start at
        this->m_statusbar->start();

        //~ storage
        if( m_storage.count() > 0 )
        {
            m_storage.clear();
        }
        m_storage = QVector<QImage>( m_startedThreads );

        for( int x = 0 ; x < m_startedThreads ; x++ )
        {
            AiZone zone (0,0,0,0,0,0,0,0,0,0,0,0,0);
            zone.deserialize( stream );  

            //
            AiThread * thread = new AiThread( this );
            thread->setZone( zone );
            thread->scriptFile( "scripts/" + m_config->item( m_selectedFractalIndex ) );

            qRegisterMetaType<AiGenerator::AiZone*>("AiGenerator::AiZone");
            this->connect( thread, SIGNAL(completed(AiGenerator::AiZone*)) , this, SLOT(completed(AiGenerator::AiZone*)));
            this->connect( thread, SIGNAL(snapshot(AiGenerator::AiZone*)) , this, SLOT(snapshot(AiGenerator::AiZone*)));
            //~ for first thread *ONLY*
            if( x == 0 )
            {
                //# theoretically all threads has started in one time,
                //# with very small deference : T0:0 nanos T1:~+1ns etc...
                //# if first thread is finish all threads finished in one time
                this->connect(thread,SIGNAL(progressStatus(int)), m_statusbar, SLOT(setGenStatus(int)));
                m_zoom = zone.getZoom();
                m_viewX= zone.getViewX();
                m_viewY= zone.getViewY();

                //~ set old setting
                m_config->setZoneWidth( zone.getZoneWidth() );
                m_config->setZoneHeight( zone.getZoneHeight());
                m_config->setZoneMinX( zone.getX() );
                m_config->setZoneMinY( zone.getY() );
                m_config->setImageWidth( zone.getImageWidth() );
                m_config->setImageHeight( zone.getImageHeight() );
                m_config->setIterations( zone.getIteration() );
                m_config->setThreadsNumber( m_startedThreads );
            }

            m_threads.push_back( thread );
        }

        for( int x = 0 ; x < m_threads.count() ; x++ )
        {
            m_threads[ x ]->start();
        }

        m_statusbar->startGenerating();
        m_statusbar->setInfo(  QString( "[Threads: %1 ]"  ).arg( m_startedThreads ) );

        //# for supporting users *BAD* manipulation =)
        this->enabledChilds( false );
        this->toolsZoomArea( false );
        m_config->disableZoomArea();
        //# for supporting users *BAD* manipulation =)
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::helpHelp()
{
    try
    {
        AiHelp * help = new AiHelp(  );
        help->setAttribute(Qt::WA_QuitOnClose, false);
        help->show();
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::helpAbout()
{
    try
    {
        AiAbout * about = new AiAbout(  );
        about->setModal( true );
        about->show();
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::toolsZoomArea(bool enable)
{
    try
    {
        m_view->view()->zooming( enable );
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::toolsRotation(int value)
{
    try
    {
        m_view->view()->retate( static_cast<qreal>( value ) );
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::refreshItems()
{
    try
    {
        QDir dir("scripts");
        if( !dir.exists() )
        {
            // create dir if not exists
            dir.mkdir("scripts");
        }

        // getting a list of QtScript files
        QFileInfoList entries = dir.entryInfoList(QStringList() << "*.js");
        QStringList list;
        for (int i = 0; i < entries.size(); ++i)
        {
            list << entries.at(i).fileName();
        }
        if( !list.empty() )
        {
            m_config->addItems( list );
        }
        else
        {
            QMessageBox::information(this, tr("Fractals scripts"), tr("Please add your fractals scripts in folder \"[AiFractals-dir]/scripts\" =)."));
        }
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::currentIndexChanged ( int index )
{
    try
    {
        m_selectedFractalIndex = index;
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::debugFractal()
{
    try
    {
       m_startedThreads = m_currentThreadsNumber;

       if( (m_config->getImageWidth() == 0 || m_config->getImageHeight() == 0) )
       {
            QMessageBox::warning(this, tr("Config"), tr("Please, setup a fractal config!"), QMessageBox::Ok) ;
            this->complexZone();
            return;
       }

       if( QMessageBox::Yes != QMessageBox::warning(this, tr("Debug...") , tr("You want really debug this script ?"), QMessageBox::Yes | QMessageBox::No, QMessageBox::No ) )
       {
            return;
       }

        //~ create a complex zone
        AiZone * zone = new AiZone(this,
                                   m_config->getZoneMinX(),
                                   m_config->getZoneMinY(),
                                   m_config->getZoneWidth(),
                                   m_config->getZoneHeight(),
                                   m_config->getImageWidth(),
                                   m_config->getImageHeight(),
                                   0, //# start point
                                   m_startedThreads, //# threads numbers
                                   m_config->getIterations() ,
                                   m_zoom,
                                   m_viewX,
                                   m_viewY);

        QScriptEngine engine;
        {
            QScriptValue qs_zone = engine.newQObject( zone );
            engine.globalObject().setProperty( "zone" , qs_zone );
        }
        QScriptEngineDebugger debugger(this);
        debugger.attachTo(&engine);
        debugger.standardWindow()->setWindowModality(Qt::ApplicationModal);


        //~ read script file
        QString fileName("scripts/" + m_config->item( m_selectedFractalIndex ));
        QString script;
        {
            QFile file( fileName );
            file.open(QIODevice::ReadOnly);
            script = file.readAll();
            file.close();
        }

        debugger.action(QScriptEngineDebugger::InterruptAction)->trigger();
        engine.evaluate(script, fileName);

        //# UPDATE view

        m_view->view()->clear();
        m_view->view()->makeUpdate( zone->image(), true );
        this->enabledChilds( true );

        delete zone;
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::currentThreads(int thread)
{
    m_currentThreadsNumber = thread;
}

void AiMainWindow::firstCompositionMode( int val )
{
    m_firstCompositionModeIndex = forgeCompositionMode( val );
}

void AiMainWindow::othersCompositionMode( int val )
{
    m_othersCompositionModeIndex = forgeCompositionMode( val );
}

void AiMainWindow::runGenerator()
{
    try
    {
        m_startedThreads = m_currentThreadsNumber;

        if( (m_config->getImageWidth() == 0 || m_config->getImageHeight() == 0) )
        {
            QMessageBox::warning(this, tr("Config"), tr("Please, setup a fractal config!"), QMessageBox::Ok) ;
            this->complexZone();
            return;
        }

        //check if threads is running
        if( m_threads.size() > 0 )
        {
            if( m_threads[ 0 ]->isRunning() )
            {
                QMessageBox::warning(this, tr("Threads"), tr("Current threads is running, please stop them before running a new threads!"), QMessageBox::Ok) ;
                return;
            }
        }

        //~ start at
        this->m_statusbar->start();

        //~ clean before run
        for( int index = 0 ; index < m_threads.size() ; index++ )
        {
            delete m_threads[ index ];
        }
        m_threads.clear();

        //~ storage
        if( m_storage.count() > 0 )
        {
            m_storage.clear();
        }
        m_storage = QVector<QImage>( m_startedThreads );

        //creating and calulating threads
        for( int index = 0 ; index < m_startedThreads ; index++ )
        { 
            AiThread *thread = new AiThread(this);
            thread->makeZone( m_config->getZoneMinX()   ,
                             m_config->getZoneMinY()   ,
                             m_config->getZoneWidth()  ,
                             m_config->getZoneHeight() ,
                             m_config->getImageWidth() ,
                             m_config->getImageHeight(),
                             index                  , //# start point
                             m_startedThreads , //# threads numbers
                             m_config->getIterations(),
                             m_zoom,
                             m_viewX,
                             m_viewY);
            thread->scriptFile( "scripts/" + m_config->item( m_selectedFractalIndex ) );

            qRegisterMetaType<AiGenerator::AiZone*>("AiGenerator::AiZone");
            this->connect( thread, SIGNAL(completed(AiGenerator::AiZone*)) , this, SLOT(completed(AiGenerator::AiZone*)));
            this->connect( thread, SIGNAL(snapshot(AiGenerator::AiZone*)) , this, SLOT(snapshot(AiGenerator::AiZone*)));
            //~ for first thread *ONLY*
            if( index == 0 )
            {
                //# theoretically all threads has started in one time,
                //# with very small deference : T0:0 nanos T1:~+1ns etc...
                //# if first thread is finish all threads finished in one time
                this->connect(thread,SIGNAL(progressStatus(int)), m_statusbar, SLOT(setGenStatus(int)));
            }

            m_threads.push_back( thread );
        }

        for( int index = 0 ; index < m_threads.size() ; index++ )
        {
            m_threads[ index ]->start();
        }

        m_statusbar->startGenerating();
        m_statusbar->setInfo(  QString( "[Threads: %1 ]"  ).arg( m_startedThreads ) );

        //# for supporting users *BAD* manipulation =)
        this->enabledChilds( false );
        this->toolsZoomArea( false );
        m_config->disableZoomArea();
        //# for supporting users *BAD* manipulation =)
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::stopGenerator()
{
    try
    {
        //check if threads is running
        if( m_threads.size() > 0 )
        {
            if( m_threads[ 0 ]->isRunning() )
            {
                if( QMessageBox::Ok == QMessageBox::warning(this, tr("Threads"), tr("Current threads is running you want to stop them ?."), QMessageBox::Ok | QMessageBox::Cancel) )
                {
                    // stoping current threads
                    for( int index = 0 ; index < m_threads.size() ; index++ )
                    {
                        m_threads[ index ]->stop();
                        m_threads[ index ]->wait();
                    }
                }
                else
                {
                    return;
                }
            }
        }
        m_statusbar->stopGenerating();
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::snapshot( AiGenerator::AiZone * zone )
{
    try
    {
        QMutex mutex;
        mutex.lock();
        {
            m_storage.replace( zone->getStartPoint(), zone->image() );

            static int count = 0;
            count++;
            if ( count == m_startedThreads && m_storage.size() > 0 )
            {
                QImage tmp;
                //~ if screen is empty create a new empty image
                if( m_view->view()->emptyScreen() && m_config->getImageWidth() > 0 && m_config->getImageHeight() > 0 )
                {
                    tmp = QImage(m_config->getImageWidth(), m_config->getImageHeight(), QImage::Format_RGB32);
                    tmp.fill(Qt::transparent);
                }
                else if( (m_config->getImageWidth() == 0 || m_config->getImageHeight() == 0) && m_storage.count() > 0  )
                {
                    tmp = QImage( m_storage[0].width(), m_storage[0].height(), QImage::Format_RGB32);
                    tmp.fill(Qt::transparent);
                }
                else //~ get a current screen image for adding to it, a generated image =)
                {
                    tmp = m_view->view()->image();
                }

                QPainter painter( &tmp );
                for( int index = 0 ; index < m_storage.size() ; index++ )
                {
                    if( index == 0 )
                    {
                        //# set to first thread a deferent compsition mode
                        painter.setCompositionMode( m_firstCompositionModeIndex );
                    }
                    else
                    {
                        painter.setCompositionMode( m_othersCompositionModeIndex );
                    }
                    painter.drawImage(0,0, m_storage[ index ] );
                }
                painter.end();

                m_view->view()->makeUpdate( tmp, false );
                count = 0;
            }
        }
        mutex.unlock();
    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::completed(AiGenerator::AiZone * zone )
{
    try
    {
        QMutex mutex;
        mutex.lock();
        {
            m_storage.replace( zone->getStartPoint(), zone->image() );

            static int count = 0;
            count++;
            if ( count == m_startedThreads && m_storage.size() > 0 )
            {
                QImage tmp;
                //~ if screen is empty create a new empty image
                if( m_view->view()->emptyScreen() && m_config->getImageWidth() > 0 && m_config->getImageHeight() > 0 )
                {
                    tmp = QImage(m_config->getImageWidth(), m_config->getImageHeight(), QImage::Format_RGB32);
                    tmp.fill(Qt::transparent);
                }
                else if( (m_config->getImageWidth() == 0 || m_config->getImageHeight() == 0) && m_storage.count() > 0  )
                {
                    tmp = QImage( m_storage[0].width(), m_storage[0].height(), QImage::Format_RGB32);
                    tmp.fill(Qt::transparent);
                }
                else //~ get a current screen image for adding to it, a generated image =)
                {
                    tmp = m_view->view()->image();
                }

                QPainter painter( &tmp );
                for( int index = 0 ; index < m_storage.size() ; index++ )
                {
                    if( index == 0 )
                    {
                        //# set to first thread a deferent compsition mode
                        painter.setCompositionMode( m_firstCompositionModeIndex );
                    }
                    else
                    {
                        painter.setCompositionMode( m_othersCompositionModeIndex );
                    }
                    painter.drawImage(0,0, m_storage[ index ] );
                }
                painter.end();

                m_view->view()->makeUpdate( tmp, true );
                m_storage.clear();

                //~ enable tools and change status
                this->enabledChilds( true );
                m_statusbar->stopGenerating();

                //~ set to zero for next generation
                count = 0;

                //~ finally clean threads
                for( int index = 0 ; index < m_threads.size() ; index++ )
                {
                    m_threads[ index ]->stop();
                    m_threads[ index ]->wait();
                }

                //~ new implementation for "God Mode"

                if( !m_serialize )
                {
                    //~ delete them all
                    for( int index = 0 ; index < m_threads.size() ; index++ )
                    {
                        delete m_threads[ index ];
                    }
                    m_threads.clear();
                }

                //# magic bool
                m_serialize = false;
                //#

                //~ end at
                this->m_statusbar->end();
            }
        }
        mutex.unlock();

    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }
}

void AiMainWindow::zoneZooming ( int startx, int starty, int endx, int endy )
{
    //check if threads is running
    if( m_threads.size() > 0 )
    {
        if( m_threads[ 0 ]->isRunning() )
        {
            QMessageBox::warning(this, tr("Threads"), tr("Current threads is running, please stop them before running a new threads!"), QMessageBox::Ok) ;
            return;
        }
    }

    if( QMessageBox::Yes != QMessageBox::warning(this, tr("Zooming...") , tr("Are you sure to zooming this area?"), QMessageBox::Yes | QMessageBox::No, QMessageBox::No ) )
    {
        return;
    }

    int w = m_config->getImageWidth();
    int h = m_config->getImageHeight();

    int minwh = w < h ? w : h;

    m_viewX = m_viewX + ( m_zoom * ( startx < endx ? startx : endx ) /   minwh );
    m_viewY = m_viewY + ( m_zoom * ( starty < endy ? starty : endy ) /   minwh );
    double tmpX = static_cast<double>(abs( startx - endx )) / w ;
    double tmpY = static_cast<double>(abs( starty - endy )) / h ;

    m_zoom = m_zoom * ( tmpX > tmpY ? tmpX : tmpY ) ;

    this->runGenerator();
}

void AiMainWindow::enabledChilds(bool e)
{
    m_menuBar->enabledChilds( e );
    m_toolbar->enabledChilds( e );
    m_config->enabledChilds ( e );
}

void AiMainWindow::undo()
{
    if ( m_undoDialog->isHidden() )
    {
        m_undoDialog->show();
    }
    else
    {
        m_undoDialog->hide();
    }
}


QPainter::CompositionMode AiMainWindow::forgeCompositionMode( int index )
{
    switch( index )
    {
        case 0 : return QPainter::CompositionMode_Clear;
        case 1 : return QPainter::CompositionMode_ColorBurn;
        case 2 : return QPainter::CompositionMode_ColorDodge;
        case 3 : return QPainter::CompositionMode_Darken;
        case 4 : return QPainter::CompositionMode_Destination;
        case 5 : return QPainter::CompositionMode_DestinationAtop;
        case 6 : return QPainter::CompositionMode_DestinationIn;
        case 7 : return QPainter::CompositionMode_DestinationOut;
        case 8 : return QPainter::CompositionMode_DestinationOver;
        case 9 : return QPainter::CompositionMode_Difference;
        case 10 : return QPainter::CompositionMode_Exclusion;
        case 11 : return QPainter::CompositionMode_HardLight;
        case 12 : return QPainter::CompositionMode_Lighten;
        case 13 : return QPainter::CompositionMode_Multiply;
        case 14 : return QPainter::CompositionMode_Overlay;
        case 15 : return QPainter::CompositionMode_Plus;
        case 16 : return QPainter::CompositionMode_Screen;
        case 17 : return QPainter::CompositionMode_SoftLight;
        case 18 : return QPainter::CompositionMode_Source;
        case 19 : return QPainter::CompositionMode_SourceAtop;
        case 20 : return QPainter::CompositionMode_SourceIn;
        case 21 : return QPainter::CompositionMode_SourceOut;
        case 22 : return QPainter::CompositionMode_SourceOver;
        case 23 : return QPainter::CompositionMode_Xor;
        default : return QPainter::CompositionMode_Plus;

    }
}

AiMainWindow::~AiMainWindow()
{
    try
    {
        //~ stop and clean threas before destroy a main window ;) oupss memory
        for( int index = 0 ; index < m_threads.size() ; index++ )
        {
            m_threads[ index ]->stop();
            m_threads[ index ]->wait();
            delete m_threads[ index ];
        }

    }
    catch(const std::exception & exception )
    {
        QMessageBox::critical(this, tr("Critical Error!"), tr("Please report this problem to adrabi[at]gmail[dot]com =)."));
    }

    if( m_config != NULL )
    {
        delete m_config;
    }
    if( m_menuBar != NULL )
    {
        delete m_menuBar;
    }
    if( m_statusbar != NULL )
    {
        delete m_statusbar;
    }
    if( m_toolbar != NULL )
    {
        delete m_toolbar;
    }
    if( m_view != NULL )
    {
        delete m_view;
    }
    if( m_undoDialog != NULL )
    {
        delete m_undoDialog;
    }
}
