/**
 \brief AiFractal is a UI project code name for Fractal Generator (AiGenerator)
 \author Adrabi Abderrahim
 \date 2009-12-15
 \package AiFractals
 \page License
    \paragraph <The Artistic License>
    The intent of this document is to state the conditions under which a Package may be copied, such that the Copyright Holder maintains some semblance of artistic control over the development of the package, while giving the users of the package the right to use and distribute the Package in a more-or-less customary fashion, plus the right to make reasonable modifications.

    Definitions:
    "Package" refers to the collection of files distributed by the Copyright Holder, and derivatives of that collection of files created through textual modification.
    "Standard Version" refers to such a Package if it has not been modified, or has been modified in accordance with the wishes of the Copyright Holder.
    "Copyright Holder" is whoever is named in the copyright or copyrights for the package.
    "You" is you, if you're thinking about copying or distributing this Package.
    "Reasonable copying fee" is whatever you can justify on the basis of media cost, duplication charges, time of people involved, and so on. (You will not be required to justify it to the Copyright Holder, but only to the computing community at large as a market that must bear the fee.)
    "Freely Available" means that no fee is charged for the item itself, though there may be fees involved in handling the item. It also means that recipients of the item may redistribute it under the same conditions they received it.

    1. You may make and give away verbatim copies of the source form of the Standard Version of this Package without restriction, provided that you duplicate all of the original copyright notices and associated disclaimers.
    2. You may apply bug fixes, portability fixes and other modifications derived from the Public Domain or from the Copyright Holder. A Package modified in such a way shall still be considered the Standard Version.
    3. You may otherwise modify your copy of this Package in any way, provided that you insert a prominent notice in each changed file stating how and when you changed that file, and provided that you do at least ONE of the following:
        a) place your modifications in the Public Domain or otherwise make them Freely Available, such as by posting said modifications to Usenet or an equivalent medium, or placing the modifications on a major archive site such as ftp.uu.net, or by allowing the Copyright Holder to include your modifications in the Standard Version of the Package.
        b) use the modified Package only within your corporation or organization.
        c) rename any non-standard executables so the names do not conflict with standard executables, which must also be provided, and provide a separate manual page for each non-standard executable that clearly documents how it differs from the Standard Version.
        d) make other distribution arrangements with the Copyright Holder.

    4. You may distribute the programs of this Package in object code or executable form, provided that you do at least ONE of the following:
        a) distribute a Standard Version of the executables and library files, together with instructions (in the manual page or equivalent) on where to get the Standard Version.
        b) accompany the distribution with the machine-readable source of the Package with your modifications.
        c) accompany any non-standard executables with their corresponding Standard Version executables, giving the non-standard executables non-standard names, and clearly documenting the differences in manual pages (or equivalent), together with instructions on where to get the Standard Version.
        d) make other distribution arrangements with the Copyright Holder.

    5. You may charge a reasonable copying fee for any distribution of this Package. You may charge any fee you choose for support of this Package. You may not charge a fee for this Package itself. However, you may distribute this Package in aggregate with other (possibly commercial) programs as part of a larger (possibly commercial) software distribution provided that you do not advertise this Package as a product of your own.
    6. The scripts and library files supplied as input to or produced as output from the programs of this Package do not automatically fall under the copyright of this Package, but belong to whomever generated them, and may be sold commercially, and may be aggregated with this Package.
    7. C or perl subroutines supplied by you and linked into this Package shall not be considered part of this Package.
    8. The name of the Copyright Holder may not be used to endorse or promote products derived from this software without specific prior written permission.
    9. THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.

    The End

 */

#include "aiview.hpp"
#include <iostream>

using namespace AiFractals;

AiView::AiView(QWidget * parent)
        : QWidget( parent) ,m_image(0,0,QImage::Format_RGB32),
        m_scaleX(0), m_scaleY(0), m_scale(false), m_retation(0),m_isZooming(false),
        m_startX(0), m_startY(0), m_endX(0), m_endY(0), m_fill2Win(false)
{
    //~ setting "undo"
    this->stack = new QUndoStack( this );
}

 void AiView::paintEvent ( QPaintEvent * event )
{
     if ( !this->m_image.isNull() )
     {
         QPainter painter( this );
         QMatrix matrix;

         //~ set a custom scale
         if( m_scaleX != .0 && m_scaleY != .0  )
         {
             static qreal scaleX = .0;
             static qreal scaleY = .0;

             scaleX = m_scale ? m_clearScale ? m_scaleX : scaleX + m_scaleX : scaleX;
             scaleY = m_scale ? m_clearScale ? m_scaleY : scaleY + m_scaleY : scaleY;

             matrix.scale(scaleX, scaleY);
             m_scale = m_clearScale = false;
         }
         //~ set a custom retation
         if( m_retation != .0 && m_retation <= 360. )
         {
             //# rotation
             matrix.rotate( m_retation );
         }

         //~ set image mirror
         else if( m_canRetate && m_retation == 361. )
         {
             //# horizonatal
             m_image = m_image.mirrored(true,false);
             m_canRetate = false ;
         }
         else if( m_canRetate && m_retation == 362. )
         {
             //# virdircal
             m_image = m_image.mirrored(false,true);
             m_canRetate = false ;
         }

         QImage tmp = this->m_image.transformed( matrix) ;
         //~ fill and draw image with current view parameters, with and height
         if( m_fill2Win )
         {
             painter.drawPixmap(0, 0, this->width(), this->height(), QPixmap::fromImage(tmp) ) ;
         }
         else
         {
             int x = ( this->width() / 2 ) - ( tmp.width() / 2 );
             int y = ( this->height() / 2 ) - ( tmp.height() / 2 );
             painter.drawPixmap(x, y, QPixmap::fromImage(tmp) ) ;
         }


         if( m_isZooming )
         {
             int x = ( this->width() / 2 ) - ( tmp.width() / 2 );
             int y = ( this->height() / 2 ) - ( tmp.height() / 2 );
             //~ tesing if zooming zone in image *ONLY*
             if( m_startX >= x && m_startY >= y && m_endX <= x + tmp.width() && m_endY <= y + tmp.height() && m_endX > m_startX && m_endY > m_startY )
             {
                painter.setPen(QColor(255,255,255));

                painter.drawRect(m_startX, m_startY, m_endX - m_startX , m_endY - m_startY );
                painter.drawPixmap(m_endX , m_endY , QPixmap(":/png/zoom-in"));
             }
         }


         painter.end();
     }
}

void AiView::mouseMoveEvent ( QMouseEvent * event )
{
    if( m_isZooming )
    {
       m_endX = event->x();
       m_endY = event->y();
    }
    this->update();
}

void AiView::mousePressEvent ( QMouseEvent * event )
{
    if( m_isZooming )
    {
        m_startX = event->x();
        m_startY = event->y();
    }
}

void AiView::mouseReleaseEvent ( QMouseEvent * event )
{
    if( m_isZooming )
    {
        int x = ( this->width() / 2 ) - ( m_image.width() / 2 );
        int y = ( this->height() / 2 ) - ( m_image.height() / 2 );
        m_endX = event->x();
        m_endY = event->y();
        //~ tesing if zooming zone in image *ONLY*
        if( m_startX >= x && m_startY >= y && m_endX <= x + m_image.width() && m_endY <= y + m_image.height() && m_endX > m_startX && m_endY > m_startY )
        {
            emit zoneZooming( m_startX - x , m_startY - y, m_endX - x, m_endY - y );
            m_startX = m_endX = 0;
        }
    }
}

void AiView::makeUpdate()
{
    this->resize( widthView(m_image.width()) , heightView(m_image.height()) );
    this->update();
}

void AiView::makeUpdate( QImage & image, bool undo )
{
    m_image = image;
    this->resize( widthView(m_image.width()) , heightView(m_image.height()) );
    this->update();

    if ( undo && !m_image.isNull() )
    {
        AiCommand * cmd = new AiCommand(m_image, this);
        this->stack->push( cmd );
        cmd->id( stack->count() - 1 );
    }
}

void AiView::makeUpdate( QString & fileName )
{
    m_image = QImage( fileName );
    this->resize( widthView(m_image.width()) , heightView(m_image.height()) );
    this->update();

    if ( !m_image.isNull() )
    {
        AiCommand * cmd = new AiCommand(m_image, this);
        this->stack->push( cmd );
        cmd->id( stack->count() - 1 );
    }
}

void AiView::makeUndoUpdate( QImage & image )
{
    m_image = image;
    this->resize( widthView(m_image.width()) , heightView(m_image.height()) );
    this->update();
    emit enabledChilds( !m_image.isNull() );
}

QImage & AiView::image()
{
    return m_image;
}

const bool AiView::emptyScreen()
{
    return m_image.isNull();
}

const qreal AiView::getScaleX() const
{
    return m_scaleX;
}

const qreal AiView::getScaleY() const
{
    return m_scaleY;
}

void AiView::clearScale()
{
    if( !m_image.isNull() )
    {
        m_clearScale = true;
    }
}

const qreal AiView::getRetation() const
{
    return m_retation;
}

const int AiView::getImageWidth()
{
    return m_image.width();
}

const int AiView::getImageHeight()
{
    return m_image.height();
}

void AiView::retate(qreal r)
{
    if( !m_image.isNull() )
    {
        m_retation = r;
        m_canRetate = true;
        this->update();
    }
}

void AiView::scale(qreal sx, qreal sy)
{
    if( !m_image.isNull() )
    {
        m_scaleX = sx;
        m_scaleY = sy;
        m_scale  = true;
        this->resize( widthView(m_image.width()) , heightView(m_image.height()) );
        this->update();
    }
}

void AiView::saveAs( QString & fileName )
{
    this->m_image.save( fileName );
}

void AiView::print()
{
    QPrintDialog printDialog( this );
    if( printDialog.exec() == QDialog::Accepted )
    {
        QPainter painter( &printDialog );
        QRect rect = painter.viewport();
        QSize size = m_image.size();
        size.scale( rect.size() ,Qt::KeepAspectRatio);
        painter.setViewport( rect.x(), rect.y(), size.width(), size.height() );
        painter.setWindow( m_image.rect() );
        painter.drawImage( 0, 0, m_image );
    }
}

void AiView::clear()
{
    m_image = QImage();
    this->update();
}

int AiView::widthView(int w)
{
    QWidget * wd = qobject_cast<QWidget *>( this->parent() );
    return w > wd->width() ? w : wd->width();
}

int AiView::heightView(int h)
{
    QWidget * wd = qobject_cast<QWidget *>( this->parent() );
    return h > wd->height() ? h : wd->height();
}

void AiView::zooming(bool enable)
{
    m_isZooming = enable;
}

void AiView::fillToWindow()
{
    m_fill2Win = true;
    this->update();
}

void AiView::fillInWindow()
{
    m_fill2Win = false;
    this->update();
}

AiView::~AiView()
{
}
