/*Copyright (c) 2010 Maxime Petitjean

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.*/

#ifndef _FRACTAL_H_
#define _FRACTAL_H_

#include "fractalparameters.h"

#include "palette.h"

#include <QtCore/QThread>
#include <QtCore/QFile>
#include <QtGui/QImage>

/**
 * Interface d'une fractale.
 * Une fractale est donc un thread. On peut stopper son calcul, la transformer en image, la charger ou l'enregistrer depuis un fichier, ou encore avoir sa progression dans son calcul.
 */
class Fractal : public QThread
{
protected:
	FractalParameters m_fractal_parameters;
	bool m_is_valid, m_stop;

public:
	/**
	 * \brief Constructeur.
	 * \param _fractal_parameters Paramtres de la fractale.
	 */
	Fractal( const FractalParameters& _fractal_parameters )
		:	m_fractal_parameters( _fractal_parameters )
		,	m_is_valid( true )
		,	m_stop( false )
	{
	}

	/**
	 * \brief Accs au paramtres de la fractale.
	 * \return Paramtres de la fractale.
	 */
	const FractalParameters& fractalParameters() const { return m_fractal_parameters; }
	/**
	 * \brief Accs  la validit de la fractale.
	 * \return True si la fractale est valide.
	 */
	bool isValid() const { return m_is_valid; }
	/**
	 * \brief Accs.
	 * \return True si la construction de la fractale a t stoppe.
	 */
	bool isUserStopped() const { return m_stop; }
	
	/**
	 * \brief Retourne la progression dans le calcul de la fractale.
	 * \return Progression du calcul [0.0, 1.0].
	 */
	virtual qreal progress() const = 0;
	/**
	 * \brief Stoppe le calcul de la fractale.
	 */
	virtual void stop()
	{
		m_stop = true;
	}

	/**
	 * \brief Gnre une image  partir de la fractale.
	 * \param _palette Palette de couleurs  utiliser pour colorer l'image.
	 * \return Image gnre.
	 */
	virtual QImage toImage( const Palette& _palette ) const = 0;
	
	/**
	 * \brief Charge une fracale depuis un fichier.
	 * \param _file Fichier ouvert en lecture.
	 * \return 0 si russite, sinon code d'erreur
	 */
	virtual int load( QFile& _file ) = 0;
	/**
	 * \brief Sauvegarde une fracale dans un fichier.
	 * \param _file Fichier ouvert en criture.
	 * \return 0 si russite, sinon code d'erreur
	 */
	virtual int save( QFile& _file ) const = 0;

};

#endif //_FRACTAL_H_