/*Copyright (c) 2010 Maxime Petitjean

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.*/

#include "accumulator.h"

Accumulator::Accumulator( const FractalParameters * _fractal_parameters )
{
	m_fractal_parameters = _fractal_parameters;

	m_inv_x_step = m_fractal_parameters->imageSize().width() / m_fractal_parameters->zone().width();
	m_inv_y_step = m_fractal_parameters->imageSize().height() / m_fractal_parameters->zone().height();

	m_buffer_size = m_fractal_parameters->imageSize().height() * m_fractal_parameters->imageSize().width();
	m_buffer.resize( m_buffer_size );
	
	//Mise  zero du buffer
	for( unsigned int i = 0 ; i < m_buffer_size ; ++i )
		m_buffer[i] = 0;
}

QImage Accumulator::toImage( const Palette& _palette ) const
{
	//Calcul de la valeur maximum
	unsigned int max_buffer_value = 0;
	for( unsigned int i = 0 ; i < m_buffer_size ; ++i )
	{
		if( m_buffer[i] > max_buffer_value )
			max_buffer_value = m_buffer[i];
	}
	
	QImage palette( _palette.generateImage( max_buffer_value+1 , 1 ) );
	QImage image( m_fractal_parameters->imageSize(), QImage::Format_ARGB32_Premultiplied );

	if( max_buffer_value > 0 )
	{
		for( unsigned int i = 0 ; i < (unsigned int)m_fractal_parameters->imageSize().height() ; ++i )
		{
			for( unsigned int j = 0 ; j < (unsigned int)m_fractal_parameters->imageSize().width() ; ++j )
			{
				const unsigned int value = m_buffer[i * m_fractal_parameters->imageSize().width() + j];

				image.setPixel( j, i, palette.pixel( value, 0 ) );
			}
		}
	}
	
	return image;
}

void Accumulator::accumulate( const Accumulator& _accumulator )
{
	const unsigned int end = qMin( m_buffer.size(), _accumulator.m_buffer.size() );
	for( unsigned int i = 0 ; i < end ; ++i )
	{
		m_buffer[ i ] += _accumulator.m_buffer[ i ];
	}
}

void Accumulator::accumulateAt( const unsigned int _position, const Accumulator& _accumulator )
{
	Q_ASSERT( _position <  (unsigned int)m_buffer.size() );

	const unsigned int end = qMin( (unsigned int)(m_buffer.size() - _position), (unsigned int)_accumulator.m_buffer.size() );
	for( unsigned int i = 0 ; i < end ; ++i )
	{
		m_buffer[ i + _position] += _accumulator.m_buffer[ i ];
	}
}

int Accumulator::load( QFile& _file )
{
	if( _file.read( (char*)m_buffer.data(), m_buffer_size * sizeof( unsigned int ) ) != m_buffer_size * sizeof( unsigned int ) )
		return 1;
	return 0;
}
int Accumulator::save( QFile& _file ) const
{
	if( _file.write( (const char*)m_buffer.constData(), m_buffer_size * sizeof( unsigned int ) ) != m_buffer_size * sizeof( unsigned int ) )
		return 1;
	return 0;
}

unsigned int Accumulator::bufferPosition( const Point& _point )
{
	unsigned int x = ( _point.x() - m_fractal_parameters->zone().x() ) * m_inv_x_step;
	unsigned int y = ( _point.y() - m_fractal_parameters->zone().y() ) * m_inv_y_step;
	return y * m_fractal_parameters->imageSize().width() + x;
}
