/*Copyright (c) 2010 Maxime Petitjean

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.*/

#ifndef _ACCUMULATOR_H_
#define _ACCUMULATOR_H_

#include "sequence.h"
#include "../fractalparameters.h"
#include "../palette.h"

#include <QtCore/QFile>
#include <QtGui/QImage>

/**
 * Classe mre des accumulateurs.
 * Un accumulateur permet d'accumuler les squences gnres dans un buffer qui ensuite pourra tre transform en image.
 * Classe abstraite.
 */
class Accumulator
{
protected:
	FractalParameters const * m_fractal_parameters;
	QVarLengthArray<unsigned int> m_buffer;

	qreal m_inv_x_step;
	qreal m_inv_y_step;
	unsigned int m_buffer_size;

public:
	/**
	 * \brief Constructeur.
	 * \param _fractal_parameters Pointeur vers les paramtres de la fractale.
	 */
	Accumulator( const FractalParameters * _fractal_parameters );

	/**
	 * \brief Construit une image  partir de l'accumulateur.
	 * \param _palette Palette de couleurs  utiliser pour colorer l'image.
	 * \return Image gnre.
	 */
	QImage toImage( const Palette& _palette ) const;

	/**
	 * \brief Accumule une squence.
	 * \param _init_point Point d'initialisation.
	 * \param _sequence Squence  accumuler.
	 */
	virtual void accumulate( const Point& _init_point, const PointsSequence& _sequence ) = 0;
	/**
	 * \brief Accumule un accumulateur.
	 * \param _accumulator Accumulator  accumuler.
	 */
	virtual void accumulate( const Accumulator& _accumulator );
	/**
	 * \brief Accumule un accumulateur  une position.
	 * \param _position Position de dpart.
	 * \param _accumulator Accumulator  accumuler.
	 */
	virtual void accumulateAt( const unsigned int _position, const Accumulator& _accumulator );

	/**
	 * \brief Charge un accumulateur depuis un fichier.
	 * \param _file Fichier ouvert en lecture.
	 * \return 0 si russite, sinon code d'erreur
	 */
	virtual int load( QFile& _file );
	/**
	 * \brief Sauvegarde un accumulateur dans un fichier.
	 * \param _file Fichier ouvert en criture.
	 * \return 0 si russite, sinon code d'erreur
	 */
	virtual int save( QFile& _file ) const;

protected:
	/**
	 * \brief Retourne la position dans le buffer de l'accumulateur en fonction du point.
	 * \param _point Point.
	 * \return Position dans le buffer.
	 */
	unsigned int bufferPosition( const Point& _point );

};

#endif //_ACCUMULATOR_H_