/*Copyright (c) 2010 Maxime Petitjean

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.*/

#ifndef _FRACTAL_GL_H_
#define _FRACTAL_GL_H_

#include "../fractal.h"

#include <QtOpenGL/QGLPixelBuffer>
#include <QtOpenGL/QGLShaderProgram>

static QGLPixelBuffer* s_pixel_buffer = NULL;
static QGLShaderProgram* s_shader_program = NULL;
static GLuint s_texture_id;


/**
 * Gnration d'une fractale grce au shader.
 * Cette classe utilise des variables statiques afin de coller  l'interface des fractales (toImage en const). La fractale est entirement recalcule lorsqu'on la transforme en image.
 */
class Fractal_GL_Mother : public Fractal
{
public:
	Fractal_GL_Mother( const FractalParameters& _fractal_parameters )
		:	Fractal( _fractal_parameters )
	{
		if( s_pixel_buffer != NULL ) delete s_pixel_buffer;
		if( s_shader_program != NULL ) delete s_shader_program;	
		s_pixel_buffer = new QGLPixelBuffer( m_fractal_parameters.imageSize() );
		s_shader_program = new QGLShaderProgram();
		s_pixel_buffer->makeCurrent();

		//Set projection
		glMatrixMode( GL_PROJECTION );
		glLoadIdentity();
#ifdef QT_OPENGL_ES
		glOrthof(0.0, +1.0, 0.0, +1.0, -90.0, +90.0);
#else
		glOrtho(0.0, +1.0, 0.0, +1.0, -90.0, +90.0);
#endif
		glMatrixMode(GL_MODELVIEW);

		glEnable( GL_TEXTURE_1D );
		glGenTextures( 1, &s_texture_id );
		glBindTexture( GL_TEXTURE_1D, s_texture_id);
		glTexParameteri( GL_TEXTURE_1D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );
		glTexParameteri( GL_TEXTURE_1D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
		glTexParameteri( GL_TEXTURE_1D, GL_TEXTURE_WRAP_S, GL_REPEAT );
	}

	virtual qreal progress() const
	{
		return 1.0;
	}

	virtual QImage toImage( const Palette& _palette ) const
	{
		s_pixel_buffer->makeCurrent();;
		
		glTexImage1D(GL_TEXTURE_1D, 0, GL_RGBA8, 256, 0, GL_RGBA, GL_UNSIGNED_BYTE, QGLWidget::convertToGLFormat( _palette.generateImage( 256, 1 ) ).bits() );	

		s_shader_program->bind();
		s_shader_program->setUniformValue( "sequence_max", (GLfloat)m_fractal_parameters.sequenceMax() );

		glBegin( GL_QUADS );
		
		glTexCoord2f( m_fractal_parameters.zone().x()										, m_fractal_parameters.zone().y() + m_fractal_parameters.zone().height() );
		glVertex3f( 0.0f, 0.0f, 0.0f );
		glTexCoord2f( m_fractal_parameters.zone().x()										, m_fractal_parameters.zone().y() );
		glVertex3f( 0.0f, 1.0f, 0.0f );
		glTexCoord2f( m_fractal_parameters.zone().x() + m_fractal_parameters.zone().width()	, m_fractal_parameters.zone().y() );
		glVertex3f( 1.0f, 1.0f, 0.0f );
		glTexCoord2f( m_fractal_parameters.zone().x() + m_fractal_parameters.zone().width()	, m_fractal_parameters.zone().y() + m_fractal_parameters.zone().height() );
		glVertex3f( 1.0f, 0.0f, 0.0f );

		glEnd();

		return s_pixel_buffer->toImage();
	}

	virtual int load( QFile& )
	{
		return -1; //Impossible de charger une fractale, car c'est un type dont la sauvegarde n'est pas gre
	}
	virtual int save( QFile& ) const
	{
		return -1; //Impossible de sauvegarder une fractale gnre  partir d'un shader
	}

protected:
	virtual void run()
	{
		quit();
	}
};


/**
 * Gnration d'une fractale grce au shader.
 * Cas gnral.
 * Impossible de crer une fractale dans le cas gnral car le shader est inconnu.
 */
template< typename TSequence >
class Fractal_GL : public Fractal_GL_Mother
{
public:
	Fractal_GL( const FractalParameters& _fractal_parameters )
		:	Fractal_GL_Mother( _fractal_parameters )
	{
		m_is_valid = false; //Ce type de shader n'est pas connu, impossible de gnrer cette fractale
	}
};


/**
 * Gnration d'une fractale grce au shader.
 * Cas de Mandelbrot.
 * Utilisation du shader mandelbrot.fsl
 */
template<  >
class Fractal_GL< Mandelbrot > : public Fractal_GL_Mother
{
public:
	Fractal_GL( const FractalParameters& _fractal_parameters )
		:	Fractal_GL_Mother( _fractal_parameters )
	{
		s_shader_program->addShaderFromSourceFile( QGLShader::Fragment, "shaders/mandelbrot.fsl" );
		s_shader_program->link();
	}
};


#endif //_FRACTAL_GL_H_