/*Copyright (c) 2010 Maxime Petitjean

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.*/

template< typename TSequence, typename TPointsGenerator, typename TAccumulator >
Fractal_MT_Mother< TSequence, TPointsGenerator, TAccumulator >::Fractal_MT_Mother( const FractalParameters& _fractal_parameters, const unsigned int _thread_count )
	:	Fractal( _fractal_parameters )
	,	m_fractal_result( _fractal_parameters )
	,	m_thread_count( _thread_count )
{
	Q_ASSERT( _thread_count > 0 );
	m_fractal_threads.resize( _thread_count );
	m_is_thread_not_generated = true;
	m_is_thread_deleted = false;
}

template< typename TSequence, typename TPointsGenerator, typename TAccumulator >
qreal Fractal_MT_Mother< TSequence, TPointsGenerator, TAccumulator >::progress() const
{
	if( m_is_thread_not_generated ) return 0.0;
	if( m_is_thread_deleted ) return 1.0;

	qreal sum = 0.0;
	for( unsigned int i = 0 ; i < m_thread_count ; ++i )
	{
		sum += m_fractal_threads[i]->progress();
	}
	return sum / m_thread_count;
}

template< typename TSequence, typename TPointsGenerator, typename TAccumulator >
void Fractal_MT_Mother< TSequence, TPointsGenerator, TAccumulator >::stop()
{
	m_stop = true;
	for( unsigned int i = 0 ; i < m_thread_count ; ++i )
	{
		m_fractal_threads[i]->stop();
	}
	for( unsigned int i = 0 ; i < m_thread_count ; ++i )
	{
		m_fractal_threads[i]->wait();
	}
}

template< typename TSequence, typename TPointsGenerator, typename TAccumulator >
QImage Fractal_MT_Mother< TSequence, TPointsGenerator, TAccumulator >::toImage( const Palette& _palette ) const
{
	return m_fractal_result.toImage( _palette );
}

template< typename TSequence, typename TPointsGenerator, typename TAccumulator >
int Fractal_MT_Mother< TSequence, TPointsGenerator, TAccumulator >::load( QFile& _file )
{
	return m_fractal_result.load( _file );
}
template< typename TSequence, typename TPointsGenerator, typename TAccumulator >
int Fractal_MT_Mother< TSequence, TPointsGenerator, TAccumulator >::save( QFile& _file ) const
{
	return m_fractal_result.save( _file );
}


template< typename TSequence, typename TPointsGenerator, typename TAccumulator >
void Fractal_MT_Mother< TSequence, TPointsGenerator, TAccumulator >::generateThreads()
{
	FractalParameters fractal_parameters( m_fractal_parameters );
	fractal_parameters.setPointsNumber( m_fractal_parameters.pointsNumber() / m_thread_count );
	
	for( unsigned int i = 0 ; i < m_thread_count ; ++i )
	{
		m_fractal_threads[i] = new TFractal_Thread( fractal_parameters );
	}
}

template< typename TSequence, typename TPointsGenerator, typename TAccumulator >
void Fractal_MT_Mother< TSequence, TPointsGenerator, TAccumulator >::generateFractalResult()
{
	for( unsigned int i = 0 ; i < m_thread_count ; ++i )
	{
		m_fractal_result.accumulate( *m_fractal_threads[i] );
	}
}

template< typename TSequence, typename TPointsGenerator, typename TAccumulator >
void Fractal_MT_Mother< TSequence, TPointsGenerator, TAccumulator >::run()
{
	//Cration des threads
	generateThreads();
	m_is_thread_not_generated = false;
	
	//Lancement des threads
	for( unsigned int i = 0 ; i < m_thread_count ; ++i )
	{
		m_fractal_threads[i]->start();
	}
	
	//Attente des threads
	for( unsigned int i = 0 ; i < m_thread_count ; ++i )
	{
		m_fractal_threads[i]->wait();
	}

	//Gnration de la fractale rsultante
	generateFractalResult();

	
	//Suppression des threads
	m_is_thread_deleted = true;
	for( unsigned int i = 0 ; i < m_thread_count ; ++i )
	{
		delete m_fractal_threads[i];
	}
}




template< typename TSequence, typename TPointsGenerator, typename TAccumulator >
Fractal_MT< TSequence, TPointsGenerator, TAccumulator >::Fractal_MT( const FractalParameters& _fractal_parameters, const unsigned int _thread_count )
	:	Fractal_MT_Mother< TSequence, TPointsGenerator, TAccumulator >( _fractal_parameters, _thread_count )
{
}



//Spcialistation partielle pour le cas d'une gnration des points  partir de l'image et d'une accumulation au centre
template< typename TSequence >
Fractal_MT< TSequence, ImagePointsGenerator, CenterAccumulator >::Fractal_MT( const FractalParameters& _fractal_parameters, const unsigned int _thread_count )
	:	Fractal_MT_Mother< TSequence, ImagePointsGenerator, CenterAccumulator >( _fractal_parameters, _thread_count )
{
}

template< typename TSequence >
void Fractal_MT< TSequence, ImagePointsGenerator, CenterAccumulator >::generateThreads()
{
	const unsigned int width	= this->m_fractal_parameters.imageSize().width();
	const unsigned int height	= this->m_fractal_parameters.imageSize().height();
	FractalParameters fractal_parameters( this->m_fractal_parameters );
	Zone zone( fractal_parameters.zone() );

	//Dfinition des premier threads
	zone.setHeight( ( zone.height() * ( (height / this->m_thread_count) * this->m_thread_count ) / (qreal)height ) / this->m_thread_count );
	fractal_parameters.setImageSize( QSize( width, height / this->m_thread_count ) );
	for( unsigned int i = 0 ; i < this->m_thread_count - 1 ; ++i )
	{
		fractal_parameters.setZone( zone );
		this->m_fractal_threads[i] = new Fractal_Thread< TSequence, ImagePointsGenerator, CenterAccumulator >( fractal_parameters );
		zone.setY( zone.y() + zone.height() );
	}

	//Le dernier thread doit aller jusqu' la fin de la zone et de l'image (dans le cas ou le nombre de thread n'est pas diviseur de la hauteur de l'image)
	zone.setHeight( this->m_fractal_parameters.zone().height() - zone.height() * (this->m_thread_count - 1) );
	fractal_parameters.setImageSize( QSize( width, height - ( (height / this->m_thread_count) * (this->m_thread_count - 1) ) ) );
	fractal_parameters.setZone( zone );
	this->m_fractal_threads[this->m_thread_count - 1] = new Fractal_Thread< TSequence, ImagePointsGenerator, CenterAccumulator >( fractal_parameters );
}

template< typename TSequence >
void Fractal_MT< TSequence, ImagePointsGenerator, CenterAccumulator >::generateFractalResult()
{
	for( unsigned int i = 0 ; i < this->m_thread_count ; ++i )
	{
		const unsigned int buffer_size = this->m_fractal_threads[i]->fractalParameters().imageSize().width() * this->m_fractal_threads[i]->fractalParameters().imageSize().height();
		this->m_fractal_result.accumulateAt( i * buffer_size , *this->m_fractal_threads[i] );
	}
}