/*Copyright (c) 2010 Maxime Petitjean

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.*/

#ifndef _SEQUENCE_H_
#define _SEQUENCE_H_

#include "point.h"
#include "../fractalparameters.h"

#include <QtCore/QVarLengthArray>

typedef QVarLengthArray<Point> PointsSequence;

/**
 * Gnration d'une squence.
 * Classe abstraite.
 */
class Sequence
{
protected:
	PointsSequence m_sequence;
	const FractalParameters * m_fractal_parameters;
	Point m_point;

public:
	/**
	 * \brief Constructeur.
	 * \param _fractal_parameters Pointeur vers les paramtres de la fractale.
	 */
	Sequence( const FractalParameters * _fractal_parameters )
	{
		m_fractal_parameters = _fractal_parameters;
		m_sequence.reserve( _fractal_parameters->sequenceMax() );
	}

	/**
	 * \brief Calcul la squence entire.
	 * \param _init_point Point de dpart.
	 * \return Rfrence sur la squence gnre.
	 */
	virtual const PointsSequence& compute( const Point& _init_point )
	{
		m_sequence.clear();

		Point point = init( _init_point );
			
		unsigned int index_sequence = 0;
		while( ! end( point ) && index_sequence < m_fractal_parameters->sequenceMax() )
		{
			m_sequence.append( point );
			index_sequence++;
			point = next( point );
		}

		if( index_sequence == m_fractal_parameters->sequenceMax() )
			m_sequence.clear();
		return m_sequence;

	}

protected:
	/**
	 * \brief Initialise la squence.
	 * \param _init_point Point de dpart.
	 * \return Premier point de la squence.
	 */
	virtual Point init( const Point& _init_point ) = 0;
	/**
	 * \brief Retourne le prochain point de la squence.
	 * \param _point Point actuel.
	 * \return Prochain point.
	 */
	virtual Point next( const Point& _point ) = 0;
	/**
	 * \brief Retourne si la squence est termine.
	 * \param _point Point actuel.
	 * \return True si la squence est termine.
	 */
	virtual bool end( const Point& _point ) = 0;
};

#endif //_SEQUENCE_H_