/*Copyright (c) 2010 Maxime Petitjean

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.*/
#define CIRCLE_RADIUS 8
#define BORDERS_SIZE 2

#include "paletteeditor.h"

#include <QtGui/QPainter>
#include <QtGui/QColorDialog>

PaletteEditor::PaletteEditor( QWidget* _parent, Qt::WFlags _flags )
	:	QFrame( _parent, _flags )
{
	setMinimumHeight( 2 * ( CIRCLE_RADIUS + BORDERS_SIZE + 1 ) );
	setFrameStyle( QFrame::StyledPanel );

	m_move_index = -1;
}

void PaletteEditor::paintEvent( QPaintEvent * _event )
{
	QPainter painter( this );
	QLinearGradient gradient( m_palette.linearGradient() );
	gradient.setStart( BORDERS_SIZE, BORDERS_SIZE );
	gradient.setFinalStop( (qreal)(width()-BORDERS_SIZE), 0.0 );
	painter.fillRect( BORDERS_SIZE, BORDERS_SIZE, width()-2*BORDERS_SIZE, height()-2*BORDERS_SIZE, gradient );


	QPainterPath painter_path;
	const QGradientStops& stops = m_palette.linearGradient().stops();
	for( int i = 0 ; i < stops.size() ; ++i )
	{
		addEllipse( painter_path, stops[i].first );
	}

	painter.setRenderHint( QPainter::Antialiasing );
	painter.fillPath( painter_path, Qt::white );
	painter.drawPath( painter_path );
	
	QFrame::paintEvent( _event );
}

void PaletteEditor::mouseMoveEvent( QMouseEvent* _event )
{
	if( m_move_index != -1 )
	{
		qreal next_position = (_event->x()-BORDERS_SIZE) / (qreal)(width()-2*BORDERS_SIZE);
		if( next_position <= m_palette.colorPosition( m_move_index - 1 ) )
		{
			next_position = ( m_palette.colorPosition( m_move_index - 1 ) * (width()-2*BORDERS_SIZE) + 1 ) / (qreal)(width()-2*BORDERS_SIZE);
		}
		else if( next_position >= m_palette.colorPosition( m_move_index + 1 ) )
		{
			next_position = ( m_palette.colorPosition( m_move_index + 1 ) * (width()-2*BORDERS_SIZE) - 1 ) / (qreal)(width()-2*BORDERS_SIZE);
		}

		m_palette.setColorPosition( m_move_index, next_position );
		repaint();
		emit paletteChanged( m_palette );
	}
}
void PaletteEditor::mousePressEvent( QMouseEvent* _event )
{
	int index = -1;

	const QGradientStops& stops = m_palette.linearGradient().stops();
	for( int i = 0 ; i < stops.size() ; ++i )
	{
		QPainterPath painter_path;
		addEllipse( painter_path, stops[i].first );
		if( painter_path.contains( _event->pos() ) )
		{
			index = i;
		}
	}

	
	if( _event->button() == Qt::LeftButton )
	{
		if( index == -1 )
		{
			QColor color = QColorDialog::getColor( Qt::white, this, tr( "Selectionnez une couleur" ), QColorDialog::ShowAlphaChannel );
			if( color.isValid() )
			{
				m_palette.setColor( (_event->x()-BORDERS_SIZE) / (qreal)(width()-2*BORDERS_SIZE), color );
				emit paletteChanged( m_palette );
			}
		}
		else
		{
			if( index != 0 && index != stops.size()-1 ) //on ne peut pas dplacer la premire et la dernire couleur
			{
				m_move_index = index;
			}
			else if( index == stops.size()-1 && stops.size() > 2 ) //afin de pouvoir rcuprer facilement un cercle sous le dernier
			{
				QPainterPath painter_path;
				addEllipse( painter_path, stops[stops.size()-2].first );
				if( painter_path.contains( _event->pos() ) )
				{
					m_move_index = stops.size()-2;
				}
			}
		}
	}
	else if( _event->button() == Qt::RightButton )
	{
		if( index != -1 )
		{
			QColor color = QColorDialog::getColor( stops[index].second, this, tr( "Selectionnez une couleur" ), QColorDialog::ShowAlphaChannel );
			if( color.isValid() )
			{
				m_palette.setColorAt( index, color );
				emit paletteChanged( m_palette );
			}
		}
	}
	else if( _event->button() == Qt::MidButton )
	{
		if( index != -1 )
		{
			if( index != 0 && index != stops.size()-1 ) //on ne peut pas supprimer la premire et la dernire couleur
			{
				m_palette.removeColorAt( index );
				m_move_index = -1;
				emit paletteChanged( m_palette );
			}
		}
	}
	repaint();
}
void PaletteEditor::mouseReleaseEvent( QMouseEvent* )
{
	m_move_index = -1;
}

void PaletteEditor::addEllipse( QPainterPath& _painter_path, qreal _position )
{
	_painter_path.addEllipse( QPointF( _position * (width()-2*BORDERS_SIZE) + BORDERS_SIZE, height()*0.5), CIRCLE_RADIUS, CIRCLE_RADIUS );
}