/*Copyright (c) 2010 Maxime Petitjean

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.*/

#include "pixmaplabel.h"

#include <QtGui/QPainter>
#include <QtGui/QWheelEvent>

PixmapLabel::PixmapLabel( QWidget *_parent, Qt::WFlags _flags )
	:	QWidget( _parent, _flags )
{
	resize( 100, 20 ); //Pour le texte "Pas d'image"
	m_scale_factor = 1.0;
	m_scale_min = 1.0 / 64.0;
	m_scale_max = 1.0 * 32.0;
	m_scale_step = 2.0;

	m_select = false;
}

void PixmapLabel::setPixmap( const QPixmap& _pixmap, qreal _scale_factor )
{
	m_pixmap = _pixmap;
	m_scale_factor = qBound( m_scale_min, _scale_factor, m_scale_max );
	resize( _scale_factor * _pixmap.size() );
	repaint();
}

void PixmapLabel::paintEvent( QPaintEvent * )
{
	if( ! m_pixmap.isNull() )
	{
		int xoffset = 0, yoffset = 0;
		if( width() > m_pixmap.width() * m_scale_factor )
			xoffset = (int) ( width() - m_pixmap.width() * m_scale_factor ) / 2;
		if( height() > m_pixmap.height() * m_scale_factor )
			yoffset = (int) ( height() - m_pixmap.height() * m_scale_factor) / 2;

	
		QPainter p( this );
		if( m_scale_factor < 1.0 )
			p.setRenderHint( QPainter::SmoothPixmapTransform );
		p.save();
		p.translate( xoffset, yoffset );
		p.scale( m_scale_factor, m_scale_factor );
		p.drawPixmap( 0, 0, m_pixmap );
		p.restore();

		if( m_select )
		{
			QPen pen( Qt::white );
			pen.setStyle( Qt::DashLine );
			p.setPen( pen );
			p.drawRect( QRectF( m_select_start, m_select_end ) );
		}
	}
	else
	{
		QPainter p( this );
		p.drawText( 0 , 0, width(), height(), Qt::AlignCenter, tr( "Pas d'image." ) );
	}



}
void PixmapLabel::wheelEvent( QWheelEvent * _event )
{
	if( ! m_pixmap.isNull() )
	{
		if( _event->orientation() == Qt::Vertical)
		{
			const int num_degrees = _event->delta() / 8;
			const int num_steps = num_degrees / 15;

			if( num_steps < 0 )
			{
				if( m_scale_factor > m_scale_min )
				{
					const qreal scale_change = 1.0 / m_scale_step;
					m_scale_factor *= scale_change;
					emit updated( m_scale_factor, scale_change, _event->pos() );
				}
			}
			else
			{
				if( m_scale_factor < m_scale_max )
				{
					const qreal scale_change = m_scale_step;
					m_scale_factor *= scale_change;
					emit updated( m_scale_factor, scale_change, _event->pos() );
				}
			}
			repaint();
			_event->accept();
		}
	}
}


void PixmapLabel::mouseMoveEvent( QMouseEvent * _event )
{
	if( ( _event->buttons() & Qt::LeftButton ) != 0 )
	{
		int x = qBound( 0, _event->x(), width()-1 );
		int y = qBound( 0, _event->y(), height()-1 );
		m_select_end = QPoint( x, y );
		repaint();
	}
	_event->ignore();
}
void PixmapLabel::mousePressEvent( QMouseEvent * _event )
{
	if( _event->button() == Qt::LeftButton )
	{
		m_select_start = m_select_end = _event->pos();
		m_select = true;
	}
	_event->ignore();
}
void PixmapLabel::mouseReleaseEvent( QMouseEvent * _event )
{
	if( ( _event->button() == Qt::LeftButton ) != 0 )
	{
		m_select_end = _event->pos();
		if( m_select_start != m_select_end )
		{
			QPoint start, end;
			start.setX( qMin( m_select_start.x(), m_select_end.x() ) );
			start.setY( qMin( m_select_start.y(), m_select_end.y() ) );
			end.setX( qMax( m_select_start.x(), m_select_end.x() ) );
			end.setY( qMax( m_select_start.y(), m_select_end.y() ) );

			emit selected( QPointF( start / m_scale_factor ), QPointF( end / m_scale_factor ) );
		}
		m_select = false;
		repaint();
	}
	_event->ignore();
}


void PixmapLabel::updatePixmap( const QPixmap& _pixmap )
{
	m_pixmap = _pixmap;
	repaint();
}

void PixmapLabel::updateScale( qreal _scale_factor )
{
	m_scale_factor = _scale_factor;
	resize( _scale_factor * m_pixmap.size() );
}