/*Copyright (c) 2010 Maxime Petitjean

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.*/

#include "pixmapviewer.h"

#include <QtGui/QMouseEvent>

PixmapViewer::PixmapViewer( QWidget *_parent, Qt::WFlags _flags )
	:	QWidget( _parent, _flags )
{
	m_pixmap_label = new PixmapLabel( this, _flags );
	connect( m_pixmap_label, SIGNAL( updated( qreal, qreal, QPoint ) ), this, SLOT( update( qreal, qreal, QPoint ) ) );
	connect( m_pixmap_label, SIGNAL( selected( QPointF, QPointF ) ), this, SIGNAL( selected( QPointF, QPointF ) ) );

	m_scroll_area = new QScrollArea();
	m_scroll_area->setBackgroundRole( QPalette::Dark );
	m_scroll_area->setAlignment( Qt::AlignCenter );
	m_scroll_area->setWidget( m_pixmap_label );

	connect( m_scroll_area->horizontalScrollBar(), SIGNAL( valueChanged( int ) ), this, SLOT( scrollBar_valueChanged( int ) ) );
	connect( m_scroll_area->verticalScrollBar(), SIGNAL( valueChanged( int ) ), this, SLOT( scrollBar_valueChanged( int ) ) );

	m_layout = new QVBoxLayout();
	m_layout->setContentsMargins( 0, 0, 0, 0 );
	m_layout->addWidget( m_scroll_area );
	setLayout( m_layout );
}
PixmapViewer::~PixmapViewer()
{
	delete m_pixmap_label;
	delete m_scroll_area;
	delete m_layout;	
}

void PixmapViewer::setPixmap(	const QPixmap& _pixmap	, unsigned int _preferred_width, unsigned int _preferred_height, bool _adapt )
{
	double scale_factor = 1.0;

	if( _adapt )
	{
		if( _preferred_width == 0 || _preferred_height == 0 )
		{
			_preferred_width = width();
			_preferred_height = height();
		}
		while( _pixmap.height() * scale_factor					>	( _preferred_height - 4 ) || _pixmap.width() * scale_factor					>	( _preferred_width - 4 ) ) scale_factor /= scaleStep();
		while( _pixmap.height() * scale_factor * scaleStep()	<=	( _preferred_height - 4 ) && _pixmap.width() * scale_factor * scaleStep()	<=	( _preferred_width - 4 ) ) scale_factor *= scaleStep();
	}

	m_pixmap_label->setPixmap( _pixmap, scale_factor );
	
	emit pixmapChanged( _pixmap );
	emit scaleChanged( scale_factor );
	emit positionChanged( QPoint( m_scroll_area->horizontalScrollBar()->value(), m_scroll_area->verticalScrollBar()->value() ) );
}

void PixmapViewer::mouseMoveEvent( QMouseEvent * _event )
{
	if( ( _event->buttons() & Qt::RightButton ) != 0 )
	{
		int x_delta = _event->x() - m_x_save;
		int y_delta = _event->y() - m_y_save;
		m_x_save = _event->x();
		m_y_save = _event->y();

		updatePosition( QPoint( m_scroll_area->horizontalScrollBar()->value() - x_delta, m_scroll_area->verticalScrollBar()->value() - y_delta ) );

		emit positionChanged( QPoint( m_scroll_area->horizontalScrollBar()->value(), m_scroll_area->verticalScrollBar()->value() ) );
	}
}

void PixmapViewer::mousePressEvent( QMouseEvent * _event )
{
	if( ( _event->buttons() & Qt::RightButton ) != 0 )
	{
		m_x_save = _event->x();
		m_y_save = _event->y();

		setCursor( Qt::PointingHandCursor );
	}
}

void PixmapViewer::mouseReleaseEvent( QMouseEvent * )
{
	setCursor( Qt::ArrowCursor );
}


void PixmapViewer::updatePixmap( const QPixmap& _pixmap )
{
	m_pixmap_label->updatePixmap( _pixmap );
	emit pixmapChanged( m_pixmap_label->pixmap() );
}

void PixmapViewer::updateScale( qreal _scale_factor )
{
	m_pixmap_label->updateScale( _scale_factor );
}

void PixmapViewer::updatePosition( const QPoint& _scroll )
{
	QScrollBar* scroll_bar;
	scroll_bar = m_scroll_area->horizontalScrollBar();
	scroll_bar->setValue( _scroll.x() );
	scroll_bar = m_scroll_area->verticalScrollBar();
	scroll_bar->setValue( _scroll.y() );
}


void PixmapViewer::update( qreal _scale_factor, qreal _scale, QPoint _position )
{
	int scroll_x, scroll_y;
	QScrollBar *x_scroll_bar, *y_scroll_bar;
	x_scroll_bar = m_scroll_area->horizontalScrollBar();
	y_scroll_bar = m_scroll_area->verticalScrollBar();
	
	scroll_x = (int) ( _position.x() * _scale - ( _position.x() - x_scroll_bar->value() ) );
	scroll_y = (int) ( _position.y() * _scale - ( _position.y() - y_scroll_bar->value() ) );
	
	updateScale( _scale_factor );
	updatePosition( QPoint( scroll_x, scroll_y ) );

	emit scaleChanged( _scale_factor );
	emit positionChanged( QPoint( m_scroll_area->horizontalScrollBar()->value(), m_scroll_area->verticalScrollBar()->value() ) );
}
void PixmapViewer::scrollBar_valueChanged( int )
{
	emit positionChanged( QPoint( m_scroll_area->horizontalScrollBar()->value(), m_scroll_area->verticalScrollBar()->value() ) );
}