/*Copyright (c) 2010 Maxime Petitjean

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.*/

#ifndef _PIXMAPVIEWER_H_
#define _PIXMAPVIEWER_H_

#include "pixmaplabel.h"

#include <QtGui/QScrollArea>
#include <QtGui/QScrollBar>
#include <QtGui/QVBoxLayout>

/**
 * Gestion de l'affichage d'un visionneur d'image. Possibilit de zoomer (roulette), de se dplacer (clic droit), de slectionner (clic gauche) une partie de l'image.
 */
class PixmapViewer : public QWidget
{
	Q_OBJECT

private:
	PixmapLabel *m_pixmap_label;
	QScrollArea *m_scroll_area;

	int m_x_save;
	int m_y_save;

	QVBoxLayout *m_layout;

public:
	/**
	 * \brief Constructeur.
	 * \param _parent Widget parent.
	 * \param _flags Flags du widget.
	 */
	PixmapViewer( QWidget* _parent = 0, Qt::WFlags _flags = 0 );
	/**
	 * \brief Destructeur.
	 * \param _flags Flags du widget.
	 */
	~PixmapViewer();

	/**
	 * \brief Accs.
	 * \return Facteur de zoom.
	 */
	qreal			scaleFactor()	const { return m_pixmap_label->scaleFactor(); }
	/**
	 * \brief Accs.
	 * \return Image affiche.
	 */
	const QPixmap&	pixmap()		const { return m_pixmap_label->pixmap(); }
	/**
	 * \brief Accs.
	 * \return Position de l'image.
	 */
	QPoint			position()		const { return QPoint( m_scroll_area->horizontalScrollBar()->value(), m_scroll_area->verticalScrollBar()->value() ); }
	/**
	 * \brief Accs.
	 * \return Zoom minimum.
	 */
	qreal			scaleMin()		const { return m_pixmap_label->scaleMin(); }
	/**
	 * \brief Accs.
	 * \return Zoom maximum.
	 */
	qreal			scaleMax()		const { return m_pixmap_label->scaleMax(); }
	/**
	 * \brief Accs.
	 * \return Le pas dans le changement de zoom.
	 */
	qreal			scaleStep()		const { return m_pixmap_label->scaleStep(); }
	
	/**
	 * \brief Affiche une image.
	 * \param _pixmap Image  afficher.
	 * \param _preferred_width Largeur prfre pour l'image [0 pour laisser le visonneur dcider].
	 * \param _preferred_height Hauteur prfre pour l'image [0 pour laisser le visonneur dcider].
	 * \param _adapt True pour laisser le visionneur choisir le facteur du zoom afin que l'image entire soit visible ou qu'elle s'adapte  la taille prfre choisie.
	 */
	void setPixmap(	const QPixmap& _pixmap	, unsigned int _preferred_width = 0, unsigned int _preferred_height = 0, bool _adapt = true );
	/**
	 * \brief Accs.
	 * \param _scale_min Zoom minimum.
	 */
	void setScaleMin(	qreal _scale_min	) { m_pixmap_label->setScaleMin(	_scale_min	); }
	/**
	 * \brief Accs.
	 * \param _scale_max Zoom maximum.
	 */
	void setScaleMax(	qreal _scale_max	) { m_pixmap_label->setScaleMax(	_scale_max	); }
	/**
	 * \brief Accs.
	 * \param _scale_step Le pas dans le changement de zoom.
	 */
	void setScaleStep(	qreal _scale_step	) { m_pixmap_label->setScaleStep(	_scale_step	); }

	
protected:
	virtual void mouseMoveEvent( QMouseEvent * _event );
	virtual void mousePressEvent( QMouseEvent * _event );
	virtual void mouseReleaseEvent( QMouseEvent * _event );

public slots:
	/**
	 * \brief Met  jour l'image affiche sans rien changer d'autre.
	 * \param _pixmap Image.
	 */
	void updatePixmap( const QPixmap& _pixmap );
	/**
	 * \brief Met  jour le zoom.
	 * \param _scale_factor Facteur de zoom.
	 */
	void updateScale( qreal _scale_factor ); 
	/**
	 * \brief Met  jour la position.
	 * \param _position Position.
	 */
	void updatePosition( const QPoint& _position );

private slots:
	/**
	 * \brief Modification du zoom.
	 * \param _scale_factor Nouveau facteur de zoom.
	 * \param _scale Changement dans le facteur de zoom.
	 * \param _positionEndroit o la modification a eu lieu.
	 */
	void update( qreal _scale_factor, qreal _scale, QPoint _position );
	
	/**
	 * \brief Modification de la position de l'image par les scrollbars.
	 * \param _value Nouvelle valeur.
	 */
	void scrollBar_valueChanged( int _value );

signals:
	/**
	 * \brief Signal mis lors d'un changement de zoom.
	 * \param _scale_factor Nouveau facteur de zoom.
	 */
	void scaleChanged( qreal _scale_factor );
	/**
	 * \brief Signal mis lors d'une modification de position.
	 * \param _position Nouvelle position.
	 */
	void positionChanged( QPoint _position );
	/**
	 * \brief Signal mis lors d'une modification de l'image.
	 * \param _pixmap Nouvelle image.
	 */
	void pixmapChanged( const QPixmap& _pixmap );
	/**
	 * \brief Signal mis lors d'une slection.
	 * \param _select_start Point en haut  gauche de slection.
	 * \param _select_end Point en bas  droite de slection.
	 */
	void selected( QPointF _select_start, QPointF _select_end );



};

#endif //_PIXMAPVIEWER_H_
