/*Copyright (c) 2010 Maxime Petitjean

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.*/

#include "mainwindow.h"

#include <QtGui/QDesktopServices>
#include <QtCore/QTime>
#include <QtCore/QLibraryInfo>

MainWindow::MainWindow( QWidget *_parent, Qt::WFlags _flags )
	:	QMainWindow( _parent, _flags )
{
	ui.setupUi( this );

	m_main_widget = new QWidget(); 
	m_layout = new QVBoxLayout();
	setCentralWidget( m_main_widget );

	//Visionneur d'images
	m_pixmap_viewer = new PixmapViewer( this );
	connect( m_pixmap_viewer, SIGNAL( pixmapChanged( const QPixmap& ) ), this, SLOT( imageChanged( const QPixmap& ) ) );
	connect( m_pixmap_viewer, SIGNAL( scaleChanged( qreal ) ), this, SLOT( imageScaleChanged( qreal ) ) );
	connect( m_pixmap_viewer, SIGNAL( selected( QPointF, QPointF ) ), this, SLOT( imageSelected( QPointF, QPointF ) ) );

	//Docks
	m_options_widget = new OptionsWidget();
	addDockWidget( Qt::RightDockWidgetArea, m_options_widget );
	ui.views_menu->addAction( m_options_widget->toggleViewAction() );
	m_parameters_widget = new ParametersWidget();
	addDockWidget( Qt::RightDockWidgetArea, m_parameters_widget );
	ui.views_menu->addAction( m_parameters_widget->toggleViewAction() );
	m_palette_widget = new PaletteWidget();
	connect( m_palette_widget, SIGNAL( paletteChanged( const Palette& ) ), this, SLOT( updateFractalImage( const Palette& ) ) );
	addDockWidget( Qt::RightDockWidgetArea, m_palette_widget );
	ui.views_menu->addAction( m_palette_widget->toggleViewAction() );
	ui.views_menu->addSeparator();
	ui.views_menu->addAction( ui.mainToolBar->toggleViewAction() );

	//Status Bar
	m_fractal_type_label = new QLabel("");
	m_image_size_label = new QLabel("");
	m_image_scale_label = new QLabel("");
	m_status_bar = new QStatusBar( this );
	m_status_bar->addPermanentWidget( m_image_scale_label );
	m_status_bar->addPermanentWidget( m_image_size_label );
	m_status_bar->addPermanentWidget( m_fractal_type_label );
	setStatusBar( m_status_bar );
	
	//Computing Widgets
	m_compute_button = new QPushButton( "" );
	m_stop_button = new QPushButton( "" );
	m_stop_button->setEnabled( false );
	m_compute_progressBar = new QProgressBar();
	m_compute_progressBar->setRange( 0, 100 );
	m_compute_progressBar->setValue( 0 );
	m_compute_layout = new QHBoxLayout();
	m_compute_layout->addWidget( m_compute_progressBar );
	m_compute_layout->addWidget( m_compute_button );
	m_compute_layout->addWidget( m_stop_button );	
	connect( m_compute_button, SIGNAL( clicked() ), this, SLOT( computeClicked() ) );
	connect( m_stop_button, SIGNAL( clicked() ), this, SLOT( stopFractal() ) );

	//Langues
	m_translator = new QTranslator();
	m_global_translator = new QTranslator();
	setLanguageFr();
	connect( ui.action_fr, SIGNAL( triggered() ), this, SLOT( setLanguageFr() ) );
	connect( ui.action_en, SIGNAL( triggered() ), this, SLOT( setLanguageEn() ) );
	
	//Timer
	m_timer = new QTimer();
	connect( m_timer, SIGNAL( timeout() ), this, SLOT( updateProgress() ) );

	//Fractales
	m_current_fractal = NULL;
	m_computing_fractal = NULL;

	
	m_directory = ".";
	//QString image_directory =  QDesktopServices::storageLocation( QDesktopServices::PicturesLocation );
	//if( ! image_directory.isEmpty() ) m_directory = image_directory;
	
	m_layout->addWidget( m_pixmap_viewer, Qt::AlignCenter );
	m_layout->addLayout( m_compute_layout );
	m_main_widget->setLayout( m_layout );

	connect( ui.action_open, SIGNAL( triggered() ), this, SLOT( load() ) );
	connect( ui.action_save, SIGNAL( triggered() ), this, SLOT( save() ) );
	connect( ui.action_export, SIGNAL( triggered() ), this, SLOT( saveToImage() ) );
	connect( ui.action_about, SIGNAL( triggered() ), this, SLOT( about() ) );
	connect( ui.action_about_qt, SIGNAL( triggered() ), this, SLOT( aboutQt() ) );
}

MainWindow::~MainWindow()
{
	if( m_current_fractal != NULL )
		delete m_current_fractal;
}

void MainWindow::changeEvent( QEvent* _event )
{
	QMainWindow::changeEvent( _event );
	switch( _event->type() )
	{
    case QEvent::LanguageChange :
        ui.retranslateUi( this );
		m_status_bar->clearMessage();
		m_compute_button->setText( tr( "Calculer" ) );
		m_stop_button->setText( tr( "Stop" ) );
        break;
	default:
		break;
    }
}

bool MainWindow::allocFractal( Fractal*& _fractal, const FractalParameters& _fractal_parameters, const QString& _fractal_type, const bool _use_shaders, const unsigned int _thread_count )
{
	if( _fractal_type == "Mandelbrot" )
	{
		if( _use_shaders )
		{
			_fractal = new Fractal_GL<Mandelbrot>( _fractal_parameters );
		}
		else
		{
			_fractal = new Fractal_MT<Mandelbrot, ImagePointsGenerator, CenterAccumulator>( _fractal_parameters, _thread_count );
		}
	}
	else if( m_computing_fractal_type == "Buddhabrot" )
	{
		_fractal = new Fractal_MT<Mandelbrot, RandomPointsGenerator, SequenceAccumulator>( _fractal_parameters, _thread_count );
	}
	else
	{
		return false;
	}
	return true;
}

void MainWindow::computeClicked()
{
	m_compute_button->setEnabled( false );
	m_stop_button->setEnabled( true );
	computeFractal();
}

void MainWindow::computeFractal()
{
	m_computing_fractal_type = m_options_widget->typeName();
	m_computing_fractal_use_shaders = m_options_widget->useShaders();
	if( ! allocFractal( m_computing_fractal, m_parameters_widget->toFractalParameters(), m_computing_fractal_type, m_computing_fractal_use_shaders, m_options_widget->threadsCount() ) )
	{
		QMessageBox::warning( this, tr( "Erreur" ), tr( "Impossible de gnrer cette fractale." ) );
		return;
	}

	if( ! m_computing_fractal->isValid() )
	{
		QMessageBox::warning( this, tr( "Erreur" ), tr( "Impossible de gnrer cette fractale." ) );
		return;
	}

	connect( m_computing_fractal, SIGNAL( finished() ), this, SLOT( finishFractal() ) );
	m_time.start();
	m_computing_fractal->start();
	m_timer->start( 100 );
}

void MainWindow::stopFractal()
{
	Q_ASSERT( m_computing_fractal != NULL );
	m_computing_fractal->stop();
	m_computing_fractal->wait();
	disconnect( m_computing_fractal, SIGNAL( finished() ), this, SLOT( finishFractal() ) );
	delete m_computing_fractal;
	
	m_status_bar->showMessage( tr("Gnration arrte.") );
	m_timer->stop();
	m_compute_progressBar->setValue( 0 );
	m_compute_button->setEnabled( true );
	m_stop_button->setEnabled( false );
}

void MainWindow::finishFractal()
{
	if( ! m_computing_fractal->isUserStopped() )
	{
		m_status_bar->showMessage( tr("Temps coul : %1 s").arg( m_time.elapsed() / 1000.0 ) );
		disconnect( m_computing_fractal, SIGNAL( finished() ), this, SLOT( finishFractal() ) );
		m_fractal_type_label->setText( m_computing_fractal_type );
		m_parameters_widget->addZone( m_computing_fractal->fractalParameters().zone() );
		m_timer->stop();
		m_compute_progressBar->setValue( 0 );
		m_compute_button->setEnabled( true );
		m_stop_button->setEnabled( false );

		if( m_current_fractal != NULL )
			delete m_current_fractal;
		m_current_fractal_use_shaders = m_computing_fractal_use_shaders;
		m_current_fractal = m_computing_fractal;
		m_current_fractal_type = m_computing_fractal_type;
		m_computing_fractal = NULL;

		setFractalImage( m_palette_widget->palette() );
	}
}

void MainWindow::setFractalImage( const Palette& _palette )
{
	if( m_current_fractal != NULL )
	{
		m_pixmap_viewer->setPixmap( QPixmap::fromImage( m_current_fractal->toImage( _palette ) ) );
	}
}

void MainWindow::updateFractalImage( const Palette& _palette )
{
	if( m_current_fractal != NULL )
	{
		m_pixmap_viewer->updatePixmap( QPixmap::fromImage( m_current_fractal->toImage( _palette ) ) );
	}
}

void MainWindow::updateProgress()
{
	qreal progress = m_computing_fractal->progress();
	m_compute_progressBar->setValue( progress * 100 );

	if( progress > 0.0 )
	{
		int remaining = (1.0 - progress) * ( m_time.elapsed() / 1000.0 )  / progress;
		m_status_bar->showMessage( tr("Temps restant : %1 s").arg( remaining+1 ) );
	}
	else
	{
		m_status_bar->showMessage( tr("Gnration des threads...") );
	}
}

void MainWindow::imageChanged( const QPixmap& _pixmap )
{
	m_image_size_label->setText( tr( "%1x%2" ).arg( _pixmap.width() ).arg( _pixmap.height() ) );
}
void MainWindow::imageScaleChanged( qreal _scale_factor )
{
	m_image_scale_label->setText( tr( "%1%" ).arg( (unsigned int)( _scale_factor * 100 ) ) );
}
void MainWindow::imageSelected( QPointF _select_start, QPointF _select_end )
{
	if( m_current_fractal != NULL )
	{
		const Zone& zone = m_current_fractal->fractalParameters().zone();
		const QPixmap& pixmap = m_pixmap_viewer->pixmap();
		qreal zone_x_start	=  _select_start.x()	/ (qreal)pixmap.width()		* zone.width()	+ zone.x();
		qreal zone_y_start	=  _select_start.y()	/ (qreal)pixmap.height()	* zone.height()	+ zone.y();
		qreal zone_x_end	=  _select_end.x()		/ (qreal)pixmap.width()		* zone.width()	+ zone.x();
		qreal zone_y_end	=  _select_end.y()		/ (qreal)pixmap.height()	* zone.height()	+ zone.y();
		
		FractalParameters fractal_parameters( m_parameters_widget->toFractalParameters() );
		fractal_parameters.setZone( Zone( zone_x_start, zone_y_start, zone_x_end-zone_x_start, zone_y_end-zone_y_start ) );
		m_parameters_widget->setFractalParameters( fractal_parameters );
	}
}

void MainWindow::about()
{
	QMessageBox::about( this, tr( "A propos" ), tr(	"Fractal par Maxime Petitjean.<br>"
													"Logiciel dvelopp lors du <a href='http://qt.developpez.com/defis/01-buddhabrot/'>dfi Qt du forum Developpez.com</a><br>"
													"Version 1.0, 28/02/2010" ) );
}
void MainWindow::aboutQt()
{
	QMessageBox::aboutQt( this );
}

void MainWindow::setLanguageFr()
{
	if( QFile::exists( "translations/fractal_fr.qm") )
	{
		qApp->removeTranslator( m_translator );
		m_translator->load("translations/fractal_fr.qm");
		qApp->installTranslator( m_translator );
		qApp->removeTranslator( m_global_translator );
		m_global_translator->load("qt_fr", QLibraryInfo::location(QLibraryInfo::TranslationsPath));
		qApp->installTranslator(m_global_translator);
		ui.action_fr->setChecked( true );
		ui.action_en->setChecked( false );
	}
}
void MainWindow::setLanguageEn()
{
	if( QFile::exists( "translations/fractal_en.qm") )
	{
		qApp->removeTranslator( m_translator );
		m_translator->load("translations/fractal_en.qm");
		qApp->installTranslator( m_translator );
		qApp->removeTranslator( m_global_translator );
		m_global_translator->load("qt_en", QLibraryInfo::location(QLibraryInfo::TranslationsPath));
		qApp->installTranslator(m_global_translator);
		ui.action_fr->setChecked( false );
		ui.action_en->setChecked( true );
	}
}
