/*Copyright (c) 2010 Maxime Petitjean

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.*/

#include "mainwindow.h"

void MainWindow::showErrorFile( const QFile& _file )
{
	if( _file.openMode() == QFile::ReadOnly )
	{
		QMessageBox::warning( this, tr( "Erreur" ), tr( "Erreur de lecture : <br>%1" ).arg( _file.errorString() ) );
	}
	else
	{
		QMessageBox::warning( this, tr( "Erreur" ), tr( "Erreur d'criture : <br>%1" ).arg( _file.errorString() ) );
	}
}

void MainWindow::load()
{
	QString file_name = QFileDialog::getOpenFileName( this, tr( "Ouvrir une fractale" ), m_directory, tr( "Fractal (*.fractal)" ) );
	if ( file_name.isNull() ) return;
	
	QFile file( file_name );
	if( ! file.open( QFile::ReadOnly ) )
	{
		QMessageBox::warning(this, tr( "Erreur" ), tr( "Impossible de lire le fichier %1 :\n%2." ).arg( file_name ).arg( file.errorString() ) );
		return;
	}

	//Test de l'entte
	char header[8];
	if( file.read( (char*)&header, 8 ) != 8 ) { showErrorFile( file ); return; }
		
	if( strncmp( header, "FRACTAL", 8 )!=0 )
	{
		QMessageBox::warning( this, tr( "Erreur" ), tr( "Mauvaise entte." ) );
		return;
	}

	//Lecture du type de fractal
	unsigned char type_size;
	char buffer[256];
	if( file.read( (char*)&type_size, sizeof(unsigned char) ) != sizeof(unsigned char) || file.read( (char*)&buffer, type_size * sizeof(char) ) != type_size * sizeof(char) ) { showErrorFile( file ); return; }

	QString fractal_type( QByteArray( buffer, type_size ) );

	//Lecture des paramtres de la fractale
	FractalParameters fractal_parameters;
	if( file.read( (char*)&fractal_parameters, sizeof(FractalParameters) ) != sizeof(FractalParameters) ) { showErrorFile( file ); return; }
	
	//Lecture de la palette
	Palette palette;
	if( palette.load( file ) != 0 ) { showErrorFile( file ); return; }

	//Lecture des donnes de la fractale
	Fractal* fractal;
	if( ! allocFractal( fractal, fractal_parameters, fractal_type ) )
	{
		QMessageBox::warning( this, tr( "Erreur" ), tr( "Mauvais type de fractale." ) );
		return;
	}
	if( fractal->load( file ) != 0 ) { showErrorFile( file ); return; }
	
	if( m_current_fractal != NULL )
		delete m_current_fractal;
	m_current_fractal = fractal;

	m_parameters_widget->setFractalParameters( fractal_parameters );
	m_palette_widget->setPalette( palette );
	setFractalImage( palette );
}
void MainWindow::save()
{
	if( m_current_fractal == NULL )
	{
		QMessageBox::warning( this, tr( "Erreur" ), tr( "Aucune fractale gnre." ) );
		return;
	}

	if( m_current_fractal_use_shaders )
	{
		QMessageBox::warning( this, tr( "Erreur" ), tr( "Impossible de sauvegarder une fractale gnre par shaders." ) );
		return;
	}

	QString file_name = QFileDialog::getSaveFileName( this, tr( "Enregistrer une fractale" ), m_directory, tr( "Fractal (*.fractal)" ) );
	if ( file_name.isNull() ) return;

	QFile file( file_name );
	if( ! file.open( QFile::WriteOnly ) )
	{
		QMessageBox::warning(this, tr( "Erreur" ), tr( "Impossible d'crire le fichier %1 :\n%2." ).arg( file_name ).arg( file.errorString() ) );
		return;
	}


	//Ecriture de l'entte
	if( file.write( "FRACTAL", 8 ) != 8 ) { showErrorFile( file ); return; }

	//Ecriture du type de fractal
	unsigned char type_size = m_current_fractal_type.size();
	if( file.write( (const char*)&type_size, 1 ) != 1 || file.write( (const char*)m_current_fractal_type.toAscii().constData(), type_size ) != type_size ) { showErrorFile( file ); return; }

	//Eriture des paramtres de la fractale
	if( file.write( (const char*)&m_current_fractal->fractalParameters(), sizeof(FractalParameters) ) != sizeof(FractalParameters) ) { showErrorFile( file ); return; }

	//Ecriture de la palette
	if( m_palette_widget->palette().save( file ) != 0 ) { showErrorFile( file ); return; }

	//Ecriture des donnes de la fractale
	if( m_current_fractal->save( file ) != 0 ) { showErrorFile( file ); return; }
}
	
void MainWindow::saveToImage()
{
	if( m_pixmap_viewer->pixmap().isNull() )
	{
		QMessageBox::warning( this, tr( "Erreur" ), tr( "Aucune image gnre." ) );
		return;
	}

	QString file_name = QFileDialog::getSaveFileName( this, tr( "Enregistrer l'image" ), m_directory, tr(		"Portable Network Graphics (*.png)"
																												";;Joint Photographic Experts Group (*.jpg *.jpeg)"
																												";;Windows Bitmap (*.bmp)"
																												";;Portable Pixmap (*.pnm)") );
	if ( ! file_name.isNull() )
	{
		m_directory = file_name;
		
		if( ! m_pixmap_viewer->pixmap().save( file_name ) )
			QMessageBox::warning( this, tr( "Erreur" ), tr( "Erreur lors de la sauvegarde." ) );
	}
}
