/*
Copyright 2010 Pierre SCHWARTZ

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated 
documentation files (the "Software"), to deal in the Software without restriction, including without 
limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies 
of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following 
conditions: The above copyright notice and this permission notice shall be included in all copies or 
substantial portions of the Software. THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF 
OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
*/

/*!
 * \file app.cpp
 * \brief corps de la classe principale qui gre tout
 * \author Pierre Schwartz
 * \version 1.0
 */


#include <QtGui>
#include <QtGui/QPushButton>
#include <QtGui/QVBoxLayout>
#include <QtGui/QHBoxLayout>
#include <QtGui/QMainWindow>
#include <QtGui/QMenu>
#include <QtGui/QMessageBox>
#include <QtGui/QFileDialog>
#include <QtGui/QGridLayout>
#include <QtGui/QDockWidget>
#include <QtGui/QButtonGroup>
#include <QtGui/QRadioButton>
#include <QtGui/QScrollArea>

#include <complex>

#include "app.h"
#include "colormaker.h"
#include "imagecalculator.h"
#include "darkbody.h"
#include "grayscale.h"
#include "scriptedcolormaker.h"
#include "fulllinear.h"
#include "fractale.h"
#include "buddhabrot.h"
#include "mandelbrot.h"
#include "julia.h"
#include "newton.h"
#include "bateauenfeu.h"
#include "hexamandelbrot.h"
#include "tricorn.h"
#include "sharingan.h"
#include "networkinput.h"
#include "pluginfractale.h"
#include "configurationloader.h"
#include "mandelbrot3d.h"
#include "lightningmaker.h"

/*!
 * \brief Constructeur.
 * Classe principale de l'application. 
 * Elle gre tout l'affichage et fait le lien entre les fractales, les modes de coloration et les vnements utilisateur
 */
App::App(QWidget *parent): QMainWindow(parent){
	applyStyle();
	
	collectVectorialPalets();
	collectPluginFractales();

	qsrand(0);

	buildUI();	
	connectSignals();
	calculatedImage = coloredImage = NULL;	
	isNetworkUsed = false;

	setCurrentWindow(buddhabrot::getBestWindow());
}

/*!
 * \brief Constructeur.
 * vide la mmoire
 */
App::~App(){
	if (coloredImage)
		delete coloredImage;
	if (calculatedImage)
		delete calculatedImage;
	if (renderingImage)
		delete renderingImage;

	for (QMap<QString, fractale*>::iterator i = pluginFractales.begin(); i!=pluginFractales.end(); i++)
		delete i.value();
}

/*!
 * \brief Connexion de tous les vnements.
 */
void App::connectSignals(){
	// vnements double clic pour zoomer/dzoomer
	QObject::connect(renderingImage, SIGNAL(zoom(int, int)), this, SLOT(s_zoom(int, int)));
	QObject::connect(renderingImage, SIGNAL(unzoom(int, int)), this, SLOT(s_unzoom(int, int)));
	QObject::connect(renderingImage, SIGNAL(move(int, int)), this, SLOT(s_displayRealCoordinates(int, int)));

	// lancement global de gnration
	QObject::connect(calculateButton, SIGNAL(clicked()), this, SLOT(s_generate()));

	// mise  jour de la progressbar
	QObject::connect(this, SIGNAL(processFinishing(int)), bar, SLOT(setValue(int)));

	// boutons de coloration
	QObject::connect(darkBodyBtn,	SIGNAL(clicked()), this, SLOT(s_colorate_darkBody()));
	QObject::connect(grayBtn,		SIGNAL(clicked()), this, SLOT(s_colorate_gray()));
	QObject::connect(fullBtn,		SIGNAL(clicked()), this, SLOT(s_colorate_full()));

	// combobox
	QObject::connect(fractalType,		SIGNAL(currentIndexChanged  (int)), this, SLOT(s_changeFractale(int)));

	// QRadio
	QObject::connect(parPointsAleatoires,		SIGNAL(toggled (bool)), this, SLOT(s_displayPointsAleatoires(bool)));

	// network
	QObject::connect(useNetwork,	SIGNAL(clicked (bool)), this, SLOT(s_enableNetwork(bool)));
	QObject::connect(clearNetwork,	SIGNAL(clicked ()), this, SLOT(s_deleteLastNetwork()));
	QObject::connect(addNetwork,	SIGNAL(clicked()), this, SLOT(s_addNetwork()));
}

/*!
 * \brief construit toute l'interface utilisateur
 */
void App::buildUI(){
	setWindowTitle(tr("DVP Gnrateur de fractales"));
	setWindowIcon(QIcon("data/qt.ico"));	

	// crer la barre de menu
	createActions();
	createMenu();

	// barre d'tat
	statusBar = QMainWindow::statusBar();
	statusBarLabel = new QLabel();	
	statusBar->addPermanentWidget(statusBarLabel, 200);
	statusBar->show();

	// cration des widgets. une progressbar, une combo de choix de fractale, des boutons de coloration
	renderingImage = new ZoomableImage();
	fractalType = new QComboBox();
	fractalType->setToolTip(tr("Choix de la fractale  calculer"));
	fractalType->addItem("BuddhaBrot", 0);	
	fractalType->addItem("MultiBrot", 1);		
	fractalType->addItem("MandelBrot", 2);	
	fractalType->addItem("Julia (0.285)", 3);	
	fractalType->addItem("Julia (-0.038088 + 0.9754633i)", 4);
	fractalType->addItem("Newton", 5);			
	fractalType->addItem("Bateau en feu", 6);	
	fractalType->addItem("Tricorn", 7);		
	fractalType->addItem("Sharingan", 8);	
	fractalType->addItem("MandelBrot3D", 9);	

	// ajouter les fractales pluggues
	int lastIndex = 10;
	for (QMap<QString, fractale*>::iterator i = pluginFractales.begin(); i!=pluginFractales.end(); i++, lastIndex++){
		fractalType->addItem(i.key(), lastIndex);
	}

	// le bouton pour dmarrer la gnration de la fractale
	calculateButton = new QPushButton(tr("Calculer"));	
	calculateButton->setToolTip(tr("Lancer le calcul"));
	
	bar = new QProgressBar();
	bar->setVisible(false);		// la barre n'est visible que quand elle est utile	

	// cration des layouts pour positionner les widgets  l'cran, rien de bien compliqu
	QVBoxLayout *mainLayout = new QVBoxLayout();	
	QHBoxLayout *hlayout = new QHBoxLayout();

	// spinbox du choix du nombre d'itrations 
	iterations = new QSpinBox();	
	iterations->setMaximum(10000);
	iterations->setMinimum(10);
	iterations->setSingleStep(10);	
	iterations->setValue(70);
	iterations->setToolTip(tr("Taille maximale des sries fractales"));

	// choix de la rsolution de sortie
	sizes = new QComboBox();
	sizes->setToolTip(tr("Rsolution de l'image rsultat"));
	
	QWidget *hlayoutWidget = new QWidget();
	hlayoutWidget->setLayout(hlayout);
	hlayout->addWidget(new QLabel(tr("Itrations")));
	hlayout->addWidget(iterations);
	hlayout->addWidget(sizes);
		sizes->addItem("128x128", 128);
		sizes->addItem("256x256", 256);
		sizes->addItem("512x512", 512);
		sizes->addItem("1024x1024", 1024);
		sizes->addItem("2048x2048", 2048);
		hlayout->addWidget(fractalType);
		hlayout->addWidget(calculateButton);
		hlayout->setSizeConstraint(QLayout::SetFixedSize);
	mainLayout->addWidget(hlayoutWidget);
	
	// coche pour animer la gnration ou non	
	showAnimation = new QCheckBox(tr("Animer la gnration"));	
	showAnimation->setChecked(false);
	showAnimation->setVisible(false);

	mainLayout->addWidget(showAnimation);
	mainLayout->addWidget(bar);

	QScrollArea *area = new QScrollArea();
	area->setWidget(renderingImage);
	renderingImage->hide();
	mainLayout->addWidget(area);

	// crer la zone de coloration
	colorationWidget = new QWidget();
	QGridLayout *colorationLayout = new QGridLayout();	
		darkBodyBtn = new QPushButton(tr("DarkBody"));
		darkBodyBtn->setStyleSheet("QWidget{ background-image: url(data/images/darkBody.png); background-repeat:repeat-y;}");		
		
		grayBtn = new QPushButton(tr("Niveau de gris"));
		grayBtn->setStyleSheet("QWidget{ background-image: url(data/images/gray.png); background-repeat:repeat-y;}");		
		
		fullBtn = new QPushButton(tr("Full linear"));
		fullBtn->setStyleSheet("QWidget{ background-image: url(data/images/fullLinear.png); background-repeat:repeat-y;}");		
		
	colorationLayout->addWidget(darkBodyBtn);
	colorationLayout->addWidget(grayBtn);
	colorationLayout->addWidget(fullBtn);
	colorationButtons.push_back(darkBodyBtn);
	colorationButtons.push_back(grayBtn);
	colorationButtons.push_back(fullBtn);

	// ajouter toutes les palettes dynamiques
	for ( QStringList::Iterator it = palettesListe.begin(); it != palettesListe.end(); ++it ) {
		QPushButton *b = new QPushButton(*it);
		
		// si un png du mme nom existe, on le place en style
		QTextCodec::setCodecForCStrings(QTextCodec::codecForName("IBM 850"));
		QString filename = "data/palettes/%1.png";		
		filename = filename.arg(*it);
		if (QFile::exists(filename)){
			QString style = QString("QWidget{ background-image: url(%1); background-repeat:repeat-y;}").arg(filename);
			b->setStyleSheet(style);
		}
		
		colorationButtons.push_back(b);
		QObject::connect(b, SIGNAL(clicked()), this, SLOT(s_run_dynamic_coloration()));
		colorationLayout->addWidget(b);
	}
	// on disable tous les boutons de coloration prsents
	enableColorationButtons(false);

	colorationWidget->setLayout(colorationLayout);

	// dock de coloration
	dockColoration = new QDockWidget(tr("Coloration"));
	dockColoration->setFeatures(QDockWidget::DockWidgetFloatable | QDockWidget::DockWidgetMovable);
	addDockWidget(Qt::RightDockWidgetArea, dockColoration);
	dockColoration->setWidget(colorationWidget);

	// dock de fentrage
	windowDock = new QDockWidget(tr("Fentrage"));
	windowDock->setFeatures(QDockWidget::DockWidgetFloatable | QDockWidget::DockWidgetMovable);
	
	QGridLayout *g = new QGridLayout();
	windowX1 = new QLineEdit();	windowX1->setInputMask(QString("#00.000"));	windowX1->setMaxLength(8); windowX1->setAlignment(Qt::AlignCenter);
	windowY1 = new QLineEdit();	windowY1->setInputMask(QString("#00.000"));	windowY1->setMaxLength(8); windowY1->setAlignment(Qt::AlignCenter);
	windowX2 = new QLineEdit(); windowX2->setInputMask(QString("#00.000"));	windowX2->setMaxLength(8); windowX2->setAlignment(Qt::AlignCenter);
	windowY2 = new QLineEdit(); windowY2->setInputMask(QString("#00.000"));	windowX2->setMaxLength(8); windowY2->setAlignment(Qt::AlignCenter);
	g->addWidget(windowY1, 0, 1);
	g->addWidget(windowY2, 2, 1);
	g->addWidget(windowX1, 1, 0);
	g->addWidget(windowX2, 1, 2);
	windowY1->setToolTip(tr("Borne complexe infrieure"));
	windowY2->setToolTip(tr("Borne complexe suprieure"));
	windowX1->setToolTip(tr("Borne relle infrieure"));
	windowX2->setToolTip(tr("Borne relle suprieure"));
	QWidget *ww = new QWidget();
	ww->setLayout(g);
	windowDock->setWidget(ww);
	
	addDockWidget(Qt::RightDockWidgetArea, windowDock);

	// dock d'itrations
	iterationsDock = new QDockWidget(tr("Itrations"));
	iterationsDock->setFeatures(QDockWidget::DockWidgetFloatable | QDockWidget::DockWidgetMovable);
	addDockWidget(Qt::RightDockWidgetArea, iterationsDock);
	QWidget *www = new QWidget();
	QVBoxLayout *vb = new QVBoxLayout();
	
	QButtonGroup *group = new QButtonGroup();
	parPointsAleatoires = new QRadioButton(tr("Par points alatoires"));	
	parPixel = new QRadioButton(tr("Par points pixels"));
	parPixel->setChecked(true);

	group->addButton(parPointsAleatoires);
	group->addButton(parPixel);
	aleatoiresPoints = new QSpinBox();
	aleatoiresPoints->setMinimum(10000);
	aleatoiresPoints->setMaximum(1000000000);	
	aleatoiresPoints->setSingleStep(10000);
	aleatoiresPoints->hide();

	vb->setSizeConstraint(QLayout::SetMinimumSize);
	vb->addWidget(parPixel);
	vb->addWidget(parPointsAleatoires);
	vb->addWidget(aleatoiresPoints);

	www->setLayout(vb);
	iterationsDock->setWidget(www);

	// paramtrage du rendu rseau
	QDockWidget *networkDock = new QDockWidget(tr("Rseau"));
	networkDock->setFeatures(QDockWidget::DockWidgetFloatable | QDockWidget::DockWidgetMovable);
	addDockWidget(Qt::RightDockWidgetArea, networkDock);	
	QWidget *wwww = new QWidget();
	vbnetworkDock = new QVBoxLayout();

	useNetwork = new QCheckBox(QString(tr("Utiliser les agents rseau")));

	vbnetworkDock->addWidget(useNetwork);
	QHBoxLayout *networkHLayout = new QHBoxLayout();

	addNetwork = new QPushButton(QString(tr("Ajouter un agent rseau")));
	networkHLayout->addWidget(addNetwork);
	clearNetwork = new QPushButton(QString(tr("Supprimer")));
	networkHLayout->addWidget(clearNetwork);
	QWidget *networkHLayoutWidget = new QWidget();
	networkHLayoutWidget->setLayout(networkHLayout); 

	vbnetworkDock->addWidget(networkHLayoutWidget);
	NetworkInput *n = new NetworkInput();
	QObject::connect(n, SIGNAL(message(const QString&)), this, SLOT(s_receiveMessageFromNetwork(QString&)));
	networkAddresses.append(n);
	vbnetworkDock->addWidget(n);

	wwww->setLayout(vbnetworkDock);
	networkDock->setWidget(wwww);
	s_enableNetwork(false);


	// zone de rendu
	QWidget *renderingWidget = new QWidget();		
	renderingWidget->setLayout(mainLayout);	
	setCentralWidget(renderingWidget);	
}

/*!
 * \brief
 * Activer ou dsactiver tous les boutons de coloration
 */
void App::enableColorationButtons(bool b){
	QList<QPushButton*>::iterator i = colorationButtons.begin();
	while (i != colorationButtons.end()){
		(*i)->setDisabled(!b);
		i++;
	}
}

/*!
 * \brief
 * Affiche les coordonnes survoles dans la barre d'tat
 */
void App::s_displayRealCoordinates(int x, int y){
	float fx = currentWindow.left() + x* (currentWindow.width() / currentSize.width());
	float fy = currentWindow.top() + y* (currentWindow.height() / currentSize.height());

	statusBarLabel->setText(QString(tr("(%1,%2)")).arg(fx).arg(fy));
}

/*!
 * \brief
 * Charge et applique un style QSS
 */
void App::applyStyle(){
	QFile file("data/style.qss");
    file.open(QFile::ReadOnly);
    QString styleSheet = QLatin1String(file.readAll());
	setStyleSheet(styleSheet);
}
/*!
 * \brief
 * Trouver toutes les palettes de couleur dfinies
 */
void App::collectVectorialPalets(){	
	QDir dir("data/palettes");
	QStringList filters;
	filters << "*.js";
	dir.setNameFilters(filters);
	
	QFileInfoList list = dir.entryInfoList();
	for (int i = 0; i < list.size(); ++i) {		
		QFileInfo fileInfo = list.at(i);
		QString fname = fileInfo.fileName();

		// oublier l'extension .js
		fname.truncate(fname.size()-3);
		
		palettesListe.append(fname);		
	}
}

/*!
 * \brief
 * Trouver tous les plugins de fractales
 */
void App::collectPluginFractales(){
	QDir pluginsDir = QDir("data/fractales");
    foreach (QString fileName, pluginsDir.entryList(QDir::Files)) {
        QPluginLoader loader(pluginsDir.absoluteFilePath(fileName));
        QObject *plugin = loader.instance();
        if (plugin) {
            pluginfractale * op = qobject_cast<pluginfractale*>(plugin);
            if (op) {
				pluginFractales[op->getName()] =op;
            }
        }
    }
}


/*!
 * \brief cration des items de menu pour l'UI
 */
void App::createActions(){	
	saveAct = new QAction(tr("&Sauver..."), this);
	saveAct->setShortcut(tr("Ctrl+S"));
	saveAct->setIcon(QIcon::fromTheme(QString(""), QIcon("data/icons/document-save.png")));
	connect(saveAct, SIGNAL(triggered()), this, SLOT(save()));

	loadConfigAct = new QAction(tr("Charger un paramtrage..."), this);
	loadConfigAct->setIcon(QIcon::fromTheme(QString(""), QIcon("data/icons/param-load.png")));
	connect(loadConfigAct, SIGNAL(triggered()), this, SLOT(s_loadConfig()));

	saveConfigAct = new QAction(tr("Sauvegarder le paramtrage..."), this);
	saveConfigAct->setIcon(QIcon::fromTheme(QString(""), QIcon("data/icons/param-save.png")));
	connect(saveConfigAct, SIGNAL(triggered()), this, SLOT(s_saveConfig()));

	quitAct = new QAction(tr("&Quitter..."), this);
	quitAct->setIcon(QIcon::fromTheme(QString(""), QIcon("data/icons/quit.png")));
	connect(quitAct, SIGNAL(triggered()), this, SLOT(close()));

	aboutAct = new QAction(tr("&A propos..."), this);
	aboutAct->setIcon(QIcon::fromTheme(QString(""), QIcon("data/icons/about.png")));
	connect(aboutAct, SIGNAL(triggered()), this, SLOT(about()));
}
/*! 
 * \brief  cration des menus
 */
void App::createMenu(){	
	QMenu *fileMenu = new QMenu(tr("&Fichier"), this);
	fileMenu->addAction(saveAct);
	fileMenu->addAction(loadConfigAct);
	fileMenu->addAction(saveConfigAct);
	fileMenu->addAction(quitAct);

	QMenu *aboutMenu = new QMenu("?", this);
	aboutMenu->addAction(aboutAct);

	menuBar()->addMenu(fileMenu);
	menuBar()->addMenu(aboutMenu);
}

/*! 
 * \brief pouvoir sauvegarder le calcul super long d'un Budhabrot (bien qu'en fait la sauvegarde est automatique dans result.bmp)
 */
void App::save(){
	QString fileName = QFileDialog::getSaveFileName(this,
                                     tr("Enregistrer le rendu"), QDir::currentPath(), QString(tr("Images PNG (*.png)")));

	if (!fileName.isEmpty())	
		renderingImage->pixmap()->save(fileName);
}

/*!
 * \brief modeste narcissisme
 */
void App::about(){
	QMessageBox::about(this, tr("A propos"), tr("Fruit de plusieurs semaines de travail, ce programme est ma modeste participation au Dfi Qt de Developpez.com"));
}



/*!
 * \brief Dtermine la fractale  dessiner
 */
void App::generateStartFractaleWindows(){
	int fractaleType = fractalType->itemData(fractalType->currentIndex()).value<int>();

	switch(fractaleType){
		case 0 : 
			generateBuddhaBrot(); 
			break;
		case 2 : 
			generateMandelBrot();
			break;
		case 3 : 
			generateJulia(std::complex<float>(.285f,0));
			break;
		case 4 : 
			generateJulia(std::complex<float>(-0.038088f, 0.9754633f));
			break;
		case 5 : 
			generateNewton();
			break;
		case 6 : 
			generateBateauEnFeu();
			break;
		case 1 : 
			generateHexaMandelBrot();
			break;
		case 7 : 
			generateTricorn();
			break;
		case 8 : 
			generateSharingan();
			break;
		case 9:
			generateMandel3D();
			break;
		default:
			// plugged fractale
			QString name = fractalType->itemText(fractalType->currentIndex());
			generatePlugged(name);
			break;
	}
}

/*!
 * \brief  gnrer une fractale, mais d'abord voir de laquelle il s'agit
 */
void App::s_generate(){
	// collecter les donnes de fentrage
	float x1 = windowX1->text().toFloat();
	float x2 = windowX2->text().toFloat();
	float y1 = windowY1->text().toFloat();
	float y2 = windowY2->text().toFloat();

	currentWindow = QRectF(x1,y1,x2-x1,y2-y1); 

	generateStartFractaleWindows();
}

/*!
 * \brief Charge un paramtrage depuis un fichier .dvp 
 */
void App::s_loadConfig(){
	QString fileName = QFileDialog::getOpenFileName(this, tr("Ouvrir un paramtrage"), QDir::currentPath(), QString(tr("Paramtrage (*.dvp)")));
	if (!fileName.isEmpty()){
		ConfigurationLoader config(fileName);
		
		iterations->setValue(config.getIterations());
		
		useNetwork->setCheckState(config.getUseNetwork()?Qt::Checked:Qt::Unchecked);
		parPixel->setChecked(!config.getUseRandom());
		parPointsAleatoires->setChecked(config.getUseRandom());
		aleatoiresPoints->setValue(config.getRandomCount());

		windowX1->setText(QString("%1").arg(config.getWindow().left()));
		windowX2->setText(QString("%1").arg(config.getWindow().right()));
		windowY1->setText(QString("%1").arg(config.getWindow().top()));
		windowY2->setText(QString("%1").arg(config.getWindow().bottom()));

		setNetworkList(config.getNetworkAgents());
	}
}

/*!
 * \brief Enregistre un paramtrage dans un fichier .dvp 
 */
void App::s_saveConfig(){
	QString filename = QFileDialog::getSaveFileName(this, tr("Sauvegarder un paramtrage"), QDir::currentPath(), QString(tr("Paramtrage (*.dvp)")));
	
	ConfigurationLoader config;	
			
	config.setIterations(iterations->text().toLong());
	config.setRandomCount(aleatoiresPoints->text().toLong());
	config.setUseNetwork(useNetwork->checkState() == Qt::Checked);
	config.setUseRandom(parPointsAleatoires->isChecked());
	
	QRectF window(windowX1->text().toFloat(), windowY1->text().toFloat(), 
		windowX2->text().toFloat() - windowX1->text().toFloat(),
		windowY2->text().toFloat() - windowY1->text().toFloat());
	config.setWindow(window);
	config.setNetworkAgents(collectNetworkAgents());

	config.save(filename);	
	
}

/*!
 * \brief positionner les bons champs de saisie de fentrage
 */
void App::setCurrentWindow(QRectF& r){
	windowX1->setText(QString("%1" ).arg(r.left()));
	windowX2->setText(QString("%1" ).arg(r.right()));
	windowY1->setText(QString("%1" ).arg(r.top()));
	windowY2->setText(QString("%1" ).arg(r.bottom()));
}

/*!
 * \brief gnrer le Buddhabrot
 */
void App::s_changeFractale(int type){
	switch(type){
		case 0:
			setCurrentWindow(buddhabrot::getBestWindow());
			break;
		case 1:
			setCurrentWindow(hexamandelbrot::getBestWindow());
			break;
		case 2:
			setCurrentWindow(mandelbrot::getBestWindow());
			break;
		case 3:
		case 4:
			setCurrentWindow(julia::getBestWindow());
			break;
		case 5:
			setCurrentWindow(newton::getBestWindow());
			break;
		case 6:
			setCurrentWindow(bateauEnFeu::getBestWindow());
			break;
		case 7:
			setCurrentWindow(tricorn::getBestWindow());
			break;
		case 8:
			setCurrentWindow(sharingan::getBestWindow());
			break;
		case 9:
			setCurrentWindow(mandelbrot::getBestWindow());
			break;
	}
}

/*!
 * \brief Lance la gnration du Buddhabrot
 */
void App::generateBuddhaBrot(){	
	buddhabrot b(std::complex<float>(), iterations->value());
	generateFractale(b);
}

/*!
 * \brief Lance la gnration du MandelBulb
 */
void App::generateMandel3D(){
	if (calculatedImage != NULL)
		delete calculatedImage;

	threeDrendering.clear();

	int resolution = sizes->itemData(sizes->currentIndex()).value<int>();
	currentSize = QSize(resolution,resolution);

	renderingImage->setMinimumHeight(resolution);
	renderingImage->setMaximumHeight(resolution);
	renderingImage->setMinimumWidth(resolution);
	renderingImage->setMaximumWidth(resolution);

	bar->setVisible(true);
	calculatedImage = new QImage(currentSize, QImage::Format_RGB888);

	// mettre l'image en noir
	calculatedImage->fill(QColor(0,0,0).rgb());

	// collecter les donnes de fentrage
	float x1 = windowX1->text().toFloat();
	float x2 = windowX2->text().toFloat();
	float y1 = windowY1->text().toFloat();
	float y2 = windowY2->text().toFloat();

	currentWindow = QRectF(x1,y1,x2-x1,y2-y1); 

	MandelBrot3d mandel(threeDrendering, normalMap, calculatedImage, currentWindow, resolution, iterations->value());
	QObject::connect(&mandel, SIGNAL(progress(int)), bar, SLOT(setValue(int)));

	maxValue = mandel.process();

	// appeler l'clairage par dfaut
	if (coloredImage != NULL)
		delete coloredImage;
	coloredImage = new QImage(currentSize, QImage::Format_RGB888);
	
	runLightning();

	renderingImage->setPixmap(QPixmap::fromImage(*coloredImage));
	coloredImage->save("colored.bmp");

	s_colorate_gray();

	renderingImage->show();
	calculatedImage->save("./result.bmp");

	bar->setVisible(false);

	enableColorationButtons(true);
	statusBarLabel->setText("");
}

/*!
 * \brief gnrer la fractale plugge
 */
void App::generatePlugged(QString& name){
	fractale *f = pluginFractales[name];
	f->setMaxIterations(iterations->value());
	generateFractale(*f);
}

/*!
 * \brief gnrer l' hexaBuddhabrot
 */
void App::generateHexaMandelBrot(){	
	hexamandelbrot b(std::complex<float>(), iterations->value());
	generateFractale(b);
}

/*!
 * \brief gnrer le Mandelbrot
 */
void App::generateMandelBrot(){	
	mandelbrot b(std::complex<float>(), iterations->value());
	generateFractale(b);
}
/*!
 * \brief Appel au clic sur une coloration dynamique
 */
void App::s_run_dynamic_coloration(){
	QObject *from = sender();
	QPushButton * button = qobject_cast<QPushButton*>(from);

	QString paletteName = button->text();

	scriptedColorMaker d(1000, paletteName);
	colorate(d);
}

/*!
 * \brief gnrer le Newton
 */
void App::generateNewton(){	
	newton b(std::complex<float>(), iterations->value());
	generateFractale(b);
}

/*!
 * \brief gnrer le Julia
 */
void App::generateJulia(std::complex<float> c){
	julia b(c, iterations->value());
	generateFractale(b);
}

/*!
 * \brief gnrer le Bateau en feu
 */
void App::generateBateauEnFeu(){
	bateauEnFeu b(std::complex<float>(), iterations->value());
	generateFractale(b);
}
/*!
 * \brief gnrer le tricorn
 */
void App::generateTricorn(){
	tricorn b(std::complex<float>(), iterations->value());
	generateFractale(b);
}

/*!
 * \brief gnrer le sharingan
 */
void App::generateSharingan(){
	sharingan b(std::complex<float>(), iterations->value());
	generateFractale(b);
}


/*!
 * \brief
 * Calculer les donnes brutes (2D) de la fractale, typiquement en niveau de gris.
 * Logiquement, cette fonction remplit calculatedImage, qui servira de base ensuite pour la coloration
 *  BuddhaBrot : [0->resolution] |-> [0->resolution].[0-2^32]    (stock en [0->resolution].[0->2^8]^4
 */
void App::generateFractale(fractale &f){
	// collecter les infos de config
	QRectF rect = currentWindow;
	int resolution = sizes->itemData(sizes->currentIndex()).value<int>();
	currentSize = QSize(resolution,resolution);

	renderingImage->setMinimumHeight(currentSize.height());
	renderingImage->setMaximumHeight(currentSize.height());
	renderingImage->setMinimumWidth(currentSize.width());
	renderingImage->setMaximumWidth(currentSize.width());

	bar->setVisible(true);
	if (calculatedImage != NULL)
		delete calculatedImage;
	calculatedImage = new QImage(currentSize, QImage::Format_RGB888);

	// mettre l'image en noir
	calculatedImage->fill(QColor(0,0,0).rgb());	

	// trouver combien de threads il faut lancer
	int nbThreads = QThread::idealThreadCount();
	if (nbThreads <= 0)
		nbThreads = 1;

	// agents rseau ?
	QStringList networkAgentsList = collectNetworkAgents();

	if (networkAgentsList.count() == 0)
		statusBarLabel->setText(QString(tr("%1 thread(s) en cours ...")).arg(nbThreads));
	else
	if (networkAgentsList.count() == 1)
		statusBarLabel->setText(QString(tr("%1 thread(s) en cours + 1 agent rseau...")).arg(nbThreads));
	else
		statusBarLabel->setText(QString(tr("%1 thread(s) en cours + %2 agents rseau...")).arg(nbThreads).arg(networkAgentsList.count()));
		
	enableColorationButtons(false);

	bool type = parPointsAleatoires->isChecked();
	long nbRandom = aleatoiresPoints->value() / nbThreads;

	ImageCalculator ic(nbThreads, calculatedImage, rect, currentSize, f, type, nbRandom, showAnimation->isChecked(), networkAgentsList);
	QObject::connect(&ic, SIGNAL(progress(int)), bar, SLOT(setValue(int)));
	QObject::connect(&ic, SIGNAL(render()), this, SLOT(s_render()));
	ic.run();
	ic.wait();
	maxValue = ic.maxValue;

	// rendu + sauvegarde auto
	renderingImage->setPixmap(QPixmap::fromImage(*calculatedImage).scaled(renderingImage->size(),Qt::IgnoreAspectRatio, Qt::FastTransformation));
	calculatedImage->save("./result.bmp");

	bar->setVisible(false);
	showAnimation->setVisible(true);

	/**
	 * Trs important : calculatedImage n'est pas colore, les informations RGB ne sont pas reprsentatives du rendu final color.
	 * Il s'agit uniquement d'un moyen de mmoriser la valeur [0->2^32-1] sur [0->2^8]^4
	 * on affiche ici btement calculatedImage histoire d'avoir quelque chose  regarder, 
	 *   mme si thoriquement a ne reprsente pas le BuddhaBrot : difficile d'afficher une image 32 bits par pixel 
	 *   sur un seul canal ;)
	 */	
	s_colorate_gray();
	colorationWidget->setVisible(true);
	renderingImage->show();
	enableColorationButtons(true);
	statusBarLabel->setText("");
}

/*! 
 * \brief Restreindre la fentre de rendu et relancer
 */
void App::s_zoom(int x, int y){
	// obtenir les coordonnes relles cliques
	QPointF center ( currentWindow.left() + float(x) * currentWindow.width() / currentSize.width(), 
					currentWindow.top() + float(y) * currentWindow.height() / currentSize.height()
					);
	
	currentWindow = QRectF(center.x() - currentWindow.width()/4, 
							center.y() - currentWindow.height()/4,
							currentWindow.width()/2,
							currentWindow.height()/2
							);
	setCurrentWindow(currentWindow);
	
	s_generate();
	//generateStartFractaleWindows();
}

/*! 
 * \brief Agrandir la fentre de rendu et relancer
 */
void App::s_unzoom(int x, int y){
	// obtenir les coordonnes relles cliques
	QPointF center ( currentWindow.left() + float(x) * currentWindow.width() / currentSize.width(), 
					currentWindow.top() + float(y) * currentWindow.height() / currentSize.height()
					);
	
	currentWindow = QRectF(center.x() - currentWindow.width(), 
							center.y() - currentWindow.height(),
							currentWindow.width()*2,
							currentWindow.height()*2
							);
	setCurrentWindow(currentWindow);
	
	s_generate();
	//generateStartFractaleWindows();
}

/*!
 * \brief Signal de raffichage de l'image
 */
void App::s_render(){
	renderingImage->setPixmap(QPixmap::fromImage(*calculatedImage).scaled(renderingImage->size(),Qt::IgnoreAspectRatio, Qt::FastTransformation));
	
	if (showAnimation->checkState() == Qt::Checked)
		s_colorate_gray();
}

/*! 
 * \brief Rendu en Orange/Noir
 */
void App::s_colorate_darkBody(){
	darkBody d(1000);
	colorate(d);
}

/*! 
 * \brief Rendu en Noir/Blanc
 */
void App::s_colorate_gray(){
	grayScale d(1000);
	colorate(d);
}

/*! 
 * \brief Rendu en FuLL Linear
 */
void App::s_colorate_full(){
	fullLinear d(100);
	colorate(d);
}

/*!
 * \brief Slot pour afficher la zone de slection du nombre de points alaoires. Cette zone n'est visible qu'en mode alatoire
 */
void App::s_displayPointsAleatoires(bool b){
	if (b)
		aleatoiresPoints->show();
	else
		aleatoiresPoints->hide();
}

/*!
 * \brief Slot de slection du mode de rendu rseau
 */
void App::s_enableNetwork(bool b){
	addNetwork->setDisabled(!b);
	clearNetwork->setDisabled(!b);
	isNetworkUsed = b;
}


/*! 
 * \brief supprime le dernier agent rseau
 */
void App::s_deleteLastNetwork(){
	if (networkAddresses.length() > 0){
		NetworkInput*e = networkAddresses.last();
		networkAddresses.pop_back();
		delete e;
	}
}

/*!
 * \brief Slot d'affichage de message d'erreur rseau
 */
void App::s_receiveMessageFromNetwork(QString& s){
	statusBar->showMessage(s);
}

/*! 
 * \brief ajoute un nouvel agent rseau
 */
void App::s_addNetwork(){
	NetworkInput *n = new NetworkInput();
	networkAddresses.append(n);
	vbnetworkDock->addWidget(n);
	QObject::connect(n, SIGNAL(message(QString&)), this, SLOT(s_receiveMessageFromNetwork(QString&)));
}

/*! 
 * \brief on colore calculatedImage pour la mettre dans coloredImage
 */
void App::colorate(colorMaker &color){
	if (coloredImage != NULL)
		delete coloredImage;
	coloredImage = new QImage(currentSize, QImage::Format_RGB888);
	
	if (maxValue == 0 || maxValue > 10000000)
		maxValue = 256;

	if (showAnimation->checkState() != Qt::Checked)
		bar->show();
	
	for (int i=0; i<currentSize.width(); i++){
		for (int j=0; j<currentSize.height(); j++){
			QRgb r = calculatedImage->pixel(i,j);
			float value = ImageCalculator::RGBAToInt(r) * color.getNbLevels() / maxValue;

			QColor newColor = color.get(value);
			coloredImage->setPixel(i,j, newColor.rgb());
		}

		if (showAnimation->checkState() != Qt::Checked)
			bar->setValue(100 * i / currentSize.width());
	}

	renderingImage->setPixmap(QPixmap::fromImage(*coloredImage));
	if (showAnimation->checkState() != Qt::Checked)
		bar->hide();
	coloredImage->save("colored.bmp");
}

/*! 
 * \brief rcuprer tous les agents rseau positionns
 */
QStringList App::collectNetworkAgents(){
	QStringList networkAgentsList;
	for (QList<NetworkInput*>::iterator i = networkAddresses.begin(); i != networkAddresses.end(); i++)
		if ((*i)->getUrl() != QString(""))
			networkAgentsList << (*i)->getUrl();		
	
	return networkAgentsList;
}

/*! 
 * \brief positionner une liste d'agents rseau
 */
void App::setNetworkList(QList<QString>& l){	
	// vider le layout	
	for (QList<NetworkInput*>::const_iterator i=networkAddresses.begin(); i!=networkAddresses.end(); i++){
		NetworkInput *n = *i;
		delete n;
	}	
	networkAddresses.clear();
	
	// recharger la liste
	for (QList<QString>::const_iterator i = l.begin(); i!=l.end(); i++){
		NetworkInput *n = new NetworkInput();
		n->setUrl(*i);
		QObject::connect(n, SIGNAL(message(const QString&)), this, SLOT(s_receiveMessageFromNetwork(QString&)));
		networkAddresses.append(n);
		vbnetworkDock->addWidget(n);
	}
	vbnetworkDock->update();
}

/*!
 * \brief Lance le calcul de l'clairage 3D
 */
void App::runLightning(){
	// le point de vue est spcifi en dur : 0,0,3   la source de lumire est aussi spcifie en dur -2,-2,-3
	lightningMaker m(threeDrendering, normalMap, currentWindow, currentSize, coloredImage, 
		triplex(0,0,-3), triplex(-2,-2,-3));
	m.run();
}