/*
Copyright 2010 Pierre SCHWARTZ

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated 
documentation files (the "Software"), to deal in the Software without restriction, including without 
limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies 
of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following 
conditions: The above copyright notice and this permission notice shall be included in all copies or 
substantial portions of the Software. THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF 
OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
*/

/*!
 * \file imagecalculator.cpp
 * \brief corps du rendu
 * \author Pierre Schwartz
 * \version 1.0
 */

#include "imagecalculator.h"
#include "imagetiler.h"
#include "networkimagetiler.h"
#include "fractale.h"

ImageCalculator::ImageCalculator(int nb, QImage *d, QRectF w, QSize r, fractale &f, bool type, long nbr, bool a, QStringList& networkList):nbThreads(nb), destination(d), window(w), size(r), frac(f), nbRandom(nbr), useRandomCalculation(type), animate(a), networkAdresses(networkList){	
	for (int j=0; j<r.width(); j++){
		QVector<long> v;
		v.fill(0,r.height());			
		result.push_back(v);
	}		
	
	nbProcessedRows = 0;
	lastProgression = 0;
}

ImageCalculator::~ImageCalculator(){
}

/*! 
* \brief calcul de la fractale, on y instancie tous les threads, tous les agents rseau et on calcule tout a
*/
void ImageCalculator::run(){
	int nbAgents = nbThreads + networkAdresses.length();

	for(int i=0; i<nbAgents; i++){
		ImageTiler *it = new ImageTiler(result, mutex, window, size, &frac, useRandomCalculation, nbRandom, i, nbAgents, i);
		
		// lier les slots
		QObject::connect(it, SIGNAL(processed()), this, SLOT(s_processing()));
		
		it->run();
		tilers.push_back(it);
	}

	for (int i=0; i<networkAdresses.length(); i++){
		NetworkImageTiler *it = new NetworkImageTiler(networkAdresses.at(i), result, mutex, window, size, &frac, useRandomCalculation, nbRandom, nbThreads+i, nbAgents);
		it->run();
		networkTilers.push_back(it);
	}

	for (int i=0; i<nbThreads; i++)
		tilers[i]->wait();
	for (int i=0; i<networkAdresses.length(); i++)
		networkTilers[i]->wait();

	sumImagesIntoDestination();
}

/*! 
* \brief passer du buffer priv  la notion d'image Qt
*/
void ImageCalculator::sumImagesIntoDestination(){
	int v;
	maxValue =  0;
	for (int x = 0; x< size.width(); x++)
		for (int y=0; y<size.height(); y++){
			v = result[x][y];
			destination->setPixel(x,y,ImageCalculator::intToRGBA( v ));
			if (v > maxValue)
				maxValue = v;
		}

	for (int i=0; i<tilers.count(); i++)		
		delete tilers[i];
	for (int i=0; i<networkTilers.count(); i++)
		delete networkTilers[i];

	emit render();
}

/*! 
* \brief signifier de l'avancement du calcul
*/
void ImageCalculator::s_processing(){
	nbProcessedRows ++;

	if (animate){
		if (useRandomCalculation){
			// rendu alatoire
			if (int(100 * nbProcessedRows / nbRandom) > lastProgression){
				sumImagesIntoDestination();
				emit render();
			}
		}else{
			sumImagesIntoDestination();
			emit render();
		}
	}

	if (useRandomCalculation){
		// rendu alatoire
		lastProgression = 100 * nbProcessedRows / nbRandom;
		emit progress(lastProgression);
	}else{
		// rendu par point
		emit progress(100 * nbProcessedRows / size.width());
	}
}


 