/*
Copyright 2010 Pierre SCHWARTZ

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated 
documentation files (the "Software"), to deal in the Software without restriction, including without 
limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies 
of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following 
conditions: The above copyright notice and this permission notice shall be included in all copies or 
substantial portions of the Software. THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF 
OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
*/

/*!
 * \file imagetiler.cpp
 * \brief corps du rendu thread
 * \author Pierre Schwartz
 * \version 1.0
 */

#include "imagetiler.h"
#include "imagecalculator.h"

ImageTiler::ImageTiler(QVector<QVector<long> >& result, QMutex& mutex, QRectF w, QSize r, fractale *f, bool type, long nbRandom, int start, int step, int id):destinationArray(result), mutex(mutex), window(w), size(r), frac(f), displayRandomPoints(type), nbRandom(nbRandom), start(start), step(step), id(id){
	maxValue = 0;
}

ImageTiler::~ImageTiler(){
}	
	
void ImageTiler::run(){
	fractale *f = frac->clone();
	maxValue = 0;

	// quel est le pas continu de nos itrations ?
	double pasX = window.width() / size.width();
	double pasY = window.height() / size.height();
	
	if (displayRandomPoints){
		for (long i=0; i<nbRandom; i++){
			float tmpX = window.left() + window.width() * qrand() / RAND_MAX;	
			float tmpY = window.top() + window.height() * qrand() / RAND_MAX;	
			std::complex<float> start(tmpX, tmpY);
			processSinglePoint(start, f, pasX, pasY);
			emit processed();
		}
	}else{

		// dessine-t-on une partie suprieure ou infrieure ? 
		bool superior =  (window.top() <= -window.bottom());

		for (int x = start;  x < size.width(); x+=step){	
			for (int y = 0; y < size.height(); y++){					
				// trouver les coordonnes complexes du point (x,y)			
				float tmpY = window.top() + y*pasY;						
				// pas la peine de tout calculer en double si on est sr d'avoir la symtrie verticale ou centrale	
				if (superior && (f->isVerticalSymetric() || f->isImpair()) && tmpY >= pasY)
					continue;
				else
				if (!superior && (f->isVerticalSymetric() || f->isImpair()) && tmpY <= 0)
					continue;

				float tmpX = window.left() + x*pasX;	

				// droulage de la suite fractale
				std::complex<float> start(tmpX, tmpY);

				processSinglePoint(start, f, pasX, pasY);
			}
			emit processed();
		}
	}	

	delete f;	
}


/*! 
* \brief droule toute une liste fractale
*/
void ImageTiler::processSinglePoint(std::complex<float>& start, fractale *f, float pasX, float pasY){
	std::complex<float> tmp = start;
	f->setStart(start);
	QVector<std::complex<float> > serie;

	serie.push_back(start);

	for (int i=0; i<f->getMaxIterations(); i++){
		tmp = f->next(tmp);	

		if (!f->stop())
			serie.push_back(tmp);
		else{	
			// si la condition de fin de la suite fractale est atteinte, on analyse le rsultat
													
			// maitenant on peut vider la serie fractale
			QVector<std::complex<float> >::iterator ii = serie.begin();
			while (ii != serie.end()){
				std::complex<float> destinationPoint = f->renderingDestination(*ii);

				// retrouver les coordonnes pixels
				int xx =  (destinationPoint.real() - window.left()) / pasX;
				int yy =  (destinationPoint.imag() - window.top()) / pasY;
				int antiXX = -1;
				int antiYY = -1;
				long value;

				if (xx >=0 && xx < size.width()){

					// incrmenter l'image au point destination 
					if (yy >= 0  && yy < size.height()){
						// on rcupre la valeur (x,y) sur l'image, et on incrmente											
						mutex.lock();
						value = destinationArray[xx][yy] + 1;
						destinationArray[xx][yy] = value;
						mutex.unlock();
					}

					// traiter l'anti point						
					if (f->isVerticalSymetric()){
						// on vite de doubler l'axe des x
						if ( fabs(destinationPoint.imag()) > 0){
							antiYY = (-destinationPoint.imag() - window.top()) / pasY;

							if (antiYY >= 0 && antiYY < size.height()){
								mutex.lock();
								value = destinationArray[xx][antiYY] + 1;
								destinationArray[xx][antiYY] = value;
								mutex.unlock();
							}
						}
					}else
					if (f->isImpair()){
						antiYY =  (-destinationPoint.imag() - window.top()) / pasY;
						if ( fabs(destinationPoint.imag()) > 0){
							if (antiYY >= 0 && antiYY < size.height()){
								antiXX = (-destinationPoint.real() - window.left()) / pasX;
								if (antiXX >= 0 && antiXX < size.width()){
									mutex.lock();
									value = destinationArray[antiXX][antiYY] + 1;
									destinationArray[antiXX][antiYY] = value;
									mutex.unlock();
								}
							}
						}
					}

					if (value > maxValue)
						maxValue = value;	
				}

				ii++;
			}					

			serie.clear();
			break;
		}
	}
			
	serie.clear();
}