/*
Copyright 2010 Pierre SCHWARTZ

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated 
documentation files (the "Software"), to deal in the Software without restriction, including without 
limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies 
of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following 
conditions: The above copyright notice and this permission notice shall be included in all copies or 
substantial portions of the Software. THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF 
OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
*/

/*!
 * \file mandelbrot3d.h
 * \brief corps de la classe de calcul du MandelBulb
 * \author Pierre Schwartz
 * \version 1.0
 */

#include "mandelbrot3d.h"
#include "imagecalculator.h"
#include <cmath>

/*!
 * \brief Constructeur, rcupre toutes les donnes de travail
 */
MandelBrot3d::MandelBrot3d(QVector<QVector<int> >&vector, QVector<QVector<triplex> >&normalMap, QImage *d, QRectF window, int reso, int iterations):reso(reso), iterations(iterations), destination(d), projectedImage(vector), normalMap(normalMap), window(window){
	step.a = window.width()/ reso;
	step.b = window.height()/ reso;
	step.c = step.a;
	
	projectedImage.clear();
	normalMap.clear();

	for (int i=0; i<reso; i++){
		QVector<int> v;
		v.fill(-1, reso);
		projectedImage.push_back(v);

		QVector<triplex> v2;
		v2.fill(triplex(), reso);	
		normalMap.push_back(v2);
	}		
}

/*!
 * \brief Dtermine si un point est ou non dans l'ensemble de MandelBrot
 */
bool MandelBrot3d::processPoint(triplex& t){
	triplex start = t;
	triplex tmp = start;

	for (int i=0; i<iterations; i++){
		if (tmp.norm2() > 4)
			return false;	// le point n'est pas dans l'ensemble de Mandelbrot
		
		tmp = tmp.square() + start;		// z^2+c	
	}

	return true;
}

/*!
 * \brief Calcul global.
 * Le principe est de calculer la projection orthographique le long de l'axe -z de l'ensemble de MandelBrot.
 * Pour cela, l'application effectue un lancer de rayon orthographique jusqu' rencontrer un point de l'ensemble.
 * L'objet projectedImage stockera la distance de projection entre l'cran et l'ensemble de MandelBrot.
 */
int MandelBrot3d::process(){
	int max = 0;
	for (int i=0; i<reso; i++){
		float x = window.left() + i*step.a;
		for (int j=0; j<reso; j++){
			float y = window.top() + j * step.b;
			for (int k=0; k<reso; k++){
				float z = window.left() + k * step.c;
				if (processPoint(triplex(x,y,z))){
					// on a trouv un point de l'ensemble de Mandelbrot
					projectedImage[i][j] = k;
					
					if (k > max)
						max = k;
					
					break;
				}
			}
		}
	
		emit progress(100*i/reso * 4/5);
	}

	//  ce stade on a une heightmap vue depuis la camra projete le long de l'axe z vers les ngatifs

	// on rebalance tout dans l'image
	for (int i=0; i<reso; i++){
		for (int j=0; j<reso; j++){
			int value = projectedImage[i][j];
			if (value == -1)
				value = 0;
			else
				value = max-value;

			destination->setPixel(i,j, ImageCalculator::intToRGBA(value));

			// calcul de la normale au point considr
			float beforex = (i==0)?0: (projectedImage[i-1][j] - value);
			float afterx  = (i==reso-1)?0: (value - projectedImage[i+1][j]);
			float beforey = (j==0)?0: (projectedImage[i][j-1] - value);
			float aftery  = (j==reso-1)?0 : (value - projectedImage[i][j+1]);

			QPointF p(afterx - beforex, aftery - beforey);
			
			triplex n(p.x(), p.y(),10);
			
			normalMap[i][j] = n.normalize();
		}

		emit progress(80 +100*i/reso * 1/5);
	}
	return max;
}