/*
Copyright 2010 Pierre SCHWARTZ

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated 
documentation files (the "Software"), to deal in the Software without restriction, including without 
limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies 
of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following 
conditions: The above copyright notice and this permission notice shall be included in all copies or 
substantial portions of the Software. THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF 
OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
*/

#include "networkagent.h"
#include "fractale.h"
#include "buddhabrot.h"
#include "julia.h"
#include "mandelbrot.h"
#include "tricorn.h"
#include "sharingan.h"
#include "newton.h"
#include "hexamandelbrot.h"
#include "bateauenfeu.h"

#include "imagetiler.h"
#include "imagecalculator.h"
#include <iostream>

NetworkAgent::NetworkAgent(QVector<QVector<long> >& result, QSize resolution, QRectF window, bool isRandom, QString fractaleName, int offset, int iterations, int nbRandom, QString output, int step):result(result){
	QMutex mutex;
	
	fractale *f = NULL;
	if (fractaleName == QString("buddha"))
		f = new buddhabrot(std::complex<float>(), iterations);
	else
	if (fractaleName == QString("mandel"))
		f = new mandelbrot(std::complex<float>(), iterations);
	else
	if (fractaleName == QString("newton"))
		f = new newton(std::complex<float>(), iterations);
	else
	if (fractaleName == QString("julia1"))
		f = new julia(std::complex<float>(.285f,0), iterations);
	else
	if (fractaleName == QString("julia1"))
		f = new julia(std::complex<float>(-0.038088f, 0.9754633f), iterations);
	else
	if (fractaleName == QString("tricorn"))
		f = new tricorn(std::complex<float>(), iterations);
	else
	if (fractaleName == QString("multibrot"))
		f = new hexamandelbrot(std::complex<float>(), iterations);
	else
	if (fractaleName == QString("sharingan"))
		f = new sharingan(std::complex<float>(), iterations);
	else
	if (fractaleName == QString("bateau"))
		f = new bateauEnFeu(std::complex<float>(), iterations);
	else
		throw 500;
		

	ImageTiler i(result, mutex, window, resolution, f, isRandom, nbRandom, offset, step, offset);
	i.run();
	i.wait();

	QImage *image = new QImage(resolution, QImage::Format_RGB888);
	
	for (int x = 0; x< resolution.width(); x++)
		for (int y=0; y<resolution.height(); y++)
			image->setPixel(x,y, ImageCalculator::intToRGBA(result[x][y]));			
		
	image->save(output);
}

NetworkAgent::~NetworkAgent(){

}

int NetworkAgent::process(CommandLineArguments& parser){
	QSize resolution = parser.getResolution();
	if (resolution.width() <= 0)
		return 0;
	QRectF window = parser.getWindow();
	if (window.top() == 0)
		return 0;
	bool useRandom = parser.getAleatoire();
	int offset = parser.getOffset();
	if (offset < 0)
		return 0;
	int iterations = parser.getIterations();
	if (iterations < 0)
		return 0;
	int nbRandom = 0;
	if (useRandom)
		nbRandom = parser.getRandomCount();
	
	
	QString output = parser.getOutput();
	if (output == QString(""))
		output = "img.png";
	
	QVector<QVector<long> > result;
	
	for (int j=0; j<resolution.width(); j++){
		QVector<long> v;
		v.fill(0,resolution.height());			
		result.push_back(v);
	}		

	try{
		NetworkAgent agent(result, resolution, window, useRandom, parser.getFractale(), offset, iterations, nbRandom, output, parser.getStep());		
		std::cout << "ok";
		return 0;
	}catch(int){
	}	
	return 0;
}