/*
Copyright 2010 Pierre SCHWARTZ

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated 
documentation files (the "Software"), to deal in the Software without restriction, including without 
limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies 
of the Software, and to permit persons to whom the Software is furnished to do so, subject to the following 
conditions: The above copyright notice and this permission notice shall be included in all copies or 
substantial portions of the Software. THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF 
OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
*/

/*!
 * \file triplex.cpp
 * \brief corps de la classe de gestion de points 3D
 * \author Pierre Schwartz
 * \version 1.0
 */

#include "triplex.h"
#include <cmath>

/*!
 * \brief Constructeur
 */
triplex::triplex(){
}

/*!
 * \brief Constructeur
 */
triplex::triplex(float a, float b, float c):a(a), b(b), c(c){
}

/*!
 * \brief Oprateur =
 */
triplex triplex::operator=(const triplex& t){
	a = t.a;
	b = t.b;
	c = t.c;
	return *this;
}

/*!
 * \brief Constructeur par recopie
 */
triplex::triplex(const triplex& t){
	a = t.a;
	b = t.b;
	c = t.c;
}

/*!
 * \brief Addition vectorielle
 */
triplex triplex::operator+(triplex& t){
	triplex newt;
	newt.a = a+t.a;
	newt.b = b+t.b;
	newt.c = c+t.c;
	return newt;
}

/*!
 * \brief Soustraction vectorielle
 */
triplex triplex::operator-(triplex& t){
	triplex newt;
	newt.a = a-t.a;
	newt.b = b-t.b;
	newt.c = c-t.c;
	return newt;
}

/*!
 * \brief Changement d'chelle par un flottant
 */
triplex triplex::operator/(float f){
	triplex newt;
	if (f != 0){
		newt.a = a/f;
		newt.b = b/f;
		newt.c = c/f;
	}
	return newt;
}

/*!
 * \brief Changement d'chelle par un flottant
 */
triplex triplex::operator*(float f){
	triplex newt;
	newt.a = a*f;
	newt.b = b*f;
	newt.c = c*f;
	return newt;
}

/*!
 * \brief Normalise le vecteur (sa norme sera de 1)
 */
triplex triplex::normalize(){
	triplex t(*this);
	
	float norm = sqrt(t.norm2());
	return t/norm;
}

/*!
 * \brief Oprateur  interne dans l'espace (1,i,j)
 */
triplex triplex::square(){
	triplex newt;
	newt.a = a*a-b*b-c*c;
	newt.b = 2*a*b - b*c;
	newt.c = 2*a*c + b*c;
	return newt;
}
/*!
 * \brief Retourne la norme au carr
 */
float triplex::norm2(){
	return a*a + b*b + c*c;
}

/*!
 * \brief Fait une rotation vectorielle autour de l'axe X : passage dans la matrice de rotation 3x3
 */
triplex triplex::rotateX(float f){
	triplex newt(a,
				 b*cos(f) - c*sin(f),
				 b*sin(f) + c*cos(f));
	return newt;
}


/*!
 * \brief Fait une rotation vectorielle autour de l'axe Y : passage dans la matrice de rotation 3x3
 */
triplex triplex::rotateY(float f){
	triplex newt(a*cos(f) + c*sin(f),
				 b,
				 -a*sin(f) + c*cos(f)
				 );

	return newt;
}

/*!
 * \brief Fait une rotation vectorielle autour de l'axe Z : passage dans la matrice de rotation 3x3
 */
triplex triplex::rotateZ(float f){
	triplex newt(a*cos(f) - b*sin(f),
				 a*sin(f) + b*cos(f),
				 c);

	return newt;
}

/*!
 * \brief Produit scalaire
 */
float triplex::scalar(triplex& f){
	return f.a*a + f.b*b + f.c*c;
}