#ifndef COLORATOR_H
#define COLORATOR_H

#include <QObject>
#include <QImage>
#include <QPixmap>

#include "fractalinfo.h"

/*!
 *	\class ColorFunction
 *	Cette interface dfinit une fonction pour coloriser l'image
 */
class ColorFunction {
public:
	virtual QPixmap colorImage(FractalInfo * info) = 0;
};

/*!
 *	\class Colorator
 *	Cette classe permet de mettre de la couleur sur une image fractale
 *	Cette classe sert de factory. Pour ajouter une nouvelle faon,
 *	il faut rajouter une classe et lui donner un nom sous forme de string.
 *	Ce nom permet de choisir la mthode.
 *	@see QObject
 */
class Colorator : public QObject
{
	QHash<QString, ColorFunction*> _functions;
	QString _default;

public:
	explicit Colorator(QObject *parent = 0);
	virtual ~Colorator();

	/// retourne la liste des colorations disponibles
	QStringList functionList() { return QStringList(_functions.keys()); }
	/// fournit une image colorise  partir de l'objet FractalInfo
	QPixmap colorImage(FractalInfo * info, QString function = "" );
	/// ajoute dynamiquement un objet de coloration
	void addFunction(QString name, ColorFunction * function);
};

/*** IMPLEMENTATION ***/

/*!
 *	\class BWColor
 *	Colorisation noir et blanc
 *
 *	@see ColorFunction
 */
class BWColor : public ColorFunction {
	virtual QPixmap colorImage(FractalInfo * info) {
		QImage image(info->imageSize(),QImage::Format_RGB32);

		for( int i = 0; i < info->imageSize().width(); i++ ) {
			for( int j = 0; j < info->imageSize().height(); j++ ) {
				if( info->pixel(i,j) == 0 )
					image.setPixel(i,j,QColor(255,255,255).rgb());
				else
					image.setPixel(i,j,QColor(0,0,0).rgb());
			}
		}

		return QPixmap::fromImage(image);
	}
};

/*!
 *	\class GrayColor
 *	Colorisation en niveau de gris
 *
 *	@see ColorFunction
 */
class GrayColor : public ColorFunction {
	virtual QPixmap colorImage(FractalInfo * info) {
		QImage image(info->imageSize(),QImage::Format_RGB32);

		for( int i = 0; i < info->imageSize().width(); i++ ) {
			for( int j = 0; j < info->imageSize().height(); j++ ) {
				if( info->pixel(i,j) == 0 )
					image.setPixel(i,j,QColor(255,255,255).rgb());
				else {
					double tmp = info->pixel(i,j)/(double)info->max();
					tmp = 1-tmp;
					image.setPixel(i,j,QColor::fromRgbF(tmp,tmp,tmp).rgb());
				}
			}
		}

		return QPixmap::fromImage(image);
	}
};

/*!
 *	\class GrayV2Color
 *	Colorisation en niveau de gris
 *
 *	@see ColorFunction
 */
class GrayV2Color : public ColorFunction {
	virtual QPixmap colorImage(FractalInfo * info) {
		uint inColor = QColor(255,255,255).rgb();
		QColor black(0,0,0);

		QList<uint> colors;
		for( int i = 0; i < 50; i++) {
			double tmp = i/50.0;
			colors << QColor::fromRgbF(tmp,tmp,tmp).rgb();
		}

		QImage image(info->imageSize(),QImage::Format_RGB32);

		for( int i = 0; i < info->imageSize().width(); i++ ) {
			for( int j = 0; j < info->imageSize().height(); j++ ) {
				if( info->pixel(i,j) == 0 )
					image.setPixel(i,j,inColor);
				else
					image.setPixel(i,j,colors[ (info->pixel(i,j)-1) % colors.size()]);
			}
		}

		return QPixmap::fromImage(image);
	}
};

/*!
 *	\class GrayV2RevertColor
 *	Colorisation en niveau de gris ( invers par rapport au GrayV2Color )
 *
 *	@see ColorFunction
 */
class GrayV2RevertColor : public ColorFunction {
	virtual QPixmap colorImage(FractalInfo * info) {
		uint inColor = QColor(255,255,255).rgb();
		QColor black(0,0,0);

		QList<uint> colors;
		for( int i = 0; i < 50; i++) {
			double tmp = 1.0 - i/50.0;
			colors << QColor::fromRgbF(tmp,tmp,tmp).rgb();
		}

		QImage image(info->imageSize(),QImage::Format_RGB32);

		for( int i = 0; i < info->imageSize().width(); i++ ) {
			for( int j = 0; j < info->imageSize().height(); j++ ) {
				if( info->pixel(i,j) == 0 )
					image.setPixel(i,j,inColor);
				else
					image.setPixel(i,j,colors[ (info->pixel(i,j)-1) % colors.size()]);
			}
		}

		return QPixmap::fromImage(image);
	}
};

/*!
 *	\class ManyColor
 *	Colorisation pour alternance de couleurs
 *
 *	@see ColorFunction
 */
class ManyColor : public ColorFunction {
	virtual QPixmap colorImage(FractalInfo * info) {
		QColor black(0,0,0);

		QList<QColor> colors;
		colors << Qt::red << Qt::darkRed <<Qt::green <<Qt::darkGreen
				<<Qt::blue <<Qt::darkBlue <<Qt::cyan <<Qt::darkCyan
				<<Qt::magenta <<Qt::darkMagenta <<Qt::yellow <<Qt::darkYellow;

		QImage image(info->imageSize(),QImage::Format_RGB32);

		for( int i = 0; i < info->imageSize().width(); i++ ) {
			for( int j = 0; j < info->imageSize().height(); j++ ) {
				if( info->pixel(i,j) == 0 )
					image.setPixel(i,j,black.rgb());
				else
					image.setPixel(i,j,colors[ (info->pixel(i,j)-1) % colors.size()].rgb());
			}
		}

		return QPixmap::fromImage(image);
	}
};

/*!
 *	\class ManyV2Color
 *	Colorisation par fondu de couleur
 *
 *	@see ColorFunction
 */
class ManyV2Color : public ColorFunction {
	virtual QPixmap colorImage(FractalInfo * info) {
		QColor black(0,0,0);

		QList<uint> red;
		QList<uint> green;
		QList<uint> blue;
		for( int i = 0; i < 50; i++) {
			double tmp = i/50.0;
			red << QColor::fromRgbF(tmp,0.,0.).rgb();
			green << QColor::fromRgbF(0.,tmp,0.).rgb();
			blue << QColor::fromRgbF(0.,0.,tmp).rgb();
		}

		QList<uint> colors;
		colors << blue << green << red;

		QImage image(info->imageSize(),QImage::Format_RGB32);

		for( int i = 0; i < info->imageSize().width(); i++ ) {
			for( int j = 0; j < info->imageSize().height(); j++ ) {
				if( info->pixel(i,j) == 0 )
					image.setPixel(i,j,black.rgb());
				else
					image.setPixel(i,j,colors[ (info->pixel(i,j)-1) % colors.size()]);
			}
		}

		return QPixmap::fromImage(image);
	}
};

#endif // COLORATOR_H
