#include "fractalbuilder.h"

FractalBuilder::FractalBuilder(QRectF fractaleSize, QSize imageSize, QObject * parent) :
		QThread(parent)
{
	_info = new FractalInfo(fractaleSize,imageSize, parent);

	_watcher = new QFutureWatcher<void>(this);

	connect(_watcher, SIGNAL(progressValueChanged(int)), SLOT(progress(int)));
	connect(this,SIGNAL(terminated()), _watcher, SLOT(cancel()));
}

struct TestMap
{
	TestMap(Fractal * fractale, Accumulator * accu)
	 : _fractale(fractale), _accu(accu) { }

	void operator() (Complex &tmp)
	{
		QVector<Complex> sequence(_fractale->limit());
		_fractale->genererSequence(tmp,sequence);
		_accu->accumuler(tmp,sequence);
		sequence.clear();
	}

	Fractal * _fractale;
	Accumulator * _accu;
};

void FractalBuilder::run()
{
	//TODO iterator
	QList<Complex> tmp;
	while( _generator->next() )
		tmp << _generator->getPoint();

	tmp.begin();

	QFuture<void> future = QtConcurrent::map(tmp,TestMap(_fractal,_accumulator));

	emit setRange(future.progressMinimum(),future.progressMaximum());
	_watcher->setFuture(future);

	future.waitForFinished();
}

void FractalBuilder::setAccumulator(QString name, bool autoParam)
{
	if( ! _accumulator.isNull() )
		delete _accumulator;

	_accumulator = AccumulatorFactory::instance(parent())->accumulator(name,_info,autoParam);
}

void FractalBuilder::setFractal(QString name, int limit, bool autoParam)
{
	if( ! _fractal.isNull() )
		delete _fractal;

	_fractal = FractalFactory::instance(parent())->fractal(name,limit,autoParam);
}

void FractalBuilder::setGenerator(QString name, bool autoParam)
{
	if( ! _generator.isNull() )
		delete _generator;

	_generator = GeneratorFactory::instance(parent())->generator(name,_info,autoParam);
}

void FractalBuilder::progress(int value)
{
	emit progressValueChanged(value);
}



QStringList FractalBuilder::_definedFractalNames
		= QStringList() << tr("Buddhabrot") << tr("Nova")
		  << tr("Crabe") << tr("Newton") << tr("Bateau en feu")
		  << tr("Bateau en feu (zoom)")
		  << tr("Mandebrot") << tr("Tricorn")
		  << tr("Sharingan") << tr("Julian")
		  << tr("Julian (2)");

QStringList FractalBuilder::definedFractalNames()
{
	return _definedFractalNames;
}

FractalBuilder * FractalBuilder::build(QString name, QSize size, QObject * parent)
{
	int limit = 1000;

	if( name == _definedFractalNames[0] ) {
		FractalBuilder * build = new FractalBuilder(
				QRectF(-2,-2,4,4),
				size,
				parent);

		build->_accumulator = new SequenceAccumulator(build->info(),parent);
		build->_fractal = new Mandelbrot(Complex(),limit,parent);
		build->_generator = new RandomPointGenerator(10*1000000,build->info(),parent);

		return build;
	}

	if( name == _definedFractalNames[1] ) {
		FractalBuilder * build = new FractalBuilder(
				QRectF(-1.5,-1.5,3,3),
				size,
				parent);

		build->_accumulator = new OriginAccumulator(build->info(),parent);
		build->_fractal = new Nova(limit,parent);
		build->_generator = new ImageGenerator(build->info(),parent);

		return build;
	}

	if( name == _definedFractalNames[2] ) {
		FractalBuilder * build = new FractalBuilder(
				QRectF(-1.5,-1.5,3,3),
				size,
				parent);

		build->_accumulator = new SequenceAccumulator(build->info(),parent);
		build->_fractal = new Nova(limit,parent);
		build->_generator = new RandomPointGenerator(10*1000000,build->info(),parent);

		return build;
	}

	if( name == _definedFractalNames[3] ) {
		FractalBuilder * build = new FractalBuilder(
				QRectF(-10,-10,20,20),
				size,
				parent);

		build->_accumulator = new OriginAccumulator(build->info(),parent);
		build->_fractal = new Newton(limit,parent);
		build->_generator = new ImageGenerator(build->info(),parent);

		return build;
	}

	if( name == _definedFractalNames[4] ) {
		FractalBuilder * build = new FractalBuilder(
				QRectF(-2,-2,4,4),
				size,
				parent);

		build->_accumulator = new OriginAccumulator(build->info(),parent);
		build->_fractal = new Boat(limit,parent);
		build->_generator = new ImageGenerator(build->info(),parent);

		return build;
	}

	if( name == _definedFractalNames[5] ) {
		FractalBuilder * build = new FractalBuilder(
				QRectF(-1.75,-0.125,0.25,0.25),
				size,
				parent);

		build->_accumulator = new OriginAccumulator(build->info(),parent);
		build->_fractal = new Boat(limit,parent);
		build->_generator = new ImageGenerator(build->info(),parent);

		return build;
	}

	if( name == _definedFractalNames[6] ) {
		FractalBuilder * build = new FractalBuilder(
				QRectF(-2,-2,4,4),
				size,
				parent);

		build->_accumulator = new OriginAccumulator(build->info(),parent);
		build->_fractal = new Mandelbrot(Complex(),limit,parent);
		build->_generator = new ImageGenerator(build->info(),parent);

		return build;
	}

	if( name == _definedFractalNames[7] ) {
		FractalBuilder * build = new FractalBuilder(
				QRectF(-2,-2,4,4),
				size,
				parent);

		build->_accumulator = new OriginAccumulator(build->info(),parent);
		build->_fractal = new Tricorn(limit,parent);
		build->_generator = new ImageGenerator(build->info(),parent);

		return build;
	}

	if( name == _definedFractalNames[8] ) {
		FractalBuilder * build = new FractalBuilder(
				QRectF(-2,-2,4,4),
				size,
				parent);

		build->_accumulator = new SequenceAccumulator(build->info(),parent);
		build->_fractal = new Tricorn(limit,parent);
		build->_generator = new RandomPointGenerator(10*1000000,build->info(),parent);

		return build;
	}
	
	if( name == _definedFractalNames[9] ) {
		FractalBuilder * build = new FractalBuilder(
				QRectF(-2,-2,4,4),
				size,
				parent);

		build->_accumulator = new OriginAccumulator(build->info(),parent);
		build->_fractal = new Julian(Complex(-0.038088,-0.9754633),limit,parent);
		build->_generator = new ImageGenerator(build->info(),parent);

		return build;
	}

	if( name == _definedFractalNames[10] ) {
		FractalBuilder * build = new FractalBuilder(
				QRectF(-2,-2,4,4),
				size,
				parent);

		build->_accumulator = new OriginAccumulator(build->info(),parent);
		build->_fractal = new Julian(Complex(0.285,0),limit,parent);
		build->_generator = new ImageGenerator(build->info(),parent);

		return build;
	}

	qCritical() << "Unknow FractalBuilder name : " << name;
	return NULL;
}
