/*
# Copyright (c) 2009 Yan Verdavaine
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
*/
#ifndef GACCUMULATEUR_HPP
#define GACCUMULATEUR_HPP
#include "Commun.hpp"
/*!
 * \file GFractale.hpp
 * \brief définition de la classe représantant une suite fractale et quelques exemples.
 * \author Yan Verdavaine
 * \version 1.0
 */

namespace Fractale
{


/*! \class GAccumulateur
 *  \brief Classe représentant la méthode pour accumuler la séquence dans l'image finale.\n
 *  Le sens des lignes de l'image correspond au sens inverse de l'axe des imaginaire :
 *  \li Le point pixel (0,0) correspond au point en bas à gauche du plan complexe.
 *  \li Le point pixel (imgH -1 ,imgL - 1)  correspond au point en bas à droite du plan complexe.
 *
 */
struct GAccumulateur
{
    zone                m_zone; /*!< Zone de l'image dans le plan complexe */

    /*!
     *  \brief Convertit le point complexe en coordonnée I dans l'image.
     *
     *  \param z : le point complexe.
     *  \return Coordonée I dans l'image.
     */
    unsigned int toI(const point &z )
    {
        return  static_cast<unsigned int>
        (

            (m_zone.imgH - 1)
            *
            (z.imag() - m_zone.y)
            /
            m_zone.h
        );
    }
    /*!
     *  \brief Convertit le point complexe en coordonée J de l'image.
     *
     *  \param z : le point complexe.
     *  \return Coordonnée J dans l'image.
     */
    unsigned int toJ(const point &z)
    {
        return static_cast<unsigned int>
        (

            (m_zone.imgL - 1)
            *
            (z.real() - m_zone.x)
            /
            m_zone.l
        );
    }

    /*!
     *  \brief Accumule la séquence dans l'image.
     *
     *  \param res : image résultante. Sa taille doit être la même que celle définie dans m_zone.
     *  \param pInit : point d'initialisation.
     *  \param seq : séquence de la suite fractale.
     */
    virtual void accumuler(Image<unsigned int> &res, const point &pInit, const std::vector<point> &seq) = 0;

};


/*! \class GAccumulateur
 *  \brief Classe permettant d'ajouter 1 sur tous les pixels de l'image correspondant aux points de séquence
 */
struct AccumulerSurLaSequence : public GAccumulateur
{

    /*!
     *  \brief Accumule la séquence dans l'image.
     *   Parcourt la séquence et ajoute 1 au pixel correspondant de l'image
     *
     *  \param resultat : image résultante. Sa taille doit être la même que celle définie dans m_zone.
     *  \param pInit : point d'initialisation (non utilisé).
     *  \param sequence : séquence de la suite fractale.
     */
    void accumuler(Image<unsigned int> &resultat, const point &pInit,const std::vector<point> & sequence)
    {
        std::vector<point>::const_iterator it = sequence.begin();
        std::vector<point>::const_iterator end = sequence.end();
        while (it != end)
        {
           //conversion du point complexe en coordonnée image
           unsigned int i = toI(*it);
           unsigned int j = toJ(*it);
           //si le point image (i,j) est dans l'image résultante
           if(resultat.isValid(i,j))
           {
                //ajoute 1 au pixel (i,j)
                ++ resultat.pixel(i,j);
           }
           //point suivant de la séquence
           ++it;
        }
    }
};


/*! \class GAccumulateur
 *  \brief Classe permettant d'ajouter la taille de séquence sur le pixel de l'image correspondant aux points d'initialisation
 */
struct AccumulerSurOrigine : public GAccumulateur
{
    void accumuler(Image<unsigned int> & resultat,const point & pInit,const std::vector<point> & sequence)
    {
        if(sequence.size() > 0 )
        {
            //conversion du point complexe en coordonnée image
            unsigned int i = toI(pInit);
            unsigned int j = toJ(pInit);
            //si le point image (i,j) est dans l'image resultat
            if(resultat.isValid(i,j))
            {
                //ajoute la taille de la séquence au pixel (i,j)
                resultat.pixel(i,j) += sequence.size() ;
            }
        }
    }
};
}
#endif
