/*
# Copyright (c) 2009 Yan Verdavaine
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
*/

#include "Generateur.hpp"
#include <limits>
#include <algorithm>
#include <fstream>
using namespace Fractale;
/*!
 * \file GFractale.hpp
 * \brief Définition de la classe représantant une suite fractale et quelques exemples.
 * \author Yan Verdavaine
 * \version 1.0
 */


/*! \class UintToUchar
   * \brief Foncteur qui permet de convertir un unsigned int en unsigned char.
   *
   */
struct UintToUchar
{
    unsigned int max; /*!< Valeur max d'entrée  */
    double       gamma;/*!< Exposant gamma  */
    UintToUchar(unsigned int max = std::numeric_limits<unsigned int>::max() ,double gamma = .5 )
        :max(max),gamma(gamma)
    {
    }

    unsigned char operator ()(unsigned int t)
    {
        if (t > max) t = max;
        unsigned int res = static_cast<unsigned int>(255. *std::pow(double(t) / max, gamma) +.4);
        return res > 255 ? 255 : res;
    }

};


void EcrireImage( const Image<unsigned int> & img, const std::string & file,double gamma = .4)
{

    //récuperation de la valeur max de l'image
    unsigned int max = *std::max_element(img.begin(),img.end());

    //conversion en image en unsigned char
    Image<unsigned char> I(img.hauteur(),img.largeur());
    std::transform(img.begin(),img.end(),I.begin(),UintToUchar(max,gamma));

    //écriture de l'image au format pgm binaire
    std::ofstream out(file.c_str(),std::ios::binary);
    out <<"P5" <<std::endl;
    out << img.largeur() << " " << img.hauteur() << std::endl;
    out << "255" << std::endl;

    out.write((char *)&I.pixel(0,0),I.hauteur() * I.largeur());
}


int main(int /*argc*/, char* /*argv*/[])
{

        std::cout << "Generation de CliffordAttractors" << std::endl;
        EcrireImage
        (
            TracerCourbe(CliffordAttractors(),10000,point(0.,0.),zone(-3., -3., 6.,6., 1000, 1000)),
            "CliffordAttractors.pgm",
            .2
        );


        std::cout << "Generation de CliffordAttractors2" << std::endl;
        EcrireImage
        (
            TracerCourbe(CliffordAttractors(1.7,1.7,1.2,1.7),10000,point(0.,0.),zone(-3., -3., 6.,6., 1000, 1000)),
            "CliffordAttractors2.pgm",
            .2
        );

        std::cout << "Generation de Nova" << std::endl;
        EcrireImage
        (
            Generateur(PointImage(),Nova(),AccumulerSurOrigine(),zone(-1.5, -1.5, 3., 3., 1000, 1000)),
            "Nova.pgm"
        );

        std::cout << "Generation du Crabe" << std::endl;
        EcrireImage
        (
            Generateur(PointAleatoire(), Nova(), AccumulerSurLaSequence(), zone(-1.5, -1.5, 3., 3., 1000, 1000)),
            "Crabe.pgm",
            .4
        );

        std::cout << "Generation de Newton" << std::endl;
        EcrireImage
        (
            Generateur(PointImage(),Newton(),AccumulerSurOrigine(),zone(-10., -10., 20., 20., 1000, 1000)),
            "Newton.pgm"
        );

        std::cout << "Generation de BateauEnFeu" << std::endl;
        EcrireImage
        (
            Generateur(PointImage(), BateauEnFeu(), AccumulerSurOrigine(), zone(-2.,-2.,4.,4.,1000,1000)),
            "BateauEnFeu.pgm"
        );

        std::cout << "Generation de BateauEnFeu_Zoom" << std::endl;
        EcrireImage
        (
            Generateur(PointImage(), BateauEnFeu(), AccumulerSurOrigine(), zone(-1.75,-.125,.25,.25,1000,1000)),
            "BateauEnFeu_Zoom.pgm"
        );

        std::cout << "Generation de Mandelbrot" << std::endl;
        EcrireImage
        (
            Generateur(PointImage(),Mandelbrot(),AccumulerSurOrigine(),zone(-2.,-2.,4.,4.,1000,1000)),
            "Mandelbrot.pgm"
        );

        std::cout << "Generation de Buddhabrot" << std::endl;
        EcrireImage
        (
            Generateur(PointAleatoire(), Mandelbrot(), AccumulerSurLaSequence(), zone(-2.,-2.,4.,4.,1000,1000)),
            "Buddhabrot.pgm",
            .8
        );

        std::cout << "Generation de Tricorn" << std::endl;
        EcrireImage
        (
            Generateur(PointImage(), Tricorn(), AccumulerSurOrigine(), zone(-2.,-2.,4.,4.,1000,1000)),
            "Tricorn.pgm"
        );

        std::cout << "Generation du Sharingan" << std::endl;
        EcrireImage
        (
            Generateur(PointAleatoire(), Tricorn(), AccumulerSurLaSequence(), zone(-2.,-2.,4.,4.,1000,1000)),
            "Sharingan.pgm",
            .8
        );

        std::cout << "Generation de Julian1" << std::endl;
        EcrireImage
        (
            Generateur(PointImage(), Julian(point(-0.038088, 0.9754633)), AccumulerSurOrigine(), zone(-2.,-2.,4.,4.,1000,1000)),
            "Julian1.pgm"
        );

        std::cout << "Generation de Julian2" << std::endl;
        EcrireImage
        (
            Generateur(PointImage(), Julian(point(0.285, 0)), AccumulerSurOrigine(), zone(-2.,-2.,4.,4.,1000,1000)),
            "Julian2.pgm"
        );
		
	return 0;
}

