/*
# Copyright (c) 2009 Yan Verdavaine
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
*/

#ifndef COMMUN_HPP
#define COMMUN_HPP
#include <complex>
/*!
 * \file GFractale.hpp
 * \brief Définition de la classe représantant une suite fractale et quelques exemples.
 * \author Yan Verdavaine
 * \version 1.0
 */

namespace Fractale
{


/*! \class Random
 *  \brief Générateur de nombre aléatoire. Peut générer :
 *  \li unsigned long long : un entier positif sur 64 bit
 *  \li unsigned int : un entier positif sur 32 bit
 *  \li double : un réel dans [0,1].
 *
 *  Adaptation du code C# : http://www.sciencecode.com/Science/Mathematics/VectorCalculus/IntegrationMultiD.cs
 */
struct Random
{
    unsigned long long u,v,w;
    /*!
     *  \brief Constructeur
     *  \param j : graine de la suite aléatoire
     */
    Random(unsigned long long j = 0)
    : v(4101842887655102017LL),
      w(1)
    {
        u = j ^ v; generateUInt64();
        v = u; generateUInt64();
        w = v; generateUInt64();
    }
    /*!
     *  \brief Génère un entier aléatoire.
     *  \return Un entier non signé sur 64 bits.
     */
    unsigned long long generateUInt64()
    {
        u = u * 2862933555777941757LL + 7046029254386353087LL;
        v ^= v >> 17; v ^= v << 31; v ^= v >> 8;
        w = 4294957665U*(w & 0xffffffff) + (w >> 32);
        unsigned long long x = u ^ (u << 21); x ^= x >> 35; x ^= x << 4;
        return (x + v) ^ w;
    }

    /*!
     *  \brief Génère un double aléatoire.
     *  \return Un double dans [0,1[.
     */
    double generateDouble()
    {
        return   5.42101086242752217E-20 * generateUInt64();
    }
    /*!
     *  \brief Génère un entier aléatoire.
     *  \return Un entier non signé sur 32 bits.
     */
    unsigned int generateUInt32()
    {
        return (unsigned int)generateUInt64();
    }
};


/*! \class zone
 *  \brief Définition entre une zone dans le plan complexe et une image.\n
 */
struct zone
{
    double        x;   /*!< Valeur réelle la plus petite de la zone complexe */
    double        y;   /*!< Valeur complexe la plus petite de la zone complexe */
    double        h;   /*!< Hauteur de la zone complexe */
    double        l;   /*!< Largeur de la zone complexe */
    unsigned int  imgH;/*!< Hauteur de l'image */
    unsigned int  imgL;/*!< Largeur de l'image */

    /*!
     *  \brief Constructeur d'une zone.
     *  Initialise les valeur de la zone.
     */
    zone(double x = 0.,double y =0.,double h = 0., double l = 0.,unsigned int imgH = 0,unsigned int imgL = 0)
        :x(x),y(y),h(h),l(l),imgH(imgH),imgL(imgL)
    {
    }
    /*!
     *  \brief Constructeur de copie d'une zone.
     *  Initialise les valeurs de la zone.
     */
    zone(const zone &z)
        :x(z.x), y(z.y), h(z.h), l(z.l), imgH(z.imgH), imgL(z.imgL)
    {}
};

/*! \var typedef std::complex<double> point
 *  \brief Point dans le plan complexe : 
 *  x est la partie réelle,
 *  y la partie imaginaire.
*/
typedef std::complex<double> point;

}
#endif
