/*
# Copyright (c) 2009 Yan Verdavaine
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
*/
#ifndef GPOINTINITIALISATION_HPP
#define GPOINTINITIALISATION_HPP
#include "Commun.hpp"
#include <iostream>
#include <iomanip>
#include <cstdlib>
#include <vector>
/*!
 * \file GFractale.hpp
 * \brief Définition de la classe représantant une suite fractale et quelques exemples.
 * \author Yan Verdavaine
 * \version 1.0
 */

namespace Fractale
{


/*! \class GPointInitialisation
   * \brief Classe qui génère des points d'initialisation utilisés pour une suite fractale.
   *
   */
struct GPointInitialisation
{
    zone    m_zone; /*!< Zone de l'image dans le plan complexe */

    /*!
     *  \brief Positionne la génération des points d'initialisation au début.
     */
    virtual void    debut() = 0;
    /*!
     *  \brief Donne le point d'initialisation courant et passe au suivant.
     *  \return Nouveau point d'initialisation.
     */
    virtual point   nouveauPoint() =0;

    /*!
     *  \brief Permet de savoir si tous les points d'initialisation ont été donnés.
     *  \return True si tous les points d'initialisation ont été donnés.
     */
    virtual bool    fin() = 0;
};


/*! \class PointAleatoire
   * \brief classe qui genère des points d'initialisation aléatoirement sur la zone.
*/
class PointAleatoire : public GPointInitialisation
{
    const unsigned long int nbMPointMax;    /*!< nombre de  point maximum à générer. En million. */
    unsigned long int       nbMPoint;       /*!< nombre de  point en million déjà généré. */
    unsigned long int       nbPoint;        /*!< nombre de  point généré entre 0 et 1 million*/
    Random                  myrand;         /*!< générateur de nombre aléatoire entre [0,1.[*/
    double                  facteurZone;    /*!< facteur d'élargissement de la zone de l'image dans le plan complexe. Permet d'avoir une zone différente pour générer les points aléatoire. */

public:
    /*!
     *  \brief Constructeur
     *
     *  \param nbMPointMax : nombre de point à générer. En million de point.
     *  \param facteurZone : facteur d'élargissement de la zone de l'image dans le plan complexe.
     */
    PointAleatoire(unsigned long int nbMPointMax = 100,double facteurZone = 1.)
        :nbMPointMax(nbMPointMax),facteurZone(facteurZone)
    {}

    void    debut()
    {
        //initialise le nombre de point générés à zero.
        nbPoint = nbMPoint = 0;
    }

    point   nouveauPoint()
    {
        //incrément de nombre de point entre 0 et 1 million.
        ++nbPoint;
        if( (nbMPoint * 1000000 + nbPoint) % 1000 == 0)
        {
            std::cout << std::setprecision(2) << std::fixed  << std::setw(5) << std::left << 100.* (nbMPoint * 1000000. + nbPoint) /(nbMPointMax* 1000000.) << " %                   \r" << std::flush;
        }
        //tous les million de point on incrémente nbMPointMax.
        if( nbPoint >= 1000000)
        {
            ++nbMPoint;
            nbPoint = 0;
        }
        //génère un point aléatoire dans le plan complexe définie par zone et un facteur d'élargissement.
        return point
                (
                    m_zone.x - (facteurZone * m_zone.l  - m_zone.l) / 2. + facteurZone * m_zone.l * myrand.generateDouble(),
                    m_zone.y - (facteurZone * m_zone.h  - m_zone.l) / 2. + facteurZone * m_zone.h * myrand.generateDouble()
                );
    }
    bool    fin()
    {
        //true si tous les points ont étés générés
        return nbMPoint >= nbMPointMax;
    }
};

/*! \class PointImage
   * \brief classe qui convertie toutes les coordonnées des pixels de l'image en points d'initialisation.
   */
class PointImage : public GPointInitialisation
{
    unsigned int i; /*!< Coordonnées i courant de l'image. */
    unsigned int j; /*!< Coordonnées j courant de l'image. */

public:
    void    debut()
    {
        //initialise les coordonné i,j courant à zéro.
        i = j = 0;
    }

    point   nouveauPoint()
    {
        //convertie la coordonnée image (i,j) en point complexe dans la zone.
        const point p
                (
                        m_zone.x + m_zone.l * (j + .5) / (m_zone.imgL - 1),
                        m_zone.y + m_zone.h * (i + .5) / (m_zone.imgH - 1)
                );
        //incrémente j courant.
        ++j;
        //Si on arrive au dernier pixel de la ligne, on passe à la ligne suivante.
        if (j >= m_zone.imgL)
        {
            std::cout << std::setprecision(2) << std::fixed  << std::setw(5) << std::left << 100.* i /m_zone.imgH  << " %                   \r" << std::flush;
            ++i;
            j = 0;
        }

        return p;
    }

    bool    fin()
    {
         //true si toutes l'image as été parcourue.
        return i >= m_zone.imgH ;
    }
};


}
#endif
