/*
# Copyright (c) 2009 Yan Verdavaine
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
*/
#ifndef GENERATEUR_HPP
#define GENERATEUR_HPP
#include "Image.hpp"
#include "GPointInitialisation.hpp"
#include "GFractale.hpp"
#include "GAccumulateur.hpp"
/*!
 * \file Generateur.hpp
 * \brief définition de la classe représantant une suite fractale et quelques exemples.
 * \author Yan Verdavaine
 * \version 1.0
 */

namespace Fractale
{
/**
 * \brief Génère une image fractale.
 *
 * \tparam T doit être un GPointInitialisation.
 * \tparam T2 doit être un GFractale.
 * \tparam T3 doit être un GAccumulateur.
 * \param  zoneImage : zone du plan complexe et taille de l'image voulue.
 * \return L'image résultante.
 */
template<typename T,typename T2,typename T3>
Image<unsigned int> Generateur
(
    T t,
    T2 t2,
    T3 t3,
    zone zoneImage
)
{
    GPointInitialisation &  in(t);
    GFractale  &fractale(t2);
    GAccumulateur& accumulateur(t3);

    in.m_zone               = zoneImage;
    accumulateur.m_zone     = zoneImage;

    in.debut();

    Image<unsigned int> resultat(zoneImage.imgH,zoneImage.imgL);
    while(!in.fin())
    {
        point z = in.nouveauPoint();

        const std::vector<point> & sequence = fractale.genererSequence(z);
        accumulateur.accumuler(resultat,z,sequence);
    }
    std::cout << std::endl;
    return resultat;

}


/**
 * \brief Dessine la suite de points d'une suite fractale.
 *
 * \tparam T doit être un GFractale. Cette fractale ne doit pas avoir de fin et la suite générée ne doit pas être nulle.
 * \param nbLoop : nombre d'itération de continuité de la suite fractale.
 * \param z : point de départ.
 * \param  zoneImage : zone du plan complexe et taille de l'image voulue.
 * \return L'image résultante.
 */
template<typename T>
Image<unsigned int> TracerCourbe
(
    T t,
    unsigned int nbLoop,
    point z,
    zone zoneImage
)
{
    GFractale  &fractale(t);
    AccumulerSurLaSequence accumulateur;

    accumulateur.m_zone             = zoneImage;



    Image<unsigned int> resultat(zoneImage.imgH,zoneImage.imgL);
    for(unsigned int i =0; i <nbLoop;++i)
    {
       std::cout << std::setprecision(2) << std::fixed  << std::setw(5) << std::left << (100.* i) / nbLoop << " %                   \r" << std::flush;

        const std::vector<point> & sequence = fractale.genererSequence(z);
        accumulateur.accumuler(resultat,z,sequence);
        z = fractale.suivant(*sequence.rbegin());
    }
    std::cout << std::endl;
    return resultat;

}

}
#endif
